# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/00_core.ipynb.

# %% auto 0
__all__ = ['logger', 'MINHASH_SEED', 'NON_ALPHA', 'hash_content', 'query_content', 'jaccard_similarity', 'BenchmarkCleaner']

# %% ../nbs/00_core.ipynb 2
import logging
import multiprocessing
import os
import re
import time

import numpy as np

from datasets import Dataset, load_dataset, Features, Sequence, Value
from datasketch import LeanMinHash, MinHash, MinHashLSH
from rich.logging import RichHandler
from tqdm.auto import tqdm

# %% ../nbs/00_core.ipynb 3
multiprocessing.set_start_method("fork", force=True)
logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)
logger.addHandler(RichHandler(rich_tracebacks=True))
logger.propagate = False

# %% ../nbs/00_core.ipynb 5
MINHASH_SEED = 42
NON_ALPHA = re.compile("[^A-Za-z_0-9]")

# %% ../nbs/00_core.ipynb 6
def hash_content(
    idx: int, # index of the document
    content: str, # content of the document
    *,
    num_perm: int # number of permutations
    ): # The MinHash signature and the index of the record.
    """
    Hash the content of a record using MinHash. This function should be
    used with multiprocessing and it scales well with the number of cores.
    """
    m = MinHash(num_perm=num_perm, seed=MINHASH_SEED)
    m.update_batch([token.encode("utf-8") for token in {t for t in NON_ALPHA.split(content) if t}])
    return {"__signature__": m.hashvalues, "__id__": idx}

# %% ../nbs/00_core.ipynb 8
def query_content(
    idx: int, # index of the document
    signature: np.ndarray, # MinHash signature of the document
    *,
    index: MinHashLSH # The MinHashLSH index. It is shared across all processes when using multiprocessing with fork without copy.
    ): # The query result.
    """
    Query the MinHashLSH index for the record. This function can be used with multiprocessing
    as long as the index is shared across processes.
    Parameters.
    """
    return {
        "__neighbors__": [
            str(dup_idx)
            for dup_idx in index.query(
                LeanMinHash(seed=MINHASH_SEED, hashvalues=signature),
            )
        ],
        "__id__": idx,
    }

# %% ../nbs/00_core.ipynb 9
def jaccard_similarity(
    s1: str, # The first string to compare.
    s2: str # The second string to compare.
    ) -> float: # The Jaccard similarity between the two strings.
    """
    Calculate the jaccard similarity between two code snippets.
    """
    tokens1 = set([t for t in NON_ALPHA.split(s1) if t.strip()])
    tokens2 = set([t for t in NON_ALPHA.split(s2) if t.strip()])
    return len(tokens1 & tokens2) / max(1, len(tokens1 | tokens2))

# %% ../nbs/00_core.ipynb 11
class BenchmarkCleaner:
    """
    A class to clean the benchmark dataset.
    """
    def __init__(
        self,
        benchmarks: list, # The list of benchmarks to clean.
        threshold: float = 0.5, # The threshold to use for the MinHashLSH index.
        num_perm: int = 128 # The number of permutations to use for the MinHashLSH index.
        ):
        self.benchmarks = benchmarks
        self.threshold = threshold
        self.num_perm = num_perm
    
    def clean(
        self,
        ds: Dataset, # The dataset to clean.
        column: str, # The column to clean.
    ):
        """
        Clean the dataset. This function does the following:
        1. Hash the content of the provided dataset using MinHash.
        2. Iterate over the benchmark datasets and hash their content.
        3. Query the MinHashLSH index for each record in the provided dataset against the benchmark datasets.
        4. Filter out the records that have a high similarity with the benchmark datasets.
        5. Return the cleaned dataset.
        """
        start_time = time.time()
        DATA_SIZE = len(ds)
        ds = ds.map(
            lambda _, idx: {"__id__": idx},
            with_indices=True,
            num_proc=os.cpu_count(),
            desc="Adding index...",
        )
        hashed_ds = ds.map(
            function=hash_content,
            fn_kwargs={"num_perm": self.num_perm},
            input_columns=["__id__", column],
            remove_columns=[column],
            num_proc=os.cpu_count(),
            desc=f"Fingerprinting...",
        )
        dup_ids = set() # The set of duplicate ids that should be filtered out.
        # Iterate over the benchmark datasets, hash their content and query the MinHashLSH index.
        for bm in self.benchmarks:
            globals()[bm["name"]] = MinHashLSH(
                threshold=self.threshold,
                num_perm=self.num_perm,
            )

            # Load the benchmark dataset and the necessary splits
            benchmark_ds = load_dataset(bm["name"], split="+".join(bm["splits"]))
            # remove unused columns
            columns_to_remove = [c for c in benchmark_ds.column_names if c not in bm["columns"]]
            benchmark_ds = benchmark_ds.remove_columns(columns_to_remove)
            benchmark_ds = benchmark_ds.map(
                    function=lambda x, idx: {
                        **hash_content(
                            idx,
                            " ".join(
                                [x[col] if isinstance(x[col], str) else " ".join(x[col]) for col in bm["columns"]]
                            ),
                            num_perm=self.num_perm,
                        ),
                        "__content__": " ".join(
                            [x[col] if isinstance(x[col], str) else " ".join(x[col]) for col in bm["columns"]]
                        ),
                    },
                    num_proc=4,
                    with_indices=True,
                    desc=f"Fingerprinting...",
                )
            
            # Update the global variable with the MinHashLSH index.
            with globals()[bm["name"]].insertion_session() as session:
                for record in benchmark_ds:
                    session.insert(record["__id__"], LeanMinHash(seed=MINHASH_SEED, hashvalues=record["__signature__"]))

            # remove unused columns
            hashed_ds = hashed_ds.remove_columns([c for c in hashed_ds.column_names if c not in ["__id__", "__signature__"]])
            queried = hashed_ds.map(
                function=lambda x, y: query_content(x, y, index=globals()[bm["name"]]),
                num_proc=os.cpu_count(),
                input_columns=[
                    "__id__",
                    "__signature__",
                ],
                remove_columns=["__signature__"],
                desc="Querying...",
                features=Features(
                    {
                        "__id__": Value("uint64"),
                        "__neighbors__": Sequence(Value("string")),
                    }
                ),
            ).filter(
                lambda x: len(x["__neighbors__"]) > 0,
                num_proc=os.cpu_count(),
                desc=f"Filtering...",
            )

            # Update the set of duplicate ids.
            for record in tqdm(
                queried,
                desc=f"Checking for false positives...",
            ):
                neighbors = set(record["__neighbors__"])
                curr_text = ds[record["__id__"]][column]
                for neighbor in neighbors:
                    reference = benchmark_ds[int(neighbor)]
                    reference_text = reference["__content__"]
                    if jaccard_similarity(curr_text, reference_text) >= self.threshold:
                        break
                else:
                    continue
                dup_ids.add(record["__id__"])

        # Filter out the duplicate ids.
        final_data = ds.filter(
            lambda idx: idx not in dup_ids,
            input_columns=["__id__"],
            num_proc=os.cpu_count(),
            desc="Filtering duplicates...",
        )

        FINAL_DATA_SIZE = len(final_data)
        DUP_SIZE = DATA_SIZE - FINAL_DATA_SIZE

        logger.info(f"{'Data Number':<30}: {DATA_SIZE}")
        logger.info(f"{'Duplicate Number':<30}: {DUP_SIZE}")
        logger.info(f"{'Duplicate Rate':<30}: {DUP_SIZE / DATA_SIZE:.2%}")
        logger.info(f"{'Total Time':<30}: {time.time() - start_time:.2f} seconds")

        return final_data
