"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubOIDCStack = exports.TokenActions = exports.ProviderUrl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const utils_1 = require("../common/utils");
const stack_1 = require("../stack");
var ProviderUrl;
(function (ProviderUrl) {
    ProviderUrl["GITHUB"] = "https://token.actions.githubusercontent.com";
})(ProviderUrl = exports.ProviderUrl || (exports.ProviderUrl = {}));
var TokenActions;
(function (TokenActions) {
    TokenActions[TokenActions["ALL"] = 0] = "ALL";
    TokenActions[TokenActions["ALL_BRANCH"] = 1] = "ALL_BRANCH";
    TokenActions[TokenActions["ALL_TAGS"] = 2] = "ALL_TAGS";
    TokenActions[TokenActions["CUSTOM"] = 3] = "CUSTOM";
})(TokenActions = exports.TokenActions || (exports.TokenActions = {}));
class GithubOIDCStack extends stack_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.githubUser = props.githubUser;
        this.githubRepository = props.githubRepository;
        this.tokenAction = props.tokenAction;
        let token = this.createTokenAction(props.tokenAction, props.githubUser, props.githubRepository, props.tokenActionCustom);
        this.oidcRole = this.createOidcRole(ProviderUrl.GITHUB, token);
        this.cdkBootstrapRole = this.createCdkBootstrapRole();
        this.cdkDeployRoleManagedPolicies = props.cdkDeployRoleManagedPolicies;
        this.cdkDeployRolePolicyStatements = props.cdkDeployRolePolicyStatements;
        this.cdkDeployRole = this.createCdkDeployRole(this.cdkDeployRoleManagedPolicies, this.cdkDeployRolePolicyStatements);
    }
    createTokenAction(tokenAction, githubUser, githubRepository, tokenActionCustom) {
        if (tokenAction === TokenActions.ALL) {
            return `repo:${githubUser}/*`;
        }
        if (tokenAction === TokenActions.ALL_BRANCH) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/heads/*`;
        }
        if (tokenAction === TokenActions.ALL_TAGS) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/tags/*`;
        }
        if (tokenAction === TokenActions.CUSTOM) {
            return `repo:${githubUser}/${githubRepository}:ref:refs/${tokenActionCustom}`;
        }
        throw new Error('tokenAction not found');
    }
    createCdkDeployRole(managed_policies, policy_statements) {
        let basePolicy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        'iam:PassRole',
                        'iam:CreateRole',
                        'iam:DeleteRole',
                        'iam:UpdateRole',
                        'iam:DetachRolePolicy',
                        'iam:AttachRolePolicy',
                        'iam:DeleteRolePolicy',
                        'iam:PutRolePolicy',
                        'ssm:GetParameters',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-deploy-role', {
            roleName: 'cdk-oidc-deploy-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKDeployBasePolicy: basePolicy,
            },
        });
        if (policy_statements) {
            for (let index = 0; index < policy_statements.length; index++) {
                role.addToPolicy(policy_statements[index]);
            }
        }
        if (managed_policies) {
            for (let index = 0; index < managed_policies.length; index++) {
                role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName(managed_policies[index]));
            }
        }
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-deploy-role-output', {
            value: role.roleArn,
            description: 'Role for cdk deploy role arn',
            exportName: 'cdk-oidc-deploy-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createCdkBootstrapRole() {
        let policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
                    actions: [
                        'sts:AssumeRole',
                        'iam:*Role*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:cloudformation:${this.region}:${this.account}:stack/CDKToolkit/*`],
                    actions: [
                        'cloudformation:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: [
                        's3:*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ecr:${this.region}:${this.account}:repository/cdk-*`],
                    actions: [
                        'ecr:SetRepositoryPolicy',
                        'ecr:GetLifecyclePolicy',
                        'ecr:PutImageScanningConfiguration',
                        'ecr:DescribeRepositories',
                        'ecr:CreateRepository',
                        'ecr:DeleteRepository',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
                new iam.PolicyStatement({
                    resources: [`arn:aws:ssm:${this.region}:${this.account}:parameter/cdk-bootstrap/*`],
                    actions: [
                        'ssm:GetParameter*',
                        'ssm:PutParameter*',
                        'ssm:DeleteParameter*',
                    ],
                    effect: iam.Effect.ALLOW,
                }),
            ],
        });
        let role = new iam.Role(this, 'cdk-oidc-bootstrap-role', {
            roleName: 'cdk-oidc-bootstrap-role',
            assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
            inlinePolicies: {
                CDKBootstrapPolicy: policy,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, 'cdk-oidc-bootstrap-role-output', {
            value: role.roleArn,
            description: 'Role for cdk bootstrap role arn',
            exportName: 'cdk-oidc-bootstrap-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
    createOidcRole(providerUrl, token) {
        const oidcProvider = new iam.OpenIdConnectProvider(this, 'OIDCProvider', {
            url: providerUrl,
            clientIds: ['sts.amazonaws.com'],
            thumbprints: ['6938fd4d98bab03faadb97b34396831e3780aea1'],
        });
        let role = new iam.Role(this, 'oidc-role', {
            roleName: 'github-oidc-workflow-role',
            assumedBy: new iam.WebIdentityPrincipal(oidcProvider.openIdConnectProviderArn, {
                StringLike: {
                    'token.actions.githubusercontent.com:sub': token,
                },
            }),
        });
        role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'iam:PassRole',
                'ssm:GetParameter*',
                'cloudformation:*',
                's3:*',
                'ecr:*',
            ],
            effect: iam.Effect.ALLOW,
        }));
        role.addToPolicy(new iam.PolicyStatement({
            resources: [`arn:aws:iam::${this.account}:role/cdk-*`],
            actions: ['sts:AssumeRole'],
            effect: iam.Effect.ALLOW,
        }));
        new aws_cdk_lib_1.CfnOutput(this, 'github-oidc-workflow-role-output', {
            value: role.roleArn,
            description: 'Role for OIDC Github Workflow role arn',
            exportName: 'github-oidc-workflow-role',
        });
        utils_1.addBaseTags(role);
        return role;
    }
}
exports.GithubOIDCStack = GithubOIDCStack;
_a = JSII_RTTI_SYMBOL_1;
GithubOIDCStack[_a] = { fqn: "neulabs-cdk-constructs.oidc.GithubOIDCStack", version: "0.1.15" };
//# sourceMappingURL=data:application/json;base64,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