"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NewRelicStack = exports.EndpointUrlMetrics = exports.EndpointUrlLogs = exports.EndpointType = exports.NEW_RELIC_AWS_ACCOUNT_ID = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const firehose = require("aws-cdk-lib/aws-kinesisfirehose");
const s3 = require("aws-cdk-lib/aws-s3");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const utils_1 = require("../common/utils");
const stack_1 = require("../stack");
exports.NEW_RELIC_AWS_ACCOUNT_ID = '754728514883';
var EndpointType;
(function (EndpointType) {
    EndpointType["METRICS"] = "metrics";
    EndpointType["LOGS"] = "logs";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
var EndpointUrlLogs;
(function (EndpointUrlLogs) {
    EndpointUrlLogs["EU_LOGS"] = "https://aws-api.eu.newrelic.com/firehose/v1";
    EndpointUrlLogs["US_LOGS"] = "https://aws-api.newrelic.com/firehose/v1";
})(EndpointUrlLogs = exports.EndpointUrlLogs || (exports.EndpointUrlLogs = {}));
var EndpointUrlMetrics;
(function (EndpointUrlMetrics) {
    EndpointUrlMetrics["EU_METRICS"] = "https://aws-api.eu01.nr-data.net/cloudwatch-metrics/v1";
    EndpointUrlMetrics["US_METRICS"] = "https://aws-api.newrelic.com/cloudwatch-metrics/v1";
})(EndpointUrlMetrics = exports.EndpointUrlMetrics || (exports.EndpointUrlMetrics = {}));
class NewRelicStack extends stack_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.newRelicIntegrationRole = this.createNewRelicRole(props.newRelicAccountId);
        this.newRelicSecret = this.createSecrets(props.newRelicAccountId, props.newRelicLicenseKey);
        this.newRelicBucket = this.createFirehoseBucket(props.newRelicBucketName);
        this.newRelicFirehoseRole = this.createFirehoseRole(this.newRelicBucket);
        if (props.newRelicApiUrlLogs) {
            this.newRelicFirehoseLogs = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.LOGS, props.newRelicApiUrlLogs, props.newRelicLicenseKey);
            this.newRelicCloudwatchLogsStreamRole = this.createCloudwatchLogsStreamRole();
        }
        if (props.newRelicApiUrlMetrics) {
            this.newRelicFirehoseMetrics = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.METRICS, props.newRelicApiUrlMetrics, props.newRelicLicenseKey);
            this.createCloudwatchMetricStream(this.newRelicFirehoseRole.roleArn, this.newRelicFirehoseMetrics.attrArn);
        }
    }
    createCloudwatchMetricStream(roleArn, firehoseArn) {
        return new cloudwatch.CfnMetricStream(this, 'newrelic-cloudwatch-stream-metrics', {
            firehoseArn: firehoseArn,
            outputFormat: 'opentelemetry0.7',
            roleArn: roleArn,
            name: 'newelic-stream-metrics',
        });
    }
    createCloudwatchLogsStreamRole() {
        let role = new iam.Role(this, 'newrelic-logstream-role', {
            roleName: 'NewRelicInfrastructure-CloudwatchStream',
            assumedBy: new iam.ServicePrincipal('logs.amazonaws.com'),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecord',
                'firehose:PutRecordBatch',
                'kinesis:PutRecords',
                'kinesis:PutRecord',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        return role;
    }
    createNewRelicRole(newRelicAccountId) {
        let role = new iam.Role(this, 'newrelic-role', {
            roleName: 'NewRelicInfrastructure-Integrations',
            assumedBy: new iam.AccountPrincipal(exports.NEW_RELIC_AWS_ACCOUNT_ID),
            externalIds: [
                newRelicAccountId,
            ],
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'budgets:ViewBudget',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        new aws_cdk_lib_1.CfnOutput(this, 'newrelic-role-output', {
            value: role.roleArn,
            description: 'New Relic role arn',
            exportName: 'newRelicRole',
        });
        return role;
    }
    createFirehoseStream(newRelicBucket, role, endpointType, endpointUrl, newRelicLicenseLey) {
        if (this.stage == 'production') {
            // Minute in one day: 1440
            // Interval: 5min
            // Sends per day: 1440/5 = 288
            // Usage per day: 288*5mb = 1.5gb
            var bufferingHints = {
                intervalInSeconds: 300,
                sizeInMBs: 5,
            };
        }
        else {
            // Usage per day: 144*3mb = 0.432gb
            var bufferingHints = {
                intervalInSeconds: 600,
                sizeInMBs: 3,
            };
        }
        let httpEndpointMetrics = {
            bufferingHints: bufferingHints,
            endpointConfiguration: {
                url: endpointUrl,
                accessKey: newRelicLicenseLey,
                name: endpointType,
            },
            s3Configuration: {
                bucketArn: newRelicBucket.bucketArn,
                roleArn: role.roleArn,
            },
            requestConfiguration: {
                contentEncoding: 'GZIP',
            },
            roleArn: role.roleArn,
        };
        let firehoseStream = new firehose.CfnDeliveryStream(this, `newrelic-firehose-${endpointType}`, {
            deliveryStreamName: `NewRelic-stream-${endpointType}`,
            deliveryStreamType: 'DirectPut',
            httpEndpointDestinationConfiguration: httpEndpointMetrics,
        });
        utils_1.addBaseTags(firehoseStream);
        return firehoseStream;
    }
    createSecrets(newRelicAccountId, newRelicLicenseLey) {
        let secret = new secretsmanager.Secret(this, 'newrelic-secret', {
            secretName: 'NEW_RELIC_LICENSE_KEY',
            secretObjectValue: {
                NrAccountId: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicAccountId),
                LicenseKey: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicLicenseLey),
            },
        });
        utils_1.addBaseTags(secret);
        return secret;
    }
    createFirehoseBucket(newRelicBucketName) {
        let bucket = new s3.Bucket(this, 'newrelic-bucket', {
            bucketName: newRelicBucketName,
            versioned: true,
            lifecycleRules: [
                {
                    id: 'raw-bucket-lifecycle-rule',
                    enabled: true,
                    noncurrentVersionExpiration: aws_cdk_lib_1.Duration.days(30),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                    ],
                },
            ],
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        utils_1.addBaseTags(bucket);
        return bucket;
    }
    createFirehoseRole(newRelicFirehoseBucket) {
        let role = new iam.Role(this, 'newrelic-firehose-role', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        utils_1.addBaseTags(role);
        // TODO: create more restrictive policy
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'glue:GetTable',
                'glue:GetTableVersion',
                'glue:GetTableVersions',
                'glue:GetSchemaByDefinition',
                'glue:GetSchemaVersion',
                'logs:PutLogEvents',
                'kinesis:DescribeStream',
                'kinesis:GetShardIterator',
                'kinesis:GetRecords',
                'kinesis:ListShards',
                'kms:Decrypt',
                'kms:GenerateDataKey',
                'kms:Decrypt',
                'kafka:GetBootstrapBrokers',
                'kafka:DescribeCluster',
                'kafka:DescribeClusterV2',
                'kafka-cluster:Connect',
                'kafka-cluster:DescribeTopic',
                'kafka-cluster:DescribeTopicDynamicConfiguration',
                'kafka-cluster:ReadData',
                'kafka-cluster:DescribeGroup',
                'lambda:InvokeFunction',
                'lambda:GetFunctionConfiguration',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
                'S3:PutObjectAcl',
            ],
            resources: [
                `${newRelicFirehoseBucket.bucketArn}`,
                `${newRelicFirehoseBucket.bucketArn}/*`,
            ],
        }));
        return role;
    }
}
exports.NewRelicStack = NewRelicStack;
_a = JSII_RTTI_SYMBOL_1;
NewRelicStack[_a] = { fqn: "neulabs-cdk-constructs.newrelic.NewRelicStack", version: "0.1.15" };
//# sourceMappingURL=data:application/json;base64,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