import importlib.util
import os
import traceback
from itertools import chain
from types import ModuleType
from typing import Any, List, Union

from ..utils.constants import Backends
from ..utils.lib_utils.integeration_utils import is_backend_available
from ..utils.lib_utils.main_utils import import_module
from ..utils.py_utils.py_utils import PyUtils


def requires_backends(obj, backends, module_name: str = None, cls_name: str = None):
    if not isinstance(backends, (list, tuple)):
        backends = [backends]
    name = obj.__name__ if hasattr(obj, "__name__") else obj.__class__.__name__
    failed = []
    for backend in backends:
        if not is_backend_available(backend):
            failed.append(backend)

    if failed:
        raise ModuleNotFoundError(
            f"`{name}` requires "
            f"{f'`{failed[0]}`' if len(failed) == 1 else failed} "
            f"which {'is' if len(failed) == 1 else 'are'} not installed!"
        )
    else:
        PyUtils.print("A library is missing which is not listed in backends of dummy_object",
                      color="red", mode=["bold", "underline"])
        if module_name is None:
            PyUtils.print(
                "The module_name is not defined to import the module and see the errors and missing libraries!",
                color="red", mode=["bold", "underline"])
        else:
            error = import_module(module_name, cls_name)
            raise error


class DummyObject(type):
    """
    Metaclass for the dummy objects. Any class inheriting from it will return the ImportError generated by
    `requires_backend` each time a user tries to access any method of that class.
    """
    _backend: List[Union[Backends, str]]
    _module: str

    def __call__(cls, *args, **kwargs):
        self = super().__call__()
        requires_backends(self, self._backend, module_name=self._module, cls_name=self.__class__.__name__)
        return self


class LazyModule(ModuleType):
    """
    Module class that surfaces all objects but only performs associated imports when the objects are requested.
    """

    # Very heavily inspired by huggingface/transformers
    # https://github.com/huggingface/transformers/blob/main/src/transformers/utils/import_utils.py
    def __init__(self, name, module_file, import_structure, module_spec=None, extra_objects=None):
        super().__init__(name)
        self._modules = set(import_structure.keys())
        self._class_to_module = {}
        self._dummy_class_to_class = {}
        for key, values in import_structure.items():
            for value in values:
                if isinstance(value, str):
                    self._class_to_module[value] = key
                elif isinstance(value, type):
                    self._dummy_class_to_class[value.__name__] = value
                else:
                    raise ValueError(f"Dummy object: {key}: {value} has not the correct format")
        # Needed for autocompletion in an IDE
        self.__all__ = list(import_structure.keys()) + list(chain(*import_structure.values()))
        self.__file__ = module_file
        self.__spec__ = module_spec
        self.__path__ = [os.path.dirname(module_file)]
        self._objects = {} if extra_objects is None else extra_objects
        self._name = name
        self._import_structure = import_structure

    # Needed for autocompletion in an IDE
    def __dir__(self):
        result = super().__dir__()
        # The elements of self.__all__ that are submodules may or may not be in the dir already, depending on whether
        # they have been accessed or not. So we only add the elements of self.__all__ that are not already in the dir.
        for attr in self.__all__:
            if attr not in result:
                result.append(attr)
        return result

    def __getattr__(self, name: str) -> Any:
        if name in self._objects:
            return self._objects[name]
        if name in self._dummy_class_to_class:
            return self._dummy_class_to_class[name]
        if name in self._modules:
            value = self._get_module(name)
        elif name in self._class_to_module.keys():
            module = self._get_module(self._class_to_module[name])
            value = getattr(module, name)
        else:
            raise AttributeError(f"module {self.__name__} has no attribute {name}")

        setattr(self, name, value)
        return value

    def _get_module(self, module_name: str):
        try:
            return importlib.import_module("." + module_name, self.__name__)
        except Exception as e:
            raise RuntimeError(
                f"Failed to import {self.__name__}.{module_name} because of the following error (look up to see its"
                f" traceback):\n{e}\n{traceback.format_exc()}"
            ) from e

    def __reduce__(self):
        return self.__class__, (self._name, self.__file__, self._import_structure)
