"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// tslint:disable:object-literal-key-quotes
// CDK parameters
const CONSTRUCT_NAME = 'MyTable';
// DynamoDB table parameters
const TABLE_NAME = 'MyTable';
const TABLE_PARTITION_KEY = { name: 'hashKey', type: lib_1.AttributeType.STRING };
const TABLE_SORT_KEY = { name: 'sortKey', type: lib_1.AttributeType.NUMBER };
// DynamoDB global secondary index parameters
const GSI_NAME = 'MyGSI';
const GSI_PARTITION_KEY = { name: 'gsiHashKey', type: lib_1.AttributeType.STRING };
const GSI_SORT_KEY = { name: 'gsiSortKey', type: lib_1.AttributeType.BINARY };
const GSI_NON_KEY = 'gsiNonKey';
function* GSI_GENERATOR() {
    let n = 0;
    while (true) {
        const globalSecondaryIndexProps = {
            indexName: `${GSI_NAME}${n}`,
            partitionKey: { name: `${GSI_PARTITION_KEY.name}${n}`, type: GSI_PARTITION_KEY.type },
        };
        yield globalSecondaryIndexProps;
        n++;
    }
}
function* NON_KEY_ATTRIBUTE_GENERATOR(nonKeyPrefix) {
    let n = 0;
    while (true) {
        yield `${nonKeyPrefix}${n}`;
        n++;
    }
}
// DynamoDB local secondary index parameters
const LSI_NAME = 'MyLSI';
const LSI_SORT_KEY = { name: 'lsiSortKey', type: lib_1.AttributeType.NUMBER };
const LSI_NON_KEY = 'lsiNonKey';
function* LSI_GENERATOR() {
    let n = 0;
    while (true) {
        const localSecondaryIndexProps = {
            indexName: `${LSI_NAME}${n}`,
            sortKey: { name: `${LSI_SORT_KEY.name}${n}`, type: LSI_SORT_KEY.type },
        };
        yield localSecondaryIndexProps;
        n++;
    }
}
describe('default properties', () => {
    test('hash key only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [{ AttributeName: 'hashKey', AttributeType: 'S' }],
            KeySchema: [{ AttributeName: 'hashKey', KeyType: 'HASH' }],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.RETAIN }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('removalPolicy is DESTROY', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, removalPolicy: core_1.RemovalPolicy.DESTROY });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', { DeletionPolicy: core_1.CfnDeletionPolicy.DELETE }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('hash + range key', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('hash + range key can also be specified in props', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('point-in-time recovery is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('server-side encryption is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('stream is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('ttl is not enabled', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        });
    });
    test('can specify new and old images', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_AND_OLD_IMAGES,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_AND_OLD_IMAGES' },
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            TableName: 'MyTable',
        });
    });
    test('can specify new images only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.NEW_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'NEW_IMAGE' },
            TableName: 'MyTable',
        });
    });
    test('can specify old images only', () => {
        const stack = new core_1.Stack();
        new lib_1.Table(stack, CONSTRUCT_NAME, {
            tableName: TABLE_NAME,
            readCapacity: 42,
            writeCapacity: 1337,
            stream: lib_1.StreamViewType.OLD_IMAGE,
            partitionKey: TABLE_PARTITION_KEY,
            sortKey: TABLE_SORT_KEY,
        });
        expect(stack).toHaveResource('AWS::DynamoDB::Table', {
            KeySchema: [
                { AttributeName: 'hashKey', KeyType: 'HASH' },
                { AttributeName: 'sortKey', KeyType: 'RANGE' },
            ],
            ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            AttributeDefinitions: [
                { AttributeName: 'hashKey', AttributeType: 'S' },
                { AttributeName: 'sortKey', AttributeType: 'N' },
            ],
            StreamSpecification: { StreamViewType: 'OLD_IMAGE' },
            TableName: 'MyTable',
        });
    });
});
test('when specifying every property', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        readCapacity: 42,
        writeCapacity: 1337,
        pointInTimeRecovery: true,
        serverSideEncryption: true,
        billingMode: lib_1.BillingMode.PROVISIONED,
        stream: lib_1.StreamViewType.KEYS_ONLY,
        timeToLiveAttribute: 'timeToLive',
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 42,
            WriteCapacityUnits: 1337,
        },
        PointInTimeRecoverySpecification: { PointInTimeRecoveryEnabled: true },
        SSESpecification: { SSEEnabled: true },
        StreamSpecification: { StreamViewType: 'KEYS_ONLY' },
        TableName: 'MyTable',
        Tags: [{ Key: 'Environment', Value: 'Production' }],
        TimeToLiveSpecification: { AttributeName: 'timeToLive', Enabled: true },
    });
});
test('when specifying sse with customer managed CMK', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'MyTableKey8597C7A6',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('when specifying only encryptionKey', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryptionKey,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('when specifying sse with customer managed CMK with encryptionKey provided by user', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
        encryptionKey,
        partitionKey: TABLE_PARTITION_KEY,
    });
    table.node.applyAspect(new core_1.Tag('Environment', 'Production'));
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        'SSESpecification': {
            'KMSMasterKeyId': {
                'Fn::GetAtt': [
                    'Key961B73FD',
                    'Arn',
                ],
            },
            'SSEEnabled': true,
            'SSEType': 'KMS',
        },
    });
});
test('fails if encryption key is used with AWS managed CMK', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.AWS_MANAGED,
        encryptionKey,
    })).toThrow('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
});
test('fails if encryption key is used with default encryption', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.DEFAULT,
        encryptionKey,
    })).toThrow('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
});
test('fails if encryption key is used with serverSideEncryption', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        serverSideEncryption: true,
        encryptionKey,
    })).toThrow(/encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead/);
});
test('fails if both encryption and serverSideEncryption is specified', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.DEFAULT,
        serverSideEncryption: true,
    })).toThrow(/Only one of encryption and serverSideEncryption can be specified, but both were provided/);
});
test('fails if both replication regions used with customer managed CMK', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        replicationRegions: ['us-east-1', 'us-east-2', 'us-west-2'],
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
    })).toThrow('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
});
test('if an encryption key is included, decrypt permissions are also added for grantStream', () => {
    const stack = new core_1.Stack();
    const encryptionKey = new kms.Key(stack, 'Key', {
        enableKeyRotation: true,
    });
    const table = new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryptionKey,
        stream: lib_1.StreamViewType.NEW_IMAGE,
    });
    const user = new iam.User(stack, 'MyUser');
    table.grantStreamRead(user);
    expect(stack).toMatchTemplate({
        'Resources': {
            'Key961B73FD': {
                'Type': 'AWS::KMS::Key',
                'Properties': {
                    'KeyPolicy': {
                        'Statement': [
                            {
                                'Action': [
                                    'kms:Create*',
                                    'kms:Describe*',
                                    'kms:Enable*',
                                    'kms:List*',
                                    'kms:Put*',
                                    'kms:Update*',
                                    'kms:Revoke*',
                                    'kms:Disable*',
                                    'kms:Get*',
                                    'kms:Delete*',
                                    'kms:ScheduleKeyDeletion',
                                    'kms:CancelKeyDeletion',
                                    'kms:GenerateDataKey',
                                    'kms:TagResource',
                                    'kms:UntagResource',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':iam::',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':root',
                                            ],
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'EnableKeyRotation': true,
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'TableA3D7B5AFA': {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'ProvisionedThroughput': {
                        'ReadCapacityUnits': 5,
                        'WriteCapacityUnits': 5,
                    },
                    'SSESpecification': {
                        'KMSMasterKeyId': {
                            'Fn::GetAtt': [
                                'Key961B73FD',
                                'Arn',
                            ],
                        },
                        'SSEEnabled': true,
                        'SSEType': 'KMS',
                    },
                    'StreamSpecification': {
                        'StreamViewType': 'NEW_IMAGE',
                    },
                    'TableName': 'MyTable',
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'MyUserDC45028B': {
                'Type': 'AWS::IAM::User',
            },
            'MyUserDefaultPolicy7B897426': {
                'Type': 'AWS::IAM::Policy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': 'dynamodb:ListStreams',
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::Join': [
                                        '',
                                        [
                                            {
                                                'Fn::GetAtt': [
                                                    'TableA3D7B5AFA',
                                                    'Arn',
                                                ],
                                            },
                                            '/stream/*',
                                        ],
                                    ],
                                },
                            },
                            {
                                'Action': [
                                    'dynamodb:DescribeStream',
                                    'dynamodb:GetRecords',
                                    'dynamodb:GetShardIterator',
                                ],
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'TableA3D7B5AFA',
                                        'StreamArn',
                                    ],
                                },
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'PolicyName': 'MyUserDefaultPolicy7B897426',
                    'Users': [
                        {
                            'Ref': 'MyUserDC45028B',
                        },
                    ],
                },
            },
        },
    });
});
test('if an encryption key is included, encrypt/decrypt permissions are also added both ways', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        partitionKey: TABLE_PARTITION_KEY,
        encryption: lib_1.TableEncryption.CUSTOMER_MANAGED,
    });
    const user = new iam.User(stack, 'MyUser');
    table.grantReadWriteData(user);
    expect(stack).toMatchTemplate({
        'Resources': {
            'TableAKey07CC09EC': {
                'Type': 'AWS::KMS::Key',
                'Properties': {
                    'KeyPolicy': {
                        'Statement': [
                            {
                                'Action': [
                                    'kms:Create*',
                                    'kms:Describe*',
                                    'kms:Enable*',
                                    'kms:List*',
                                    'kms:Put*',
                                    'kms:Update*',
                                    'kms:Revoke*',
                                    'kms:Disable*',
                                    'kms:Get*',
                                    'kms:Delete*',
                                    'kms:ScheduleKeyDeletion',
                                    'kms:CancelKeyDeletion',
                                    'kms:GenerateDataKey',
                                    'kms:TagResource',
                                    'kms:UntagResource',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':iam::',
                                                {
                                                    'Ref': 'AWS::AccountId',
                                                },
                                                ':root',
                                            ],
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                            {
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::GetAtt': [
                                            'MyUserDC45028B',
                                            'Arn',
                                        ],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'Description': 'Customer-managed key auto-created for encrypting DynamoDB table at Table A',
                    'EnableKeyRotation': true,
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'TableA3D7B5AFA': {
                'Type': 'AWS::DynamoDB::Table',
                'Properties': {
                    'KeySchema': [
                        {
                            'AttributeName': 'hashKey',
                            'KeyType': 'HASH',
                        },
                    ],
                    'AttributeDefinitions': [
                        {
                            'AttributeName': 'hashKey',
                            'AttributeType': 'S',
                        },
                    ],
                    'ProvisionedThroughput': {
                        'ReadCapacityUnits': 5,
                        'WriteCapacityUnits': 5,
                    },
                    'SSESpecification': {
                        'KMSMasterKeyId': {
                            'Fn::GetAtt': [
                                'TableAKey07CC09EC',
                                'Arn',
                            ],
                        },
                        'SSEEnabled': true,
                        'SSEType': 'KMS',
                    },
                    'TableName': 'MyTable',
                },
                'UpdateReplacePolicy': 'Retain',
                'DeletionPolicy': 'Retain',
            },
            'MyUserDC45028B': {
                'Type': 'AWS::IAM::User',
            },
            'MyUserDefaultPolicy7B897426': {
                'Type': 'AWS::IAM::Policy',
                'Properties': {
                    'PolicyDocument': {
                        'Statement': [
                            {
                                'Action': [
                                    'dynamodb:BatchGetItem',
                                    'dynamodb:GetRecords',
                                    'dynamodb:GetShardIterator',
                                    'dynamodb:Query',
                                    'dynamodb:GetItem',
                                    'dynamodb:Scan',
                                    'dynamodb:BatchWriteItem',
                                    'dynamodb:PutItem',
                                    'dynamodb:UpdateItem',
                                    'dynamodb:DeleteItem',
                                ],
                                'Effect': 'Allow',
                                'Resource': [
                                    {
                                        'Fn::GetAtt': [
                                            'TableA3D7B5AFA',
                                            'Arn',
                                        ],
                                    },
                                    {
                                        'Ref': 'AWS::NoValue',
                                    },
                                ],
                            },
                            {
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Resource': {
                                    'Fn::GetAtt': [
                                        'TableAKey07CC09EC',
                                        'Arn',
                                    ],
                                },
                            },
                        ],
                        'Version': '2012-10-17',
                    },
                    'PolicyName': 'MyUserDefaultPolicy7B897426',
                    'Users': [
                        {
                            'Ref': 'MyUserDC45028B',
                        },
                    ],
                },
            },
        },
    });
});
test('when specifying PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
        ],
        TableName: 'MyTable',
    });
});
test('error when specifying read or write capacity with a PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    expect(() => new lib_1.Table(stack, 'Table A', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table B', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => new lib_1.Table(stack, 'Table C', {
        tableName: TABLE_NAME,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding a global secondary index with hash key only', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.ALL,
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value, gsiNonKeyAttributeGenerator.next().value],
        readCapacity: 42,
        writeCapacity: 1337,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
            { AttributeName: 'gsiSortKey', AttributeType: 'B' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                    { AttributeName: 'gsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['gsiNonKey0', 'gsiNonKey1'], ProjectionType: 'INCLUDE' },
                ProvisionedThroughput: { ReadCapacityUnits: 42, WriteCapacityUnits: 1337 },
            },
        ],
    });
});
test('when adding a global secondary index on a table with PAY_PER_REQUEST billing mode', () => {
    const stack = new core_1.Stack();
    new lib_1.Table(stack, CONSTRUCT_NAME, {
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
        partitionKey: TABLE_PARTITION_KEY,
        sortKey: TABLE_SORT_KEY,
    }).addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        BillingMode: 'PAY_PER_REQUEST',
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('error when adding a global secondary index with projection type INCLUDE, but without specifying non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
    })).toThrow(/non-key attributes should be specified when using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type ALL, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type KEYS_ONLY, but with non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
        nonKeyAttributes: [gsiNonKeyAttributeGenerator.next().value],
    })).toThrow(/non-key attributes should not be specified when not using INCLUDE projection type/);
});
test('error when adding a global secondary index with projection type INCLUDE, but with more than 20 non-key attributes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(GSI_NON_KEY);
    const gsiNonKeyAttributes = [];
    for (let i = 0; i < 101; i++) {
        gsiNonKeyAttributes.push(gsiNonKeyAttributeGenerator.next().value);
    }
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: gsiNonKeyAttributes,
    })).toThrow(/a maximum number of nonKeyAttributes across all of secondary indexes is 100/);
});
test('error when adding a global secondary index with read or write capacity on a PAY_PER_REQUEST table', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        partitionKey: TABLE_PARTITION_KEY,
        billingMode: lib_1.BillingMode.PAY_PER_REQUEST,
    });
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
        sortKey: GSI_SORT_KEY,
        readCapacity: 1,
        writeCapacity: 1,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('when adding multiple global secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const gsiGenerator = GSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addGlobalSecondaryIndex(gsiGenerator.next().value);
    }
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey0', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey1', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey2', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey3', AttributeType: 'S' },
            { AttributeName: 'gsiHashKey4', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI0',
                KeySchema: [
                    { AttributeName: 'gsiHashKey0', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI1',
                KeySchema: [
                    { AttributeName: 'gsiHashKey1', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI2',
                KeySchema: [
                    { AttributeName: 'gsiHashKey2', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI3',
                KeySchema: [
                    { AttributeName: 'gsiHashKey3', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
            {
                IndexName: 'MyGSI4',
                KeySchema: [
                    { AttributeName: 'gsiHashKey4', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a global secondary index without specifying read and write capacity', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'gsiHashKey', AttributeType: 'S' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        GlobalSecondaryIndexes: [
            {
                IndexName: 'MyGSI',
                KeySchema: [
                    { AttributeName: 'gsiHashKey', KeyType: 'HASH' },
                ],
                Projection: { ProjectionType: 'ALL' },
                ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
            },
        ],
    });
});
test('when adding a local secondary index with hash + range key', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'ALL' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type KEYS_ONLY', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.KEYS_ONLY,
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { ProjectionType: 'KEYS_ONLY' },
            },
        ],
    });
});
test('when adding a local secondary index with projection type INCLUDE', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiNonKeyAttributeGenerator = NON_KEY_ATTRIBUTE_GENERATOR(LSI_NON_KEY);
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
        projectionType: lib_1.ProjectionType.INCLUDE,
        nonKeyAttributes: [lsiNonKeyAttributeGenerator.next().value, lsiNonKeyAttributeGenerator.next().value],
    });
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        AttributeDefinitions: [
            { AttributeName: 'hashKey', AttributeType: 'S' },
            { AttributeName: 'sortKey', AttributeType: 'N' },
            { AttributeName: 'lsiSortKey', AttributeType: 'N' },
        ],
        KeySchema: [
            { AttributeName: 'hashKey', KeyType: 'HASH' },
            { AttributeName: 'sortKey', KeyType: 'RANGE' },
        ],
        ProvisionedThroughput: { ReadCapacityUnits: 5, WriteCapacityUnits: 5 },
        LocalSecondaryIndexes: [
            {
                IndexName: 'MyLSI',
                KeySchema: [
                    { AttributeName: 'hashKey', KeyType: 'HASH' },
                    { AttributeName: 'lsiSortKey', KeyType: 'RANGE' },
                ],
                Projection: { NonKeyAttributes: ['lsiNonKey0', 'lsiNonKey1'], ProjectionType: 'INCLUDE' },
            },
        ],
    });
});
test('error when adding more than 5 local secondary indexes', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    const lsiGenerator = LSI_GENERATOR();
    for (let i = 0; i < 5; i++) {
        table.addLocalSecondaryIndex(lsiGenerator.next().value);
    }
    expect(() => table.addLocalSecondaryIndex(lsiGenerator.next().value))
        .toThrow(/a maximum number of local secondary index per table is 5/);
});
test('error when adding a local secondary index with the name of a global secondary index', () => {
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY, sortKey: TABLE_SORT_KEY });
    table.addGlobalSecondaryIndex({
        indexName: 'SecondaryIndex',
        partitionKey: GSI_PARTITION_KEY,
    });
    expect(() => table.addLocalSecondaryIndex({
        indexName: 'SecondaryIndex',
        sortKey: LSI_SORT_KEY,
    })).toThrow(/a duplicate index name, SecondaryIndex, is not allowed/);
});
test('error when validating construct if a local secondary index exists without a sort key of the table', () => {
    var _a;
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { partitionKey: TABLE_PARTITION_KEY });
    table.addLocalSecondaryIndex({
        indexName: LSI_NAME,
        sortKey: LSI_SORT_KEY,
    });
    const errors = core_1.ConstructNode.validate(table.node);
    expect(errors.length).toBe(1);
    expect((_a = errors[0]) === null || _a === void 0 ? void 0 : _a.message).toBe('a sort key of the table must be specified to add local secondary indexes');
});
test('can enable Read AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:ReadCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBReadCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('can enable Write AutoScaling', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // WHEN
    table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        MaxCapacity: 500,
        MinCapacity: 50,
        ScalableDimension: 'dynamodb:table:WriteCapacityUnits',
        ServiceNamespace: 'dynamodb',
    });
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
        PolicyType: 'TargetTrackingScaling',
        TargetTrackingScalingPolicyConfiguration: {
            PredefinedMetricSpecification: { PredefinedMetricType: 'DynamoDBWriteCapacityUtilization' },
            TargetValue: 75,
        },
    });
});
test('cannot enable AutoScaling twice on the same property', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 75 });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/Read AutoScaling already enabled for this table/);
});
test('error when enabling AutoScaling on the PAY_PER_REQUEST table', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { billingMode: lib_1.BillingMode.PAY_PER_REQUEST, partitionKey: TABLE_PARTITION_KEY });
    table.addGlobalSecondaryIndex({
        indexName: GSI_NAME,
        partitionKey: GSI_PARTITION_KEY,
    });
    // WHEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => {
        table.autoScaleWriteCapacity({ minCapacity: 50, maxCapacity: 500 });
    }).toThrow(/PAY_PER_REQUEST/);
    expect(() => table.autoScaleGlobalSecondaryIndexReadCapacity(GSI_NAME, {
        minCapacity: 1,
        maxCapacity: 5,
    })).toThrow(/PAY_PER_REQUEST/);
});
test('error when specifying Read Auto Scaling with invalid scalingTargetValue < 10', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => {
        table.autoScaleReadCapacity({ minCapacity: 50, maxCapacity: 500 }).scaleOnUtilization({ targetUtilizationPercent: 5 });
    }).toThrow(/targetUtilizationPercent for DynamoDB scaling must be between 10 and 90 percent, got: 5/);
});
test('error when specifying Read Auto Scaling with invalid minimumCapacity', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, { readCapacity: 42, writeCapacity: 1337, partitionKey: TABLE_PARTITION_KEY });
    // THEN
    expect(() => table.autoScaleReadCapacity({ minCapacity: 10, maxCapacity: 5 }))
        .toThrow(/minCapacity \(10\) should be lower than maxCapacity \(5\)/);
});
test('can autoscale on a schedule', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, CONSTRUCT_NAME, {
        readCapacity: 42,
        writeCapacity: 1337,
        partitionKey: { name: 'Hash', type: lib_1.AttributeType.STRING },
    });
    // WHEN
    const scaling = table.autoScaleReadCapacity({ minCapacity: 1, maxCapacity: 100 });
    scaling.scaleOnSchedule('SaveMoneyByNotScalingUp', {
        schedule: appscaling.Schedule.cron({}),
        maxCapacity: 10,
    });
    // THEN
    expect(stack).toHaveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
        ScheduledActions: [
            {
                ScalableTargetAction: { 'MaxCapacity': 10 },
                Schedule: 'cron(* * * * ? *)',
                ScheduledActionName: 'SaveMoneyByNotScalingUp',
            },
        ],
    });
});
describe('metrics', () => {
    test('Can use metricConsumedReadCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedReadCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedReadCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricConsumedWriteCapacityUnits on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConsumedWriteCapacityUnits())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConsumedWriteCapacityUnits',
            statistic: 'Sum',
        });
    });
    test('Can use metricSystemErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSystemErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SystemErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricUserErrors on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricUserErrors())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'UserErrors',
            statistic: 'Sum',
        });
    });
    test('Can use metricConditionalCheckFailedRequests on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricConditionalCheckFailedRequests())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'ConditionalCheckFailedRequests',
            statistic: 'Sum',
        });
    });
    test('Can use metricSuccessfulRequestLatency on a Dynamodb Table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'id', type: lib_1.AttributeType.STRING },
        });
        // THEN
        expect(stack.resolve(table.metricSuccessfulRequestLatency())).toEqual({
            period: core_1.Duration.minutes(5),
            dimensions: { TableName: { Ref: 'TableCD117FA1' } },
            namespace: 'AWS/DynamoDB',
            metricName: 'SuccessfulRequestLatency',
            statistic: 'Average',
        });
    });
});
describe('grants', () => {
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grant(user, 'dynamodb:action1', 'dynamodb:action2');
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:action1',
                            'dynamodb:action2',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'mytable0324D45C',
                                    'Arn',
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'userDefaultPolicy083DF682',
            'Users': [
                {
                    'Ref': 'user2C2B57AE',
                },
            ],
        });
    });
    test('"grant" allows adding arbitrary actions associated with this table resource', () => {
        testGrant(['action1', 'action2'], (p, t) => t.grant(p, 'dynamodb:action1', 'dynamodb:action2'));
    });
    test('"grantReadData" allows the principal to read data from the table', () => {
        testGrant(['BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan'], (p, t) => t.grantReadData(p));
    });
    test('"grantWriteData" allows the principal to write data to the table', () => {
        testGrant(['BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'], (p, t) => t.grantWriteData(p));
    });
    test('"grantReadWriteData" allows the principal to read/write data', () => {
        testGrant([
            'BatchGetItem', 'GetRecords', 'GetShardIterator', 'Query', 'GetItem', 'Scan',
            'BatchWriteItem', 'PutItem', 'UpdateItem', 'DeleteItem'
        ], (p, t) => t.grantReadWriteData(p));
    });
    test('"grantFullAccess" allows the principal to perform any action on the table ("*")', () => {
        testGrant(['*'], (p, t) => t.grantFullAccess(p));
    });
    test('"Table.grantListStreams" allows principal to list all streams', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const user = new iam.User(stack, 'user');
        // WHEN
        lib_1.Table.grantListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': '*',
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantTableListStreams" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantTableListStreams(user)).toThrow(/DynamoDB Streams must be enabled on the table my-table/);
    });
    test('"grantTableListStreams" allows principal to list all streams for this table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('"grantStreamRead" should fail if streaming is not enabled on table"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        expect(() => table.grantStreamRead(user)).toThrow(/DynamoDB Streams must be enabled on the table my-table/);
    });
    test('"grantStreamRead" allows principal to read and describe the table stream"', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            stream: lib_1.StreamViewType.NEW_IMAGE,
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantStreamRead(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': 'dynamodb:ListStreams',
                        'Effect': 'Allow',
                        'Resource': { 'Fn::Join': ['', [{ 'Fn::GetAtt': ['mytable0324D45C', 'Arn'] }, '/stream/*']] },
                    },
                    {
                        'Action': [
                            'dynamodb:DescribeStream',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'StreamArn',
                            ],
                        },
                    },
                ],
                'Version': '2012-10-17',
            },
            'Users': [{ 'Ref': 'user2C2B57AE' }],
        });
    });
    test('if table has an index grant gives access to the index', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
        table.addGlobalSecondaryIndex({ indexName: 'MyIndex', partitionKey: { name: 'Age', type: lib_1.AttributeType.NUMBER } });
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'mytable0324D45C',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'mytable0324D45C',
                                                'Arn',
                                            ],
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'userDefaultPolicy083DF682',
            'Users': [
                {
                    'Ref': 'user2C2B57AE',
                },
            ],
        });
    });
    test('grant for an imported table', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = lib_1.Table.fromTableName(stack, 'MyTable', 'my-table');
        const user = new iam.User(stack, 'user');
        // WHEN
        table.grant(user, 'dynamodb:*');
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'dynamodb:*',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                Ref: 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            Users: [
                {
                    Ref: 'user2C2B57AE',
                },
            ],
        });
    });
});
describe('secondary indexes', () => {
    // See https://github.com/aws/aws-cdk/issues/4398
    test('attribute can be used as key attribute in one index, and non-key in another', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: { name: 'pkey', type: lib_1.AttributeType.NUMBER },
        });
        // WHEN
        table.addGlobalSecondaryIndex({
            indexName: 'IndexA',
            partitionKey: { name: 'foo', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['bar'],
        });
        // THEN
        expect(() => table.addGlobalSecondaryIndex({
            indexName: 'IndexB',
            partitionKey: { name: 'baz', type: lib_1.AttributeType.STRING },
            sortKey: { name: 'bar', type: lib_1.AttributeType.STRING },
            projectionType: lib_1.ProjectionType.INCLUDE,
            nonKeyAttributes: ['blah'],
        })).not.toThrow();
    });
});
describe('import', () => {
    test('report error when importing an external/existing table from invalid arn missing resource name', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1::table/';
        // WHEN
        expect(() => lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn)).toThrow(/ARN for DynamoDB table must be in the form: .../);
    });
    test('static fromTableArn(arn) allows importing an external/existing table from arn', () => {
        const stack = new core_1.Stack();
        const tableArn = 'arn:aws:dynamodb:us-east-1:11111111:table/MyTable';
        const table = lib_1.Table.fromTableArn(stack, 'ImportedTable', tableArn);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            tableArn,
                            { 'Ref': 'AWS::NoValue' },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe(tableArn);
        expect(stack.resolve(table.tableName)).toBe('MyTable');
    });
    test('static fromTableName(name) allows importing an external/existing table from table name', () => {
        const stack = new core_1.Stack();
        const tableName = 'MyTable';
        const table = lib_1.Table.fromTableName(stack, 'ImportedTable', tableName);
        const role = new iam.Role(stack, 'NewRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        table.grantReadWriteData(role);
        // it is possible to obtain a permission statement for a ref
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                            'dynamodb:BatchWriteItem',
                            'dynamodb:PutItem',
                            'dynamodb:UpdateItem',
                            'dynamodb:DeleteItem',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            'Ref': 'AWS::Partition',
                                        },
                                        ':dynamodb:',
                                        {
                                            'Ref': 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            'Ref': 'AWS::AccountId',
                                        },
                                        ':table/MyTable',
                                    ],
                                ],
                            },
                            {
                                'Ref': 'AWS::NoValue',
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'NewRoleDefaultPolicy90E8F49D',
            'Roles': [{ 'Ref': 'NewRole99763075' }],
        });
        expect(table.tableArn).toBe('arn:${Token[AWS::Partition.3]}:dynamodb:${Token[AWS::Region.4]}:${Token[AWS::AccountId.0]}:table/MyTable');
        expect(stack.resolve(table.tableName)).toBe(tableName);
    });
    describe('stream permissions on imported tables', () => {
        test('throw if no tableStreamArn is specified', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(() => table.grantTableListStreams(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
            expect(() => table.grantStreamRead(role)).toThrow(/DynamoDB Streams must be enabled on the table/);
        });
        test('creates the correct list streams grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantTableListStreams(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
        test('creates the correct stream read grant', () => {
            const stack = new core_1.Stack();
            const tableName = 'MyTable';
            const tableStreamArn = 'arn:foo:bar:baz:TrustMeThisIsATableStream';
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', { tableName, tableStreamArn });
            const role = new iam.Role(stack, 'NewRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
            expect(table.grantStreamRead(role)).toBeDefined();
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'dynamodb:ListStreams',
                            Effect: 'Allow',
                            Resource: stack.resolve(`${table.tableArn}/stream/*`),
                        },
                        {
                            Action: ['dynamodb:DescribeStream', 'dynamodb:GetRecords', 'dynamodb:GetShardIterator'],
                            Effect: 'Allow',
                            Resource: tableStreamArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [stack.resolve(role.roleName)],
            });
        });
        test('creates the correct index grant if indexes have been provided when importing', () => {
            const stack = new core_1.Stack();
            const table = lib_1.Table.fromTableAttributes(stack, 'ImportedTable', {
                tableName: 'MyTableName',
                globalIndexes: ['global'],
                localIndexes: ['local'],
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.AnyPrincipal(),
            });
            table.grantReadData(role);
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'dynamodb:BatchGetItem',
                                'dynamodb:GetRecords',
                                'dynamodb:GetShardIterator',
                                'dynamodb:Query',
                                'dynamodb:GetItem',
                                'dynamodb:Scan',
                            ],
                            Resource: [
                                {
                                    'Fn::Join': ['', [
                                            'arn:',
                                            { Ref: 'AWS::Partition' },
                                            ':dynamodb:',
                                            { Ref: 'AWS::Region' },
                                            ':',
                                            { Ref: 'AWS::AccountId' },
                                            ':table/MyTableName',
                                        ]],
                                },
                                {
                                    'Fn::Join': ['', [
                                            'arn:',
                                            { Ref: 'AWS::Partition' },
                                            ':dynamodb:',
                                            { Ref: 'AWS::Region' },
                                            ':',
                                            { Ref: 'AWS::AccountId' },
                                            ':table/MyTableName/index/*',
                                        ]],
                                },
                            ],
                        },
                    ],
                },
            });
        });
    });
});
describe('global', () => {
    test('create replicas', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-west-2',
            },
            Condition: 'TableStackRegionNotEqualseuwest2A03859E7',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(stack).toHaveResource('Custom::DynamoDBReplica', {
            Properties: {
                ServiceToken: {
                    'Fn::GetAtt': [
                        'awscdkawsdynamodbReplicaProviderNestedStackawscdkawsdynamodbReplicaProviderNestedStackResource18E3F12D',
                        'Outputs.awscdkawsdynamodbReplicaProviderframeworkonEventF9504691Arn',
                    ],
                },
                TableName: {
                    Ref: 'TableCD117FA1',
                },
                Region: 'eu-central-1',
            },
            Condition: 'TableStackRegionNotEqualseucentral199D46FC0',
        }, assert_1.ResourcePart.CompleteDefinition);
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toEqual({
            TableStackRegionNotEqualseuwest2A03859E7: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-west-2', { Ref: 'AWS::Region' }] },
                ],
            },
            TableStackRegionNotEqualseucentral199D46FC0: {
                'Fn::Not': [
                    { 'Fn::Equals': ['eu-central-1', { Ref: 'AWS::Region' }] },
                ],
            },
        });
    });
    test('grantReadData', () => {
        const stack = new core_1.Stack();
        const table = new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        table.addGlobalSecondaryIndex({
            indexName: 'my-index',
            partitionKey: {
                name: 'key',
                type: lib_1.AttributeType.STRING,
            },
        });
        const user = new iam.User(stack, 'User');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::GetAtt': [
                                    'TableCD117FA1',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'TableCD117FA1',
                                                'Arn',
                                            ],
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/',
                                        {
                                            Ref: 'TableCD117FA1',
                                        },
                                        '/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('grantReadData across regions', () => {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1', {
            env: { region: 'us-east-1' },
        });
        const table = new lib_1.Table(stack1, 'Table', {
            tableName: 'my-table',
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        table.addGlobalSecondaryIndex({
            indexName: 'my-index',
            partitionKey: {
                name: 'key',
                type: lib_1.AttributeType.STRING,
            },
        });
        const stack2 = new core_1.Stack(app, 'Stack2', {
            env: { region: 'eu-west-2' },
        });
        const user = new iam.User(stack2, 'User');
        // WHEN
        table.grantReadData(user);
        // THEN
        expect(stack2).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'dynamodb:BatchGetItem',
                            'dynamodb:GetRecords',
                            'dynamodb:GetShardIterator',
                            'dynamodb:Query',
                            'dynamodb:GetItem',
                            'dynamodb:Scan',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/index/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('grantTableListStreams across regions', () => {
        // GIVEN
        const app = new core_1.App();
        const stack1 = new core_1.Stack(app, 'Stack1', {
            env: { region: 'us-east-1' },
        });
        const table = new lib_1.Table(stack1, 'Table', {
            tableName: 'my-table',
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        const stack2 = new core_1.Stack(app, 'Stack2', {
            env: { region: 'eu-west-2' },
        });
        const user = new iam.User(stack2, 'User');
        // WHEN
        table.grantTableListStreams(user);
        // THEN
        expect(stack2).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'dynamodb:ListStreams',
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:us-east-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-west-2:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':dynamodb:eu-central-1:',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':table/my-table/stream/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
        });
    });
    test('throws with PROVISIONED billing mode', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            billingMode: lib_1.BillingMode.PROVISIONED,
        })).toThrow(/`PAY_PER_REQUEST`/);
    });
    test('throws when stream is set and not set to NEW_AND_OLD_IMAGES', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
            stream: lib_1.StreamViewType.OLD_IMAGE,
        })).toThrow(/`NEW_AND_OLD_IMAGES`/);
    });
    test('throws with replica in same region as stack', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'us-east-1' },
        });
        // THEN
        expect(() => new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-1',
                'us-east-1',
                'eu-west-2',
            ],
        })).toThrow(/`replicationRegions` cannot include the region where this stack is deployed/);
    });
    test('no conditions when region is known', () => {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'Stack', {
            env: { region: 'eu-west-1' },
        });
        // WHEN
        new lib_1.Table(stack, 'Table', {
            partitionKey: {
                name: 'id',
                type: lib_1.AttributeType.STRING,
            },
            replicationRegions: [
                'eu-west-2',
                'eu-central-1',
            ],
        });
        // THEN
        expect(assert_1.SynthUtils.toCloudFormation(stack).Conditions).toBeUndefined();
    });
});
function testGrant(expectedActions, invocation) {
    // GIVEN
    const stack = new core_1.Stack();
    const table = new lib_1.Table(stack, 'my-table', { partitionKey: { name: 'ID', type: lib_1.AttributeType.STRING } });
    const user = new iam.User(stack, 'user');
    // WHEN
    invocation(user, table);
    // THEN
    const action = expectedActions.length > 1 ? expectedActions.map(a => `dynamodb:${a}`) : `dynamodb:${expectedActions[0]}`;
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        'PolicyDocument': {
            'Statement': [
                {
                    'Action': action,
                    'Effect': 'Allow',
                    'Resource': [
                        {
                            'Fn::GetAtt': [
                                'mytable0324D45C',
                                'Arn',
                            ],
                        },
                        {
                            'Ref': 'AWS::NoValue',
                        },
                    ],
                },
            ],
            'Version': '2012-10-17',
        },
        'PolicyName': 'userDefaultPolicy083DF682',
        'Users': [
            {
                'Ref': 'user2C2B57AE',
            },
        ],
    });
}
//# sourceMappingURL=data:application/json;base64,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