# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/05_trig.ipynb.

# %% auto 0
__all__ = ['rotmat', 'rotate', 'ntheta', 'ctheta', 'catrad', 'rad2pnt', 'chord']

# %% ../nbs/05_trig.ipynb 6
from functools import singledispatch

# %% ../nbs/05_trig.ipynb 8
from typing import Iterable

# %% ../nbs/05_trig.ipynb 11
#| export

# %% ../nbs/05_trig.ipynb 13
try: import numpy as np
except ImportError: ...

# %% ../nbs/05_trig.ipynb 15
#| export


# %% ../nbs/05_trig.ipynb 17
#| export


# %% ../nbs/05_trig.ipynb 19
from quac import dim1, real

# %% ../nbs/05_trig.ipynb 21
from .atyp import XYArray, RotationMatrix

# %% ../nbs/05_trig.ipynb 24
def rotmat(theta: real) -> RotationMatrix:
    '''Create the rotation matrix for `theta`.
    
    Parameters
    ----------
    theta : real
        The rotation angle in radians.
    
    Returns
    -------
    RotationMatrix
        A 2x2 numpy array representing the rotation matrix
    '''
    return np.array([[np.cos(theta), -np.sin(theta)], 
                     [np.sin(theta),  np.cos(theta)]])

def rotate(m: XYArray, theta: real) -> XYArray:
    '''Rotate a matrix of 2D points by a given angle `theta`.
    
    Parameters
    ----------
    m : XYArray
        Input array of 2D points.
        
    theta : real
        The rotation angle in radians.
    
    Returns
    -------
    XYArray
        The rotated array of 2D points.
    '''
    return np.matmul(m, rotmat(theta).T)

# %% ../nbs/05_trig.ipynb 26
def ntheta(n: int) -> np.ndarray:
    '''Get `n` equidistant angles (thetas) from the unit circle.
    
    Parameters
    ----------
    n : int
        The number of angles to generate.
    
    Returns
    -------
    np.ndarray
        An array of `n` equidistant angles.
    '''
    return np.linspace(0, 2 * np.pi, n, endpoint=False)

def ctheta(n: int = 1) -> float:
    '''Categorical theta. Calculate the degree of rotation per category for a given number of categories.
    
    Parameters
    ----------
    n : int, default: 1
        The number of categories, by default 1.
    
    Returns
    -------
    float
        The degree of rotation per category.
    '''
    if n < 2: return 0
    return ntheta(n)[1] - ntheta(n)[0]

def catrad(i: int, n: int = 1) -> float:
    '''Calculates categorical theta for category `i` of `n` categories.
    
    Parameters
    ----------
    i : int
        The category index.
        
    n : int, default: 1
        The total number of categories, by default 1.
    
    Returns
    -------
    float
        The categorical theta for category `i`.
    '''
    return ctheta(n) * i

# %% ../nbs/05_trig.ipynb 28
@singledispatch
def rad2pnt(theta: real, sx: real = 1, sy: real = 1) -> np.ndarray | NotImplementedError:
    '''Polar (radians) --> Cartesian (x, y) coordinates.
    
    Parameters
    ----------
    theta : real
        Angle in radians.
        
    sx : real, default: 1
        Scale factor for the x-axis, by default 1.

    sy : real, default: 1
        Scale factor for the y-axis, by default 1.
    
    Returns
    -------
    np.ndarray | NotImplementedError
        The converted coordinates or NotImplementedError 
    '''
    raise NotImplementedError('Unsupported type')

@rad2pnt.register(np.ndarray)
@rad2pnt.register(list)
@rad2pnt.register(tuple)
def _(theta: np.ndarray, sx: float = 1, sy: float = 1):
    coord = np.column_stack([sx * np.cos(theta), sy * np.sin(theta)])
    return coord

@rad2pnt.register(float)
@rad2pnt.register(int)
@rad2pnt.register(real)
def _(theta: float, sx: float = 1, sy: float = 1):
    return rad2pnt.dispatch(np.ndarray)(theta, sx, sy)[0]

# %% ../nbs/05_trig.ipynb 30
@singledispatch
def chord(a: real, b: real, sx: real = 1, sy: real = 1) -> np.ndarray | float | NotImplementedError:
    '''Calculate the chord length between two angles.
    
    Parameters
    ----------
    a : real
        The starting angle in radians.
        
    b : real
        The ending angle in radians.
        
    sx : real, default: 1
        Scale factor for the x-axis, by default 1.
        
    sy : real, default: 1
        Scale factor for the y-axis, by default 1.
    
    Returns
    -------
    The chord length for specified conditions or raises NotImplementedError for unsupported types.
    '''
    raise NotImplementedError('Unsupported type')

@chord.register(np.ndarray)
def _(thetas: np.ndarray, y: np.ndarray = None, sx: real = 1, sy: real = 1):
    shape = np.asarray(thetas).shape        
    if isinstance(y, np.ndarray) and y.shape == shape: 
        return chord.dispatch(np.ndarray)(np.stack((thetas, y)), sx=sx, sy=sy)
    if len(shape) == 1 and shape[0] >= 2: return chord.dispatch(dim1)(thetas, sx=sx, sy=sy)
    if len(shape) >= 2 and shape[1] >= 2: x, y, *_ = thetas[:2]
    return np.linalg.norm(x - y) 
   
@chord.register(list)
def _(thetas: dim1, sx: real = 1, sy: real = 1):
    return chord.dispatch(np.ndarray)(np.asarray(thetas), sx=sx, sy=sy)

@chord.register(dim1)
def _(thetas: dim1, sx: real = 1, sy: real = 1):
    return chord.dispatch(np.ndarray)(rad2pnt(thetas, sx, sy)[:2], sx=sx, sy=sy)
   
@chord.register(float)
@chord.register(int)
@chord.register(real)
def _(x: real, y: real, sx: real = 1, sy: real = 1):
    return chord.dispatch(np.ndarray)(rad2pnt([x, y], sx, sy), sx=sx, sy=sy)

# %% ../nbs/05_trig.ipynb 32
#| export
