# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/20_poly.ipynb.

# %% auto 0
__all__ = ['ellipse', 'orbit', 'circle', 'subring', 'diamond_quadrant', 'diamond']

# %% ../nbs/20_poly.ipynb 6
from itertools import product

# %% ../nbs/20_poly.ipynb 8
#| export


# %% ../nbs/20_poly.ipynb 11
#| export

# %% ../nbs/20_poly.ipynb 13
try: import numpy as np
except ImportError: ...

# %% ../nbs/20_poly.ipynb 15
#| export

# %% ../nbs/20_poly.ipynb 17
from quac import real

# %% ../nbs/20_poly.ipynb 19
from .atyp import XYArray
from .trig import rotate, ntheta, rad2pnt

# %% ../nbs/20_poly.ipynb 21
def ellipse(p: int, sx: real = 1, sy: real = 1, ra: real = 0) -> XYArray:
    '''Create an ellipse.
    
    Parameters
    ----------
    p : int
        Number of points to generate around the ellipse.
    
    sx : real, default: 1
        Scale factor for the x-axis of the ellipse, by default 1.
    
    sy : real, default: 1
        Scale factor for the y-axis of the ellipse, by default 1.
    
    ra : real, default: 0
        Rotation angle of the ellipse in radians, by default 0.
    
    Returns
    -------
    XYArray
        An array of points representing the ellipse.
    '''
    return rotate(rad2pnt(ntheta(p), sx, sy), ra)

def orbit(p: int, sx: real = 0.2) -> XYArray:
    '''Generates an orbit.
    
    Parameters
    ----------
    p : int
        Number of points to generate along the orbit.
    
    sx : real, default: 0.2
        Scale factor for the x-axis of the orbit, by default 0.2.
    
    Returns
    -------
    XYArray
        An array of points representing the orbit.
    '''
    return ellipse(p, sx, 1, 0)

def circle(p: int, r: real = 1) -> XYArray:
    '''Create a circle.
    
    Parameters
    ----------
    p : int
        Number of points to generate around the circle.
        
    r : real, default: 1
        Radius of the circle, by default 1.
    
    Returns
    -------
    XYArray
        An array of points representing the circle.
    '''
    return ellipse(p, r, r, 0)

def subring(p: int, r: real = 1, t: real = 0, sx: real = 0.5, sy: real = 0.5) -> XYArray:
    '''Subring along a circle.
    
    Parameters
    ----------
    p : int
        Number of points to generate around the subring.
        
    r : real, default: 1
        Radius of the circle from which the subring is generated, by default 1.
    
    t : real, default: 0
        Theta, the angle at which to generate the subring, by default 0.
    
    sx : real, default: 0.5
        Scale factor for the x-axis, by default 0.5.
    
    sy : real, default: 0.5
        Scale factor for the y-axis, by default 0.5.
    
    Returns
    -------
    XYArray
        An array of points representing the subring.
    '''
    return circle(p, r) - rad2pnt(t, sx, sy)

# %% ../nbs/20_poly.ipynb 23
def diamond_quadrant(p: int, w: real = 1, h: real = 1, top: bool = True, left: bool = True) -> XYArray:
    '''Generates one quadrant of a diamond.
    
    Parameters
    ----------
    p : int
        Number of points to generate within the quadrant.
        
    w : real, default: 1
        Width of the diamond.
        
    h : real, default: 1
        Height of the diamond.
        
    top : bool, default: True
        Whether the quadrant is at the top, by default True.
        
    left : bool, default: True
        Whether the quadrant is on the left, by default True.
    
    Returns
    -------
    XYArray
        An array of points representing one quadrant of a diamond.
    '''
    hw, hh = w / 2, h / 2
    match (top, left):
        case (True, True): # top left
            x = np.linspace(-hw, 0, p)
            y = np.linspace( hh, h, p)
        case (True, False): # top right
            x = np.linspace(hw, 0, p)
            y = np.linspace(hh, h, p)
        case (False, True): # bottom left
            x = np.linspace(0, -hw, p)
            y = np.linspace(0,  hh, p)
        case (False, False): # bottom right
            x = np.linspace(0, hw, p)
            y = np.linspace(0, hh, p)
        case _: raise NotImplementedError
    quadrant = np.vstack((x, y)).T
    return quadrant
     
def diamond(p: int, w: real = 1, h: real = 1) -> XYArray:
    '''Generates the four vertices of a diamond.
    
    Parameters
    ----------
    p : int
        Total number of points to generate for the diamond.
        
    w : real, default: 1
        Width of the diamond.
        
    h : real, default: 1
        Height of the diamond.
    
    Returns
    -------
    XYArray
        An array of points representing the diamond.
    '''
    q = int(np.floor(p / 4))
    combs = product((True, False), (True, False))
    return np.vstack(tuple(diamond_quadrant(q, w, h, t, l) for t, l in combs))

# %% ../nbs/20_poly.ipynb 25
#| export
