# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/10_data.ipynb.

# %% auto 0
__all__ = ['addz', 'npxy', 'addx', 'dfxy']

# %% ../nbs/10_data.ipynb 6
#| export


# %% ../nbs/10_data.ipynb 8
#| export


# %% ../nbs/10_data.ipynb 11
#| export

# %% ../nbs/10_data.ipynb 13
try: import numpy as np
except ImportError: ...

try: import pandas as pd
except ImportError: ...

# %% ../nbs/10_data.ipynb 15
#| export


# %% ../nbs/10_data.ipynb 17
#| export


# %% ../nbs/10_data.ipynb 19
from quac import intq
from utrc.seed import seedall

# %% ../nbs/10_data.ipynb 21
from .atyp import XYArray, LabelArray
from .cats import catx

# %% ../nbs/10_data.ipynb 24
def addz(data: XYArray = None, zscale: float = 0.02, seed: int = 3) -> XYArray:
    '''Add Gaussian noize to data.
    
    Parameters
    ----------
    data : XYArray, optional
        The input array to add noise to. If None, a new array will be created.
        
    zscale : float, default: 0.02
        The standard deviation of the Gaussian noise, by default 0.02.
        
    seed : int, default: 3
        A seed for the random number generator to ensure reproducibility, by default 3.
    
    Returns
    -------
    XYArray
        The input array with Gaussian noise added.
    '''
    seedall(seed)
    # add Gaussian noise
    return data + np.random.normal(0, zscale, np.array(data).shape)

# %% ../nbs/10_data.ipynb 26
def npxy(data: XYArray = None, cats: LabelArray = None, seed: int = 3) -> tuple[XYArray, LabelArray]:
    '''Initializes storage variables for 2D data and categories using numpy.
    
    Parameters
    ----------
    data : XYArray, optional
        Initial 2D data array. If None, an empty array is created.
        
    cats : LabelArray, optional
        Initial categories array. If None, an empty array is created.
        
    seed : int, default: 3
        Seed for random number generation, used for reproducibility, by default 3.
    
    Returns
    -------
    tuple[XYArray, LabelArray]
        A tuple containing the initialized data and categories arrays.
    '''
    seedall(seed)
    if data is None: data = np.empty((0, 2))
    if cats is None: cats = np.empty(0, dtype=int)
    return data, cats

# %% ../nbs/10_data.ipynb 28
def addx(
    i: int, newx: XYArray, data: XYArray = None, cats: LabelArray = None,
    zscale: float = 0.02, distcats: bool = True, ncats: intq = 1, seed: int = 3,
) -> tuple[XYArray, LabelArray]:
    '''Adds new points and their categories to the dataset.
    
    Parameters
    ----------
    i : int
        The category index for the new points.
        
    newx : XYArray
        The new points to be added.
        
    data : XYArray, optional
        The existing dataset to which the new points will be added. If None, it is initialized.
        
    cats : LabelArray, optional
        The existing category labels to which the new labels will be added. If None, it is initialized.
        
    zscale : float, default: 0.02
        Scale of the Gaussian noise to add to new points, by default 0.02.
        
    distcats : bool, default: True
        Whether to categorize the new data by distance from the origin, by default True.
        
    ncats : int, default: 1
        The number of categories to distribute the new points into, by default 1.
        
    seed : int, default: 3
        Seed for random number generation, used for reproducibility, by default 3.
    
    Returns
    -------
    tuple[XYArray, LabelArray]
        The updated data and categories arrays.
    '''
    data, cats = npxy(data, cats, seed)
    newx = addz(newx, zscale, seed)
    newy = catx(newx, i, ncats, distcats)
    data, cats = np.vstack((data, newx)), np.hstack((cats, newy))
    return data, cats

# %% ../nbs/10_data.ipynb 30
def dfxy(x: XYArray, y: LabelArray, label: str = 'label', use_index: bool = True) -> pd.DataFrame:
    '''Generates a dataset from 2D coordinates (`x`) and labels (`y`)
    
    Parameters
    ----------
    x : XYArray
        2D coordinates array.
        
    y : LabelArray
        Labels array corresponding to the x coordinates.
        
    label : str, default: 'label'
        The name of the column to store the labels, by default 'label'.
        
    use_index : bool, default: True
        Whether to use the labels as the DataFrame index, by default True.
    
    Returns
    -------
    pd.DataFrame
        The generated DataFrame containing coordinates and labels.
    '''
    index = pd.CategoricalIndex(y, name=label)
    df = pd.DataFrame(x, columns=['x', 'y'], index=index)
    if not use_index: df.reset_index(inplace=True)
    return df

# %% ../nbs/10_data.ipynb 32
#| export
