# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/06_cats.ipynb.

# %% auto 0
__all__ = ['catdists', 'catx', 'negcats', 'categorical_collate_fn', 'torch_categorical_sample']

# %% ../nbs/06_cats.ipynb 6
from functools import wraps, partial

# %% ../nbs/06_cats.ipynb 8
#| export


# %% ../nbs/06_cats.ipynb 11
#| export

# %% ../nbs/06_cats.ipynb 13
try: import numpy as np
except ImportError: ...

try: import pandas as pd
except ImportError: ...

# %% ../nbs/06_cats.ipynb 15
try: import torch
except ImportError: ...

# %% ../nbs/06_cats.ipynb 17
#| export


# %% ../nbs/06_cats.ipynb 19
from nlit import DROP
from quac import intq

from nlit import LABEL, MPS
from utrc.atyp import P
from utrc.smpl import categorical_sample
from etrc import BatchReturn, DataFormat

# %% ../nbs/06_cats.ipynb 21
from .atyp import XYArray, LabelArray
from .cons import FORMAT, OUTPUT

# %% ../nbs/06_cats.ipynb 23
#| export


# %% ../nbs/06_cats.ipynb 26
def catdists(data: XYArray, n: int) -> LabelArray:
    '''Categorizes data by distances from the origin into `n` classes.
    
    Parameters
    ----------
    data : XYArray
        Input data array where each element is a point in 2D space.
        
    n : int
        Number of categories to divide the data into, based on distance from the origin.
    
    Returns
    -------
    LabelArray
        An array of category labels for each point in `data`.
    '''
    dist = np.linalg.norm(data, axis=1)
    return pd.qcut(dist, q=n, labels=False, duplicates=DROP)

# %% ../nbs/06_cats.ipynb 28
def catx(data: XYArray = None, i: int = 0, ncats: intq = 1, distcats: bool = True) -> LabelArray:
    '''Categorizes data into `ncats` categories, either uniformly or based on distance.
    
    Parameters
    ----------
    data : XYArray, optional
        Input data array where each element is a point in 2D space.
        
    i : int, default: 0
        Index to use for category assignment if `distcats` is False, by default 0.
    
    ncats : int, default: 1
        Number of categories, by default 1.
        
    distcats : bool, default: True
        Whether to categorize by distance (`True`) or uniformly (`False`), by default True.
    
    Returns
    -------
    LabelArray
        An array of category labels for each point in `data`.
    '''
    return catdists(data, ncats) if distcats else np.full(data.shape[0], i)

# %% ../nbs/06_cats.ipynb 30
def negcats(df: pd.DataFrame, label: str = 'label') -> pd.DataFrame:
    '''Converts categories with x / y values less than zero into new "negative" categories
    temporarily, before rescaling all categories.
    
    Parameters
    ----------
    df : pd.DataFrame
        The input DataFrame containing 'x' and 'y' columns.
        
    label : str, default: 'label'
        The column name to store category labels, by default 'label'.
    
    Returns
    -------
    pd.DataFrame
        The DataFrame with updated categories.
    '''

    idx = ((df.x <= 0) & (0 < df.y)) | ((df.y <= 0) & (0 < df.x))
    df[label] += 1
    df.loc[idx, label] *= -1
    df[label] += df[label].min()
    df[label] *= -1
    return df

# %% ../nbs/06_cats.ipynb 32
def categorical_collate_fn(xys: tuple[tuple[torch.Tensor, torch.Tensor], ...]) -> tuple[torch.Tensor, torch.Tensor]:
    return (torch.concat(tuple(x for x, _ in xys), dim=1), torch.concat(tuple(y for _, y in xys), dim=1))

# %% ../nbs/06_cats.ipynb 33
@wraps(categorical_sample)
def torch_categorical_sample(
    df: pd.DataFrame, 
    col: str = LABEL, n: int = 10, 
    replace: bool = False, 
    device: str = MPS, 
    *args: P.args, 
    **kwargs: P.kwargs,
):
    
    kwargs.pop(FORMAT, None)
    kwargs.pop(OUTPUT, None)
    fn = partial(categorical_sample, device=device, format=DataFormat.torch, output=BatchReturn.xy)
    return fn(df, col, n, replace, *args, **kwargs)

# %% ../nbs/06_cats.ipynb 35
#| export
