"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GlueStartJobRun = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Starts an AWS Glue job in a Task state.
 *
 * OUTPUT: the output of this task is a JobRun structure, for details consult
 * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-JobRun
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-glue.html
 */
class GlueStartJobRun extends sfn.TaskStateBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS);
        this.taskPolicies = this.getPolicies();
        this.taskMetrics = {
            metricPrefixSingular: 'GlueJob',
            metricPrefixPlural: 'GlueJobs',
            metricDimensions: { GlueJobName: this.props.glueJobName },
        };
    }
    /**
     * @internal
     */
    _renderTask() {
        var _a, _b;
        const notificationProperty = this.props.notifyDelayAfter ? { NotifyDelayAfter: this.props.notifyDelayAfter.toMinutes() } : null;
        return {
            Resource: task_utils_1.integrationResourceArn('glue', 'startJobRun', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                JobName: this.props.glueJobName,
                Arguments: (_a = this.props.arguments) === null || _a === void 0 ? void 0 : _a.value,
                Timeout: (_b = this.props.timeout) === null || _b === void 0 ? void 0 : _b.toMinutes(),
                SecurityConfiguration: this.props.securityConfiguration,
                NotificationProperty: notificationProperty,
            }),
            TimeoutSeconds: undefined,
        };
    }
    getPolicies() {
        let iamActions;
        if (this.integrationPattern === sfn.IntegrationPattern.REQUEST_RESPONSE) {
            iamActions = ['glue:StartJobRun'];
        }
        else if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            iamActions = [
                'glue:StartJobRun',
                'glue:GetJobRun',
                'glue:GetJobRuns',
                'glue:BatchStopJobRun',
            ];
        }
        return [new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(this).formatArn({
                        service: 'glue',
                        resource: 'job',
                        resourceName: this.props.glueJobName,
                    }),
                ],
                actions: iamActions,
            })];
    }
}
exports.GlueStartJobRun = GlueStartJobRun;
GlueStartJobRun.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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