"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsRunTask = exports.EcsEc2LaunchTarget = exports.EcsFargateLaunchTarget = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Configuration for running an ECS task on Fargate.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-fargate
 */
class EcsFargateLaunchTarget {
    /**
     *
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the Fargate launch type configured on RunTask.
     */
    bind(_task, launchTargetOptions) {
        var _a;
        if (!launchTargetOptions.taskDefinition.isFargateCompatible) {
            throw new Error('Supplied TaskDefinition is not compatible with Fargate');
        }
        return {
            parameters: {
                LaunchType: 'FARGATE',
                PlatformVersion: (_a = this.options) === null || _a === void 0 ? void 0 : _a.platformVersion,
            },
        };
    }
}
exports.EcsFargateLaunchTarget = EcsFargateLaunchTarget;
/**
 * Configuration for running an ECS task on EC2.
 *
 * @see https://docs.aws.amazon.com/AmazonECS/latest/userguide/launch_types.html#launch-type-ec2
 */
class EcsEc2LaunchTarget {
    /**
     *
     */
    constructor(options) {
        this.options = options;
    }
    /**
     * Called when the EC2 launch type is configured on RunTask.
     */
    bind(_task, launchTargetOptions) {
        var _a, _b, _c, _d, _e;
        if (!launchTargetOptions.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not compatible with EC2');
        }
        if (!((_a = launchTargetOptions.cluster) === null || _a === void 0 ? void 0 : _a.hasEc2Capacity)) {
            throw new Error('Cluster for this service needs Ec2 capacity. Call addCapacity() on the cluster.');
        }
        return {
            parameters: {
                LaunchType: 'EC2',
                // takes an array of placement constraints each of which contain a single item array of constraints, flattens it
                // and renders the Json to be passed as a parameter in the state machine.
                // input: [ecs.PlacementConstraint.distinctInstances()] - distinctInstances() returns [{ type: 'distinctInstance' }]
                // output: {Type: 'distinctInstance'}
                PlacementConstraints: noEmpty(flatten(((_c = (_b = this.options) === null || _b === void 0 ? void 0 : _b.placementConstraints) !== null && _c !== void 0 ? _c : []).map((c) => c.toJson().map(uppercaseKeys)))),
                PlacementStrategy: noEmpty(flatten(((_e = (_d = this.options) === null || _d === void 0 ? void 0 : _d.placementStrategies) !== null && _e !== void 0 ? _e : []).map((c) => c.toJson().map(uppercaseKeys)))),
            },
        };
        function uppercaseKeys(obj) {
            const ret = {};
            for (const key of Object.keys(obj)) {
                ret[key.slice(0, 1).toUpperCase() + key.slice(1)] = obj[key];
            }
            return ret;
        }
        function flatten(xs) {
            return Array.prototype.concat([], ...xs);
        }
        function noEmpty(xs) {
            if (xs.length === 0) {
                return undefined;
            }
            return xs;
        }
    }
}
exports.EcsEc2LaunchTarget = EcsEc2LaunchTarget;
/**
 * Run a Task on ECS or Fargate.
 */
class EcsRunTask extends sfn.TaskStateBase {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, props);
        this.props = props;
        /**
         * Manage allowed network traffic for this service.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.containerOverrides)) {
            throw new Error('Task Token is required in `containerOverrides` for callback. Use Context.taskToken to set the token.');
        }
        if (!this.props.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
        if (this.props.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworking();
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroup from ASG.
            this.validateNoNetworkingProps();
            this.connections.addSecurityGroup(...this.props.cluster.connections.securityGroups);
        }
        for (const override of (_b = this.props.containerOverrides) !== null && _b !== void 0 ? _b : []) {
            const name = override.containerDefinition.containerName;
            if (!cdk.Token.isUnresolved(name)) {
                const cont = this.props.taskDefinition.node.tryFindChild(name);
                if (!cont) {
                    throw new Error(`Overrides mention container with name '${name}', but no such container in task definition`);
                }
            }
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('ecs', 'runTask', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                Cluster: this.props.cluster.clusterArn,
                TaskDefinition: this.props.taskDefinition.taskDefinitionArn,
                NetworkConfiguration: this.networkConfiguration,
                Overrides: renderOverrides(this.props.containerOverrides),
                ...this.props.launchTarget.bind(this, { taskDefinition: this.props.taskDefinition, cluster: this.props.cluster }).parameters,
            }),
        };
    }
    configureAwsVpcNetworking() {
        var _a, _b;
        const subnetSelection = (_a = this.props.subnets) !== null && _a !== void 0 ? _a : { subnetType: this.props.assignPublicIp ? ec2.SubnetType.PUBLIC : ec2.SubnetType.PRIVATE };
        this.networkConfiguration = {
            AwsvpcConfiguration: {
                AssignPublicIp: this.props.assignPublicIp ? (this.props.assignPublicIp ? 'ENABLED' : 'DISABLED') : undefined,
                Subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                SecurityGroups: cdk.Lazy.listValue({ produce: () => { var _a; return (_a = this.securityGroups) === null || _a === void 0 ? void 0 : _a.map(sg => sg.securityGroupId); } }),
            },
        };
        // Make sure we have a security group if we're using AWSVPC networking
        this.securityGroups = (_b = this.props.securityGroups) !== null && _b !== void 0 ? _b : [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc: this.props.cluster.vpc })];
        this.connections.addSecurityGroup(...this.securityGroups);
    }
    validateNoNetworkingProps() {
        if (this.props.subnets !== undefined || this.props.securityGroups !== undefined) {
            throw new Error(`Supplied TaskDefinition must have 'networkMode' of 'AWS_VPC' to use 'vpcSubnets' and 'securityGroup'. Received: ${this.props.taskDefinition.networkMode}`);
        }
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/ecs-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.props.taskDefinition.taskDefinitionArn],
            }),
            new iam.PolicyStatement({
                actions: ['ecs:StopTask', 'ecs:DescribeTasks'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: this.taskExecutionRoles().map((r) => r.roleArn),
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForECSTaskRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
    taskExecutionRoles() {
        // Need to be able to pass both Task and Execution role, apparently
        const ret = new Array();
        ret.push(this.props.taskDefinition.taskRole);
        if (this.props.taskDefinition.executionRole) {
            ret.push(this.props.taskDefinition.executionRole);
        }
        return ret;
    }
}
exports.EcsRunTask = EcsRunTask;
EcsRunTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderOverrides(containerOverrides) {
    var _a;
    if (!containerOverrides || containerOverrides.length === 0) {
        return undefined;
    }
    const ret = new Array();
    for (const override of containerOverrides) {
        ret.push({
            Name: override.containerDefinition.containerName,
            Command: override.command,
            Cpu: override.cpu,
            Memory: override.memoryLimit,
            MemoryReservation: override.memoryReservation,
            Environment: (_a = override.environment) === null || _a === void 0 ? void 0 : _a.map((e) => ({
                Name: e.name,
                Value: e.value,
            })),
        });
    }
    return { ContainerOverrides: ret };
}
//# sourceMappingURL=data:application/json;base64,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