"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunBatchJob = void 0;
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * (deprecated) A Step Functions Task to run AWS Batch.
 *
 * @deprecated use `BatchSubmitJob`
 */
class RunBatchJob {
    /**
     * @deprecated
     */
    constructor(props) {
        var _a;
        this.props = props;
        // validate integrationPattern
        this.integrationPattern =
            props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call RunBatchJob.`);
        }
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`timeout must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is reuqired since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_a = props.containerOverrides) === null || _a === void 0 ? void 0 : _a.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
    }
    /**
     * (deprecated) Called when the task object is used in a workflow.
     *
     * @deprecated
     */
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('batch', 'submitJob', this.integrationPattern),
            policyStatements: this.configurePolicyStatements(_task),
            parameters: {
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: this.props.payload,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            },
        };
    }
    configurePolicyStatements(task) {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: containerOverrides.memory,
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.RunBatchJob = RunBatchJob;
//# sourceMappingURL=data:application/json;base64,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