from .ssh_client_base import SSHClient

# ------------------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------------------
def generate_key_cmd(args):
    client = SSHClient()
    client.generate_ssh_key(
        key_type=args.type,
        bits=args.bits,
        email=args.email,
        key_name=args.key_name,
        passphrase=args.passphrase,
        overwrite_key=args.overwrite,
    )

def list_local_keys_cmd(args):
    client = SSHClient()
    keys = client.list_ssh_keys_from_local_env()
    if keys:
        print("🔑 Available SSH Public Keys:")
        for key in keys:
            print(f" - {key}")
    else:
        print("⚠️ No SSH public keys found in ~/.ssh")

def delete_local_key_cmd(args):
    client = SSHClient()
    client.delete_ssh_key_from_local_env(args.key_name)

def expose_public_key_cmd(args):
    client = SSHClient()
    client.expose_public_key(args.key_name)

def set_git_credentials_cmd(args):
    scope_map = {"global": True, "local": False}
    SSHClient.set_git_credentials(
        user_name=args.name,
        user_email=args.email,
        global_scope=scope_map[args.scope]
    )

def update_ssh_config_cmd(args):
    client = SSHClient()
    client.update_ssh_config(
        private_key_path=args.private_key_path,
        hostname=args.hostname,
        alias=args.alias,
        user=args.user,
        port=args.port
    )

def reset_ssh_config_cmd(args):
    client = SSHClient()
    client.reset_ssh_config(
        delete_config=args.reset_config,
        delete_known_hosts=args.reset_known_hosts
    )
# ------------------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------------------
def add_git_ssh_subcommands(subparsers):
    git_ssh_parser = subparsers.add_parser("git-ssh", help="SSH key and Git provider management")
    git_ssh_subparsers = git_ssh_parser.add_subparsers(dest="git_ssh_cmd", required=True)
# ------------------------------------------------------------------------------------------------------------
# ------------------------------------------------------------------------------------------------------------
    gen_parser = git_ssh_subparsers.add_parser(
        "generate-key", 
        help="Generate a new SSH key pair"
    )

    gen_parser.add_argument(
        "--type",
        default="ed25519",
        choices=["rsa", "ecdsa", "ed25519"],
        help=(
            "Specifies the type of SSH key to generate.\n"
            "Options: [rsa, ecdsa, ed25519].\n"
            "Default: ed25519"
        )
    )

    gen_parser.add_argument(
        "--bits",
        type=int,
        help=(
            "Sets the key size in bits.\n"
            "Options:\n"
            "  - RSA: 2048, 4096\n"
            "  - ECDSA: 256, 384, 521\n"
            "  - ED25519: ignored.\n"
            "Default: None"
        )
    )

    gen_parser.add_argument(
        "--email",
        type=str,
        default="",
        help=(
            "Associates an email comment with the public key.\n"
            "Default: \"\""
        )
    )

    gen_parser.add_argument(
        "--key-name",
        type=str,
        help=(
            "Sets the base name for the key pair files.\n"
            "Default: autogenerated if not provided"
        )
    )

    gen_parser.add_argument(
        "--passphrase",
        type=str,
        default="",
        help=(
            "Encrypts the private key using the given passphrase.\n"
            "Default: \"\""
        )
    )

    gen_parser.add_argument(
        "--overwrite",
        action="store_true",
        help=(
            "Overwrites existing key files if they already exist.\n"
            "Default: False"
        )
    )

    gen_parser.set_defaults(func=generate_key_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
    list_parser = git_ssh_subparsers.add_parser(
        "list-local-keys",
        help="List all local SSH public keys"
    )

    list_parser.set_defaults(func=list_local_keys_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
    delete_parser = git_ssh_subparsers.add_parser(
        "delete-local-key",
        help="Delete a local SSH key pair"
    )

    delete_parser.add_argument(
        "key_name",
        help=(
            "Name of the SSH key pair to delete (do not include the .pub extension).\n"
            "Deletes both private and public key files from the local environment."
        )
    )

    delete_parser.set_defaults(func=delete_local_key_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
    expose_parser = git_ssh_subparsers.add_parser(
        "expose-public-key",
        help="Expose and/or copy a public SSH key to clipboard"
    )

    expose_parser.add_argument(
        "key_name",
        help=(
            "Name of the public key file to expose (e.g., id_ed25519.pub).\n"
            "Prints the key to terminal and attempts to copy it to clipboard."
        )
    )

    expose_parser.set_defaults(func=expose_public_key_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
    git_config_parser = git_ssh_subparsers.add_parser(
        "set-git-credentials",
        help="Configure Git user name and email"
    )

    git_config_parser.add_argument(
        "--name",
        required=True,
        type=str,
        help="Set Git user name for Git credentials."
    )

    git_config_parser.add_argument(
        "--email",
        required=True,
        type=str,
        help="Set Git user email for Git credentials."
    )

    git_config_parser.add_argument(
        "--scope",
        choices=["global", "local"],
        default="global",
        help=(
            "Scope to apply Git credentials.\n"
            "Options: [global, local].\n"
            "Default: global"
        )
    )

    git_config_parser.set_defaults(func=set_git_credentials_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
    reset_config_parser = git_ssh_subparsers.add_parser(
        "reset-ssh-config",
        help="Reset SSH config and known_hosts files"
    )

    reset_config_parser.add_argument(
        "--reset-config",
        action="store_true",
        help="Remove all entries from the SSH config file."
    )

    reset_config_parser.add_argument(
        "--reset-known-hosts",
        action="store_true",
        help="Remove all entries from the known_hosts file."
    )

    reset_config_parser.set_defaults(func=reset_ssh_config_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
    update_config_parser = git_ssh_subparsers.add_parser(
        "update-ssh-config",
        help="Add a new SSH config entry and scan known host"
    )

    update_config_parser.add_argument(
        "--private-key-path",
        required=True,
        type=str,
        help="Path to the private SSH key to be used for this host."
    )

    update_config_parser.add_argument(
        "--hostname",
        required=True,
        type=str,
        help="Hostname of the SSH server (e.g., github.com)."
    )

    update_config_parser.add_argument(
        "--alias",
        required=True,
        type=str,
        help="Alias name to use in SSH config (used with ssh <alias>)."
    )

    update_config_parser.add_argument(
        "--user",
        required=True,
        type=str,
        help="SSH username for authentication."
    )

    update_config_parser.add_argument(
        "--port",
        type=int,
        default=22,
        help=(
            "Port number for the SSH connection.\n"
            "Default: 22"
        )
    )

    update_config_parser.set_defaults(func=update_ssh_config_cmd)
    # ------------------------------------------------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------
