from typing import List

from pypanther.base import PantherRule, PantherRuleTest, PantherSeverity
from pypanther.helpers.panther_base_helpers import crowdstrike_process_alert_context, is_base64
from pypanther.log_types import LogType

crowdstrike_base64_encoded_args_tests: List[PantherRuleTest] = [
    PantherRuleTest(
        Name="Command Line Tool Execution with Base64 Argument (Positive)",
        ExpectedResult=True,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": '"C:\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe" -EncodedCommand "aGVsbG93b3JsZA==" -SomeExtraCommand "HelloWorld"',
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution with Base64 Argument 2 (Positive)",
        ExpectedResult=True,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": '"C:\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe" -EncodedCommand c29tZXRoaW5n -SomeExtraCommand "HelloWorld"',
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution without Base64 Argument (Negative)",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": '"C:\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe" -WriteHost "HelloWorld123" wget test = good',
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Mac - Git",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.4.0016304.11",
            "ConfigStateHash": "3521399940",
            "Entitlements": "15",
            "aid": "1ba46982062b43redacted",
            "aip": "1.1.1.1",
            "cid": "712bcd164963442ea43d5redacted",
            "event": {
                "CodeSigningFlags": "570503953",
                "CommandLine": "/Applications/Sourcetree.app/Contents/Resources/git_local/bin/git --no-pager --EncodedString aGVsbG93b3JsZA==",
                "ConfigBuild": "1007.4.0016304.11",
                "ConfigStateHash": "3521399940",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "GID": "20",
                "ImageFileName": "/Applications/Sourcetree.app/Contents/Resources/git_local/bin/git",
                "MD5HashData": "redacted",
                "MachOSubType": "1",
                "ParentBaseFileName": "Sourcetree",
                "ParentProcessId": "466750419375415990",
                "ProcessEndTime": "",
                "ProcessGroupId": "468193711305251738",
                "ProcessStartTime": "1675441378.504",
                "RGID": "501",
                "RUID": "501",
                "RawProcessId": "30138",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "f154d8e18001e2fb7ae36d1eca1e833ddac057dd946fbb76ee14121a5e293538",
                "SVGID": "20",
                "SVUID": "501",
                "SessionProcessId": "466750413792797092",
                "SigningId": "git",
                "SourceProcessId": "466750419375415990",
                "SourceThreadId": "0",
                "Tags": "12094627905582, 12094627906234",
                "TargetProcessId": "468193711305251738",
                "UID": "501",
                "aid": "1ba46982062b43redacted",
                "aip": "1.1.1.1",
                "cid": "712bcd164963442ea43d5redacted",
                "event_platform": "Mac",
                "event_simpleName": "ProcessRollup2",
                "id": "21fd2f8b-de5f-4ab0-b188-4bbb8c80224d",
                "name": "ProcessRollup2MacV7",
                "timestamp": "1675441380830",
            },
            "event_platform": "Mac",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "21fd2f8b-de5f-4ab0-b188-4bbb8c80224d",
            "name": "ProcessRollup2MacV7",
            "p_any_ip_addresses": ["1.1.1.1"],
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-02-03 16:23:00.83",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution without Base64 Argument (Negative) 2",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": "/c powercfg /requests",
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution without Base64 Argument (Negative) 3",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": "powershell.exe  -c D:\\Atlas\\Scripts\\CopyBakToS3Job.ps1",
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution without Base64 Argument (Negative) 4",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": "Powershell.exe  -NonInteractive -NoProfile -ExecutionPolicy Bypass -NoLogo -File C:\\opt\\sensu\\plugins\\check-windows-disk.ps1 90 95",
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution without Base64 Argument (Negative) 5",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": 'C:\\windows\\system32\\cmd.exe  /S /D /c" ECHO F"',
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
    PantherRuleTest(
        Name="Command Line Tool Execution without Base64 Argument (Negative) 6",
        ExpectedResult=False,
        Log={
            "ConfigBuild": "1007.3.0016606.11",
            "ConfigStateHash": "3645117824",
            "Entitlements": "15",
            "TreeId": "4295752857",
            "aid": "877761efa8db44d792ddc2redacted",
            "aip": "1.1.1.1",
            "cid": "cfe698690964434083fecdredacted",
            "event": {
                "AuthenticationId": "293628",
                "AuthenticodeHashData": "98a4762f52a",
                "CommandLine": 'cmd /c "node -v"',
                "ConfigBuild": "1007.3.0016606.11",
                "ConfigStateHash": "3645117824",
                "EffectiveTransmissionClass": "2",
                "Entitlements": "15",
                "ImageFileName": "\\Device\\HarddiskVolume2\\Windows\\System32\\WindowsPowerShell\\v1.0\\powershell.exe",
                "ImageSubsystem": "3",
                "IntegrityLevel": "12288",
                "MD5HashData": "c031e215b8b08c752bf362f6d4c5d3ad",
                "ParentAuthenticationId": "293628",
                "ParentBaseFileName": "pwsh.exe",
                "ParentProcessId": "4370948876",
                "ProcessCreateFlags": "1024",
                "ProcessEndTime": "",
                "ProcessParameterFlags": "24577",
                "ProcessStartTime": "1682368414.719",
                "ProcessSxsFlags": "64",
                "RawProcessId": "3120",
                "SHA1HashData": "0000000000000000000000000000000000000000",
                "SHA256HashData": "840e1f9dc5a29bebf01626822d7390251e9cf05bb3560ba7b68bdb8a41cf08e3",
                "SessionId": "2",
                "SignInfoFlags": "8683538",
                "SourceProcessId": "4370948876",
                "SourceThreadId": "112532918543",
                "Tags": "25, 27, 40, 151, 874, 924, 12094627905582, 12094627906234, 211106232533012, 263882790666253",
                "TargetProcessId": "10413665481",
                "TokenType": "1",
                "TreeId": "4295752857",
                "UserSid": "S-1-5-21-239183934-720705223-383019856-500",
                "aid": "877761efa8db44d792ddc2redacted",
                "aip": "1.1.1.1",
                "cid": "cfe698690964434083fecdredacted",
                "event_platform": "Win",
                "event_simpleName": "ProcessRollup2",
                "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
                "name": "ProcessRollup2V19",
                "timestamp": "1682368416719",
            },
            "event_platform": "Win",
            "event_simpleName": "ProcessRollup2",
            "fdr_event_type": "ProcessRollup2",
            "id": "b0c07877-f288-49f8-8cb3-150149a557b2",
            "name": "ProcessRollup2V19",
            "p_log_type": "Crowdstrike.FDREvent",
            "timestamp": "2023-04-24 20:33:36.719",
        },
    ),
]


class CrowdstrikeBase64EncodedArgs(PantherRule):
    RuleID = "Crowdstrike.Base64EncodedArgs-prototype"
    DisplayName = "Execution of Command Line Tool with Base64 Encoded Arguments"
    LogTypes = [LogType.Crowdstrike_FDREvent]
    Tags = ["Execution", "Obfuscation"]
    Severity = PantherSeverity.Medium
    Description = "Detects the execution of common command line tools (e.g., PowerShell, cmd.exe) with Base64 encoded arguments, which could indicate an attempt to obfuscate malicious commands."
    Runbook = "Investigate the endpoint for signs of command line tool execution with Base64 encoded arguments. Review the executed command, decode the Base64 string, and analyze the original content."
    Reference = "https://www.crowdstrike.com/blog/blocking-fileless-script-based-attacks-using-falcon-script-control-feature/"
    Tests = crowdstrike_base64_encoded_args_tests
    DECODED = ""
    # List of command line tools to monitor for execution with Base64 encoded arguments
    COMMAND_LINE_TOOLS = {"powershell.exe", "cmd.exe", "cscript.exe", "wscript.exe", "rundll32.exe"}

    def rule(self, event):
        # Filter by CS event type, Windows platform, and process name
        if not all(
            [
                event.get("fdr_event_type") == "ProcessRollup2",
                event.get("event_platform") == "Win",
                event.udm("process_name").lower() in self.COMMAND_LINE_TOOLS,
            ]
        ):
            return False
        # Split arguments from process path
        command_line_args = event.udm("cmd")
        command_line_args = command_line_args.replace('"', "")
        command_line_args = command_line_args.replace("'", "")
        command_line_args = command_line_args.split(" ")[1:]
        # Check if Base64 encoded arguments are present in the command line
        for arg in command_line_args:
            self.DECODED = is_base64(arg)
            if self.DECODED:
                return True
        return False

    def title(self, event):
        process_name = event.udm("process_name").lower()
        command_line = event.udm("cmd")
        return (
            f"Crowdstrike: Execution with base64 encoded args: [{process_name}] - [{command_line}]"
        )

    def alert_context(self, event):
        context = crowdstrike_process_alert_context(event)
        context["decoded arg"] = self.DECODED
        return context
