import numpy as np
from typing import List, Tuple


class Exploit(object):
    def __init__(
        self,
        exploit_config: dict = {
            "strategy": "truncation",
            "selection_percent": 0.2,
        },
        maximize_objective: bool = False,
    ):
        """Exploitation Strategies for PBT (Jaderberg et al. 17).

        Args:
            exploit_config (dict, optional): Configuration for exploitation step.
                Defaults to {"strategy": "truncation",
                             "selection_percent": 0.2}.
            maximize_objective (bool, optional): Whether to maximize objective.
                Defaults to False.
        """
        self.exploit_config = exploit_config
        assert self.exploit_config["strategy"] in [
            "truncation",
            "binary_tournament",
        ]
        self.maximize_objective = maximize_objective

    def truncation_selection(
        self,
        worker_id: int,
        batch_proposals: list,
        perf_measures: list,
        ckpt_paths: List[str],
    ) -> Tuple[dict, dict, str]:
        """Rank networks in population by performance. If network is in bottom
           x% sample another uniformly from top x%, and copy it.

        Args:
            worker_id (int): Integer id of considered worker
            batch_proposals (list): List of all active workers' hyperparameters.
            perf_measures (list): List of performances of workers.
            ckpt_paths (List[str]): List of paths to corresponding ckpts.

        Returns:
            Tuple[dict, dict, str]:
                Summary info dictionary about exploitation step.
                New hyperparameters for worker after exploitation.
                Checkpoint path for worker to continue training.
        """
        top_ids, bottom_ids = get_truncation_ids(
            perf_measures,
            self.exploit_config["selection_percent"],
            self.maximize_objective,
        )
        # Check if worker is in top_df - if not sample from top
        if worker_id not in bottom_ids:
            # Keep data from worker to continue training
            copy_info, hyperparams, ckpt_path = get_exploit_data(
                0,
                worker_id,
                worker_id,
                batch_proposals,
                perf_measures,
                ckpt_paths,
            )
        else:
            # Sample worker to copy from top dataframe (20%)
            copy_from_id = np.random.choice(top_ids)
            copy_info, hyperparams, ckpt_path = get_exploit_data(
                1,
                worker_id,
                copy_from_id,
                batch_proposals,
                perf_measures,
                ckpt_paths,
            )
        return copy_info, hyperparams, ckpt_path

    def binary_tournament(
        self,
        worker_id: int,
        batch_proposals: list,
        perf_measures: list,
        ckpt_paths: List[str],
    ) -> Tuple[dict, dict, str]:
        """Each member of population randomly selects another one & copies all
           its parameters (hyper & weights) if other member’s score is better.

        Args:
            worker_id (int): Integer id of considered worker
            batch_proposals (list): List of all active workers' hyperparameters.
            perf_measures (list): List of performances of workers.
            ckpt_paths (List[str]): List of paths to corresponding ckpts.

        Returns:
            Tuple[dict, dict, str]:
                Summary info dictionary about exploitation step.
                New hyperparameters for worker after exploitation.
                Checkpoint path for worker to continue training.
        """
        valid_ids = [
            id for id in range(len(batch_proposals)) if id != worker_id
        ]
        copy_from_id = np.random.choice(valid_ids)
        # Check if worker is worse than other randomly sampled member
        if self.maximize_objective:
            bottom_performer = (
                perf_measures[worker_id] < perf_measures[copy_from_id]
            )
        else:
            bottom_performer = (
                perf_measures[worker_id] > perf_measures[copy_from_id]
            )
        # Keep data from worker to continue training otw. copy better worker
        if not bottom_performer:
            copy_info, hyperparams, ckpt_path = get_exploit_data(
                False,
                worker_id,
                worker_id,
                batch_proposals,
                perf_measures,
                ckpt_paths,
            )
        else:
            id_range = range(len(batch_proposals))
            valid_ids = [i for i in id_range if i != worker_id]
            copy_from_id = np.random.choice(valid_ids)
            copy_info, hyperparams, ckpt_path = get_exploit_data(
                True,
                worker_id,
                copy_from_id,
                batch_proposals,
                perf_measures,
                ckpt_paths,
            )
        return copy_info, hyperparams, ckpt_path

    def __call__(
        self, batch_proposals: list, perf_measures: list, ckpt_paths: List[str]
    ) -> Tuple[List[dict], List[dict], List[str]]:
        """Run exploitation step for all workers in a loop.

        Args:
            batch_proposals (list): List of all active workers' hyperparameters.
            perf_measures (list): List of performances of workers.
            ckpt_paths (List[str]): List of paths to corresponding ckpts.

        Returns:
            Tuple[List[dict], List[dict], List[str]]:
                Summary info dictionaries about exploitation step.
                New hyperparameters for all workers after exploitation.
                Checkpoint paths for all workers to continue training.
        """
        copy_infos, hyperparams, ckpts = [], [], []
        for w_id in range(len(batch_proposals)):
            if self.exploit_config["strategy"] == "truncation":
                explore, hyper, ckpt = self.truncation_selection(
                    w_id, batch_proposals, perf_measures, ckpt_paths
                )
            elif self.exploit_config["strategy"] == "binary_tournament":
                explore, hyper, ckpt = self.binary_tournament(
                    w_id, batch_proposals, perf_measures, ckpt_paths
                )
            copy_infos.append(explore)
            hyperparams.append(hyper)
            ckpts.append(ckpt)
        return copy_infos, hyperparams, ckpts


def get_truncation_ids(
    perf_measures: list,
    selection_percent: float,
    maximize_objective: bool = False,
) -> Tuple[np.ndarray, np.ndarray]:
    """Get top and bottom performers - based on selection percentage.

    Args:
        perf_measures (list): [description]
        selection_percent (float): % top and bottom performing workers.
        maximize_objective (bool, optional): Whether to maximize objective.
            Defaults to False.

    Returns:
        Tuple[np.ndarray, np.ndarray]: [description]
    """
    assert 0 < selection_percent < 1
    bucket_sizes = np.maximum(1, round(len(perf_measures) * selection_percent))
    sorted_idx = np.argsort(perf_measures)
    if not maximize_objective:
        best_idx = sorted_idx[:bucket_sizes]
        worst_idx = sorted_idx[::-1][:bucket_sizes]
    else:
        best_idx = sorted_idx[::-1][:bucket_sizes]
        worst_idx = sorted_idx[:bucket_sizes]
    return best_idx, worst_idx


def get_exploit_data(
    explore: bool,
    worker_id: int,
    copy_from_id: int,
    batch_proposals: list,
    perf_measures: list,
    ckpt_paths: List[str],
) -> Tuple[dict, dict, str]:
    """Collect exploitation data for PBT worker at next step.

    Args:
        explore (bool): [description]
        worker_id (int): [description]
        copy_from_id (int): [description]
        batch_proposals (list): [description]
        perf_measures (list): [description]
        ckpt_paths (List[str]): [description]

    Returns:
        Tuple[dict, dict, str]:
            Summary info dictionary about exploitation step.
            New hyperparameters for worker after exploitation.
            Checkpoint path for worker to continue training.
    """
    copy_info = {
        "explore": explore,
        "copy_id": copy_from_id,
        "old_params": batch_proposals[worker_id],
        "copy_params": batch_proposals[copy_from_id],
        "old_performance": perf_measures[worker_id],
        "copy_performance": perf_measures[copy_from_id],
    }
    hyperparams = dict(batch_proposals[copy_from_id])
    ckpt_path = ckpt_paths[copy_from_id]
    return copy_info, hyperparams, ckpt_path
