/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
/// <reference types="node" />
import { DynamoDB, SecretsManager } from 'aws-sdk';
import { LambdaContext } from '../lib/aws-lambda';
import { CfnRequestEvent, DynamoBackedCustomResource } from '../lib/custom-resource';
import { CompositeStringIndexTable } from '../lib/dynamodb';
import { Key } from '../lib/kms';
import { IX509CertificateEncodePkcs12, IX509CertificateGenerate, IX509ResourceProperties } from './types';
declare abstract class X509Common extends DynamoBackedCustomResource {
    protected readonly secretsManagerClient: SecretsManager;
    constructor(dynamoDbClient: DynamoDB, secretsManagerClient: SecretsManager);
    doDelete(physicalId: string, resourceProperties: IX509ResourceProperties): Promise<void>;
    protected secretsPermissionsCheck(tags?: Array<{
        Key: string;
        Value: string;
    }>): Promise<void>;
    /**
     * Helper for creating a Secret and storing its ARN in the database.
     * CustomResources must be idempotent, and it is theoretically possible that Cfn
     * may invoke our lambda more than once for the same operation. If this happens,
     * then we may already have a Secret ARN stored in the database; if we do,
     * then we must reuse that ARN to avoid resource leakage.
     *
     * It's theoretically possible that this function may race with another invocation
     * of itself, but that is so unlikely. Handling it would complicate this function
     * substantially, so we do not implement for that case.
     * @param args
     */
    protected createAndStoreSecret(args: {
        readonly database: CompositeStringIndexTable;
        readonly name: string;
        readonly physicalId: string;
        readonly purpose: string;
        readonly data: string | Buffer;
        readonly description: string;
        readonly tags: Array<{
            Key: string;
            Value: string;
        }>;
        readonly encryptionKey?: Key;
    }): Promise<string>;
}
export declare class X509CertificateGenerator extends X509Common {
    constructor(dynamoDbClient: DynamoDB, secretsManagerClient: SecretsManager);
    validateInput(data: object): boolean;
    doCreate(physicalId: string, resourceProperties: IX509CertificateGenerate): Promise<object>;
}
export declare class X509CertificateConverter extends X509Common {
    constructor(dynamoDbClient: DynamoDB, secretsManagerClient: SecretsManager);
    validateInput(data: object): boolean;
    doCreate(physicalId: string, resourceProperties: IX509CertificateEncodePkcs12): Promise<object>;
}
/**
 * The handler used to generate an X.509 certificate and then store it in SecretsManager
 */
export declare function generate(event: CfnRequestEvent, context: LambdaContext): Promise<string>;
/**
 * The handler used to convert an X.509 certificate to PKCS #12 and store that in SecretsManager
 */
export declare function convert(event: CfnRequestEvent, context: LambdaContext): Promise<string>;
export {};
