"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.configureSEP = exports.SEPConfiguratorResource = void 0;
// eslint-disable-next-line import/no-extraneous-dependencies
const aws_sdk_1 = require("aws-sdk");
const configure_spot_event_plugin_1 = require("../lib/configure-spot-event-plugin");
const custom_resource_1 = require("../lib/custom-resource");
const deadline_client_1 = require("../lib/deadline-client");
const secrets_manager_1 = require("../lib/secrets-manager");
const conversion_1 = require("./conversion");
/**
 * A custom resource used to save Spot Event Plugin server data and configurations.
 */
class SEPConfiguratorResource extends custom_resource_1.SimpleCustomResource {
    constructor(secretsManagerClient) {
        super();
        this.secretsManagerClient = secretsManagerClient;
    }
    /**
     * @inheritdoc
     */
    validateInput(data) {
        return this.implementsSEPConfiguratorResourceProps(data);
    }
    /**
     * @inheritdoc
     */
    async doCreate(_physicalId, resourceProperties) {
        const spotEventPluginClient = await this.spotEventPluginClient(resourceProperties.connection);
        if (resourceProperties.spotFleetRequestConfigurations) {
            const convertedSpotFleetRequestConfigs = conversion_1.convertSpotFleetRequestConfiguration(resourceProperties.spotFleetRequestConfigurations);
            const stringConfigs = JSON.stringify(convertedSpotFleetRequestConfigs);
            const response = await spotEventPluginClient.saveServerData(stringConfigs);
            if (!response) {
                throw new Error('Failed to save spot fleet request with configuration');
            }
        }
        if (resourceProperties.spotPluginConfigurations) {
            const convertedSpotPluginConfigs = conversion_1.convertSpotEventPluginSettings(resourceProperties.spotPluginConfigurations);
            const pluginSettings = this.toKeyValueArray(convertedSpotPluginConfigs);
            const securitySettings = this.securitySettings();
            const response = await spotEventPluginClient.configureSpotEventPlugin([...pluginSettings, ...securitySettings]);
            if (!response) {
                throw new Error('Failed to save Spot Event Plugin Configurations');
            }
        }
        return undefined;
    }
    /**
     * @inheritdoc
     */
    async doDelete(_physicalId, _resourceProperties) {
        // Nothing to do -- we don't modify anything.
        return;
    }
    implementsSEPConfiguratorResourceProps(value) {
        if (!value || typeof (value) !== 'object' || Array.isArray(value)) {
            return false;
        }
        if (!this.implementsConnectionOptions(value.connection)) {
            return false;
        }
        return true;
    }
    implementsConnectionOptions(value) {
        if (!value || typeof (value) !== 'object' || Array.isArray(value)) {
            return false;
        }
        if (!value.hostname || typeof (value.hostname) !== 'string') {
            return false;
        }
        if (!value.port || typeof (value.port) !== 'string') {
            return false;
        }
        const portNum = Number.parseInt(value.port, 10);
        if (Number.isNaN(portNum) || portNum < 1 || portNum > 65535) {
            return false;
        }
        if (!value.protocol || typeof (value.protocol) !== 'string') {
            return false;
        }
        if (value.protocol !== 'HTTP' && value.protocol !== 'HTTPS') {
            return false;
        }
        if (!this.isSecretArnOrUndefined(value.caCertificateArn)) {
            return false;
        }
        return true;
    }
    isSecretArnOrUndefined(value) {
        if (value) {
            if (typeof (value) !== 'string' || !secrets_manager_1.isArn(value)) {
                return false;
            }
        }
        return true;
    }
    async spotEventPluginClient(connection) {
        return new configure_spot_event_plugin_1.SpotEventPluginClient(new deadline_client_1.DeadlineClient({
            host: connection.hostname,
            port: Number.parseInt(connection.port, 10),
            protocol: connection.protocol,
            tls: {
                ca: connection.caCertificateArn ? await secrets_manager_1.readCertificateData(connection.caCertificateArn, this.secretsManagerClient) : undefined,
            },
        }));
    }
    toKeyValueArray(input) {
        const configs = [];
        for (const [key, value] of Object.entries(input)) {
            if (value === undefined) {
                throw new Error(`Value for ${key} should be defined.`);
            }
            configs.push({
                Key: key,
                Value: value,
            });
        }
        return configs;
    }
    securitySettings() {
        return [
            {
                Key: 'UseLocalCredentials',
                Value: true,
            },
            {
                Key: 'NamedProfile',
                Value: '',
            },
        ];
    }
}
exports.SEPConfiguratorResource = SEPConfiguratorResource;
/**
 * The lambda handler that is used to log in to MongoDB and perform some configuration actions.
 */
/* istanbul ignore next */
async function configureSEP(event, context) {
    const handler = new SEPConfiguratorResource(new aws_sdk_1.SecretsManager());
    return await handler.handler(event, context);
}
exports.configureSEP = configureSEP;
//# sourceMappingURL=data:application/json;base64,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