"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let versionedInstallers;
let dependencyStack;
let dockerContainer;
let images;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let vpc;
let workerFleet;
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        versionedInstallers = new lib_1.VersionQuery(dependencyStack, 'VersionQuery');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: versionedInstallers,
            }),
            version: versionedInstallers,
        });
        stack = new core_1.Stack(app, 'Stack', { env });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(stack, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        licenses = [lib_1.UsageBasedLicense.forMaya()];
    });
    test('creates an ECS cluster', () => {
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'UBL', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
                certificateSecret,
                desiredCount: 2,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // WHEN
            const logGroup = ubl.node.findChild('UBLLogGroup');
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // WHEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // WHEN
                new lib_1.UsageBasedLicensing(stack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                // WHEN
                new lib_1.UsageBasedLicensing(stack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                    desiredCount,
                });
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // WHEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // WHEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: certificateSecret.secretArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('License Forwarder is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = 'licenseForwarder';
        // WHEN
        new lib_1.UsageBasedLicensing(stack, id, {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
                vpc,
                images,
                certificateSecret,
                renderQueue,
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4151, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5555, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
            workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(workerStack, 'SG', 'sg-123456789', {
                    allowAllOutbound: false,
                }),
            });
            // WHEN
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                assert_1.expect(workerStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(ubl).stackName}:ExportsOutputFnGetAttUBLClusterASGInstanceSecurityGroup*`),
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
                    vpc,
                    images,
                    certificateSecret: certificateSecret,
                    licenses: [],
                    renderQueue,
                });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule for asg', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 8080,
                ToPort: 8080,
                GroupId: {
                    'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLBSecurityGroup*`),
                },
                SourceSecurityGroupId: {
                    'Fn::GetAtt': [
                        'UBLClusterASGInstanceSecurityGroupAA1A7A2D',
                        'GroupId',
                    ],
                },
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // WHEN
            new lib_1.UsageBasedLicensing(stack, 'UBL', {
                certificateSecret,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'http://',
                                        {
                                            'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLB*`),
                                        },
                                        ':8080',
                                    ],
                                ],
                            },
                        }),
                    },
                ],
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                new lib_1.UsageBasedLicensing(stack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                });
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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