"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('ThinkboxDockerRecipes', () => {
    let app;
    let stack;
    let images;
    let userAwsThinkboxEulaAcceptance;
    describe('defaults', () => {
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
            userAwsThinkboxEulaAcceptance = lib_1.AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA;
            // WHEN
            images = new lib_1.ThinkboxDockerImages(stack, 'Images', {
                userAwsThinkboxEulaAcceptance,
            });
        });
        test('fails validation when EULA is not accepted', () => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            const expectedError = `
The ThinkboxDockerImages will install Deadline onto one or more EC2 instances.

Deadline is provided by AWS Thinkbox under the AWS Thinkbox End User License
Agreement (EULA). By installing Deadline, you are agreeing to the terms of this
license. Follow the link below to read the terms of the AWS Thinkbox EULA.

https://www.awsthinkbox.com/end-user-license-agreement

By using the ThinkboxDockerImages to install Deadline you agree to the terms of
the AWS Thinkbox EULA.

Please set the userAwsThinkboxEulaAcceptance property to
USER_ACCEPTS_AWS_THINKBOX_EULA to signify your acceptance of the terms of the
AWS Thinkbox EULA.
`;
            userAwsThinkboxEulaAcceptance = lib_1.AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA;
            new lib_1.ThinkboxDockerImages(newStack, 'Images', {
                userAwsThinkboxEulaAcceptance,
            });
            // WHEN
            function synth() {
                assert_1.SynthUtils.synthesize(newStack);
            }
            // THEN
            expect(synth).toThrow(expectedError);
        });
        test('creates Custom::RFDK_ECR_PROVIDER', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_EcrProvider', {
                ForceRun: assert_1.stringLike('*'),
            }));
        });
        describe('provides container images for', () => {
            test.each([
                [
                    'RCS',
                    () => {
                        return images.remoteConnectionServer;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER,
                ],
                [
                    'License Forwarder',
                    () => {
                        return images.licenseForwarder;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER,
                ],
            ])('%s', (_label, imageGetter, recipe) => {
                // GIVEN
                const taskDefStack = new core_1.Stack(app, 'TaskDefStack');
                const image = imageGetter();
                const taskDefinition = new aws_ecs_1.TaskDefinition(taskDefStack, 'TaskDef', {
                    compatibility: aws_ecs_1.Compatibility.EC2,
                });
                const ecrProvider = images.node.defaultChild;
                const expectedImage = `${ecrProvider.getAtt('EcrURIPrefix')}${recipe}`;
                // WHEN
                new aws_ecs_1.ContainerDefinition(taskDefStack, 'ContainerDef', {
                    image,
                    taskDefinition,
                    memoryReservationMiB: 1024,
                });
                // THEN
                assert_1.expect(taskDefStack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({
                        Image: taskDefStack.resolve(expectedImage),
                    })),
                }));
            });
        });
        describe('.forRenderQueue()', () => {
            let rcsImage;
            let rqImages;
            beforeEach(() => {
                // GIVEN
                rcsImage = images.remoteConnectionServer;
                // WHEN
                rqImages = images.forRenderQueue();
            });
            test('returns correct RCS image', () => {
                // THEN
                expect(rqImages.remoteConnectionServer).toBe(rcsImage);
            });
        });
        describe('.forUsageBasedLicensing()', () => {
            let lfImage;
            let ublImages;
            beforeEach(() => {
                // GIVEN
                lfImage = images.licenseForwarder;
                // WHEN
                ublImages = images.forUsageBasedLicensing();
            });
            test('returns correct RCS image', () => {
                // THEN
                expect(ublImages.licenseForwarder).toBe(lfImage);
            });
        });
    });
    describe('with version', () => {
        let version;
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
            version = new lib_1.VersionQuery(stack, 'Version');
            // WHEN
            images = new lib_1.ThinkboxDockerImages(stack, 'Images', {
                version,
                userAwsThinkboxEulaAcceptance,
            });
        });
        describe('provides container images for', () => {
            test.each([
                [
                    'RCS',
                    () => {
                        return images.remoteConnectionServer;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.REMOTE_CONNECTION_SERVER,
                ],
                [
                    'License Forwarder',
                    () => {
                        return images.licenseForwarder;
                    },
                    lib_1.ThinkboxManagedDeadlineDockerRecipes.LICENSE_FORWARDER,
                ],
            ])('%s', (_label, imageGetter, recipe) => {
                // GIVEN
                const taskDefStack = new core_1.Stack(app, 'TaskDefStack');
                const image = imageGetter();
                const taskDefinition = new aws_ecs_1.TaskDefinition(taskDefStack, 'TaskDef', {
                    compatibility: aws_ecs_1.Compatibility.EC2,
                });
                const ecrProvider = images.node.defaultChild;
                const expectedImage = `${ecrProvider.getAtt('EcrURIPrefix')}${recipe}:${core_1.Token.asString(version.majorVersion)}.${core_1.Token.asString(version.minorVersion)}.${core_1.Token.asString(version.releaseVersion)}`;
                // WHEN
                new aws_ecs_1.ContainerDefinition(taskDefStack, 'ContainerDef', {
                    image,
                    taskDefinition,
                    memoryReservationMiB: 1024,
                });
                // THEN
                assert_1.expect(taskDefStack).to(assert_1.haveResource('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.objectLike({
                        Image: taskDefStack.resolve(expectedImage),
                    })),
                }));
            });
        });
        test('validates VersionQuery is not in a different stack', () => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            new lib_1.ThinkboxDockerImages(newStack, 'Images', {
                version,
                userAwsThinkboxEulaAcceptance,
            });
            // WHEN
            function synth() {
                assert_1.SynthUtils.synthesize(newStack);
            }
            // THEN
            expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGhpbmtib3gtZG9ja2VyLWltYWdlcy50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGhpbmtib3gtZG9ja2VyLWltYWdlcy50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7O0FBRUgsNENBT3lCO0FBQ3pCLDhDQUswQjtBQUMxQix3Q0FLdUI7QUFFdkIsZ0NBUWdCO0FBRWhCLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRSxHQUFHLEVBQUU7SUFDckMsSUFBSSxHQUFRLENBQUM7SUFDYixJQUFJLEtBQVksQ0FBQztJQUNqQixJQUFJLE1BQTRCLENBQUM7SUFDakMsSUFBSSw2QkFBd0QsQ0FBQztJQUU3RCxRQUFRLENBQUMsVUFBVSxFQUFFLEdBQUcsRUFBRTtRQUN4QixVQUFVLENBQUMsR0FBRyxFQUFFO1lBQ2QsUUFBUTtZQUNSLEdBQUcsR0FBRyxJQUFJLFVBQUcsRUFBRSxDQUFDO1lBQ2hCLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDaEMsNkJBQTZCLEdBQUcsK0JBQXlCLENBQUMsOEJBQThCLENBQUM7WUFFekYsT0FBTztZQUNQLE1BQU0sR0FBRyxJQUFJLDBCQUFvQixDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ2pELDZCQUE2QjthQUM5QixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyw0Q0FBNEMsRUFBRSxHQUFHLEVBQUU7WUFDdEQsUUFBUTtZQUNSLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM1QyxNQUFNLGFBQWEsR0FBRzs7Ozs7Ozs7Ozs7Ozs7O0NBZTNCLENBQUM7WUFDSSw2QkFBNkIsR0FBRywrQkFBeUIsQ0FBQyw4QkFBOEIsQ0FBQztZQUN6RixJQUFJLDBCQUFvQixDQUFDLFFBQVEsRUFBRSxRQUFRLEVBQUU7Z0JBQzNDLDZCQUE2QjthQUM5QixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsU0FBUyxLQUFLO2dCQUNaLG1CQUFVLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ2xDLENBQUM7WUFFRCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN2QyxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7WUFDN0MsT0FBTztZQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQywwQkFBMEIsRUFBRTtnQkFDM0QsUUFBUSxFQUFFLG1CQUFVLENBQUMsR0FBRyxDQUFDO2FBQzFCLENBQUMsQ0FBQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsK0JBQStCLEVBQUUsR0FBRyxFQUFFO1lBQzdDLElBQUksQ0FBQyxJQUFJLENBQXVFO2dCQUM5RTtvQkFDRSxLQUFLO29CQUNMLEdBQUcsRUFBRTt3QkFDSCxPQUFPLE1BQU0sQ0FBQyxzQkFBc0IsQ0FBQztvQkFDdkMsQ0FBQztvQkFDRCwwQ0FBb0MsQ0FBQyx3QkFBd0I7aUJBQzlEO2dCQUNEO29CQUNFLG1CQUFtQjtvQkFDbkIsR0FBRyxFQUFFO3dCQUNILE9BQU8sTUFBTSxDQUFDLGdCQUFnQixDQUFDO29CQUNqQyxDQUFDO29CQUNELDBDQUFvQyxDQUFDLGlCQUFpQjtpQkFDdkQ7YUFDRixDQUFDLENBQUMsSUFBSSxFQUFFLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxNQUFNLEVBQUUsRUFBRTtnQkFDdkMsUUFBUTtnQkFDUixNQUFNLFlBQVksR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsY0FBYyxDQUFDLENBQUM7Z0JBQ3BELE1BQU0sS0FBSyxHQUFHLFdBQVcsRUFBRSxDQUFDO2dCQUM1QixNQUFNLGNBQWMsR0FBRyxJQUFJLHdCQUFjLENBQUMsWUFBWSxFQUFFLFNBQVMsRUFBRTtvQkFDakUsYUFBYSxFQUFFLHVCQUFhLENBQUMsR0FBRztpQkFDakMsQ0FBQyxDQUFDO2dCQUNILE1BQU0sV0FBVyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBOEIsQ0FBQztnQkFDL0QsTUFBTSxhQUFhLEdBQUcsR0FBRyxXQUFXLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxHQUFHLE1BQU0sRUFBRSxDQUFDO2dCQUV2RSxPQUFPO2dCQUNQLElBQUksNkJBQW1CLENBQUMsWUFBWSxFQUFFLGNBQWMsRUFBRTtvQkFDcEQsS0FBSztvQkFDTCxjQUFjO29CQUNkLG9CQUFvQixFQUFFLElBQUk7aUJBQzNCLENBQUMsQ0FBQztnQkFFSCxPQUFPO2dCQUNQLGVBQVMsQ0FBQyxZQUFZLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQywwQkFBMEIsRUFBRTtvQkFDbEUsb0JBQW9CLEVBQUUsa0JBQVMsQ0FBQyxtQkFBVSxDQUFDO3dCQUN6QyxLQUFLLEVBQUUsWUFBWSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUM7cUJBQzNDLENBQUMsQ0FBQztpQkFDSixDQUFDLENBQUMsQ0FBQztZQUNOLENBQUMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsbUJBQW1CLEVBQUUsR0FBRyxFQUFFO1lBQ2pDLElBQUksUUFBd0IsQ0FBQztZQUM3QixJQUFJLFFBQTJCLENBQUM7WUFFaEMsVUFBVSxDQUFDLEdBQUcsRUFBRTtnQkFDZCxRQUFRO2dCQUNSLFFBQVEsR0FBRyxNQUFNLENBQUMsc0JBQXNCLENBQUM7Z0JBRXpDLE9BQU87Z0JBQ1AsUUFBUSxHQUFHLE1BQU0sQ0FBQyxjQUFjLEVBQUUsQ0FBQztZQUNyQyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7Z0JBQ3JDLE9BQU87Z0JBQ1AsTUFBTSxDQUFDLFFBQVEsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUN6RCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLDJCQUEyQixFQUFFLEdBQUcsRUFBRTtZQUN6QyxJQUFJLE9BQXVCLENBQUM7WUFDNUIsSUFBSSxTQUFvQyxDQUFDO1lBRXpDLFVBQVUsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2QsUUFBUTtnQkFDUixPQUFPLEdBQUcsTUFBTSxDQUFDLGdCQUFnQixDQUFDO2dCQUVsQyxPQUFPO2dCQUNQLFNBQVMsR0FBRyxNQUFNLENBQUMsc0JBQXNCLEVBQUUsQ0FBQztZQUM5QyxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7Z0JBQ3JDLE9BQU87Z0JBQ1AsTUFBTSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztZQUNuRCxDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsY0FBYyxFQUFFLEdBQUcsRUFBRTtRQUM1QixJQUFJLE9BQWlCLENBQUM7UUFFdEIsVUFBVSxDQUFDLEdBQUcsRUFBRTtZQUNkLFFBQVE7WUFDUixHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztZQUNoQixLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1lBQ2hDLE9BQU8sR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRTdDLE9BQU87WUFDUCxNQUFNLEdBQUcsSUFBSSwwQkFBb0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO2dCQUNqRCxPQUFPO2dCQUNQLDZCQUE2QjthQUM5QixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILFFBQVEsQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7WUFDN0MsSUFBSSxDQUFDLElBQUksQ0FBdUU7Z0JBQzlFO29CQUNFLEtBQUs7b0JBQ0wsR0FBRyxFQUFFO3dCQUNILE9BQU8sTUFBTSxDQUFDLHNCQUFzQixDQUFDO29CQUN2QyxDQUFDO29CQUNELDBDQUFvQyxDQUFDLHdCQUF3QjtpQkFDOUQ7Z0JBQ0Q7b0JBQ0UsbUJBQW1CO29CQUNuQixHQUFHLEVBQUU7d0JBQ0gsT0FBTyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7b0JBQ2pDLENBQUM7b0JBQ0QsMENBQW9DLENBQUMsaUJBQWlCO2lCQUN2RDthQUNGLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLE1BQU0sRUFBRSxFQUFFO2dCQUN2QyxRQUFRO2dCQUNSLE1BQU0sWUFBWSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxjQUFjLENBQUMsQ0FBQztnQkFDcEQsTUFBTSxLQUFLLEdBQUcsV0FBVyxFQUFFLENBQUM7Z0JBQzVCLE1BQU0sY0FBYyxHQUFHLElBQUksd0JBQWMsQ0FBQyxZQUFZLEVBQUUsU0FBUyxFQUFFO29CQUNqRSxhQUFhLEVBQUUsdUJBQWEsQ0FBQyxHQUFHO2lCQUNqQyxDQUFDLENBQUM7Z0JBQ0gsTUFBTSxXQUFXLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUE4QixDQUFDO2dCQUMvRCxNQUFNLGFBQWEsR0FBRyxHQUFHLFdBQVcsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLEdBQUcsTUFBTSxJQUFJLFlBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxJQUFJLFlBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxJQUFJLFlBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxFQUFFLENBQUM7Z0JBRWpNLE9BQU87Z0JBQ1AsSUFBSSw2QkFBbUIsQ0FBQyxZQUFZLEVBQUUsY0FBYyxFQUFFO29CQUNwRCxLQUFLO29CQUNMLGNBQWM7b0JBQ2Qsb0JBQW9CLEVBQUUsSUFBSTtpQkFDM0IsQ0FBQyxDQUFDO2dCQUVILE9BQU87Z0JBQ1AsZUFBUyxDQUFDLFlBQVksQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDBCQUEwQixFQUFFO29CQUNsRSxvQkFBb0IsRUFBRSxrQkFBUyxDQUFDLG1CQUFVLENBQUM7d0JBQ3pDLEtBQUssRUFBRSxZQUFZLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQztxQkFDM0MsQ0FBQyxDQUFDO2lCQUNKLENBQUMsQ0FBQyxDQUFDO1lBQ04sQ0FBQyxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvREFBb0QsRUFBRSxHQUFHLEVBQUU7WUFDOUQsUUFBUTtZQUNSLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM1QyxJQUFJLDBCQUFvQixDQUFDLFFBQVEsRUFBRSxRQUFRLEVBQUU7Z0JBQzNDLE9BQU87Z0JBQ1AsNkJBQTZCO2FBQzlCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxTQUFTLEtBQUs7Z0JBQ1osbUJBQVUsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDbEMsQ0FBQztZQUVELE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLDJEQUEyRCxDQUFDLENBQUM7UUFDckYsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtcbiAgYXJyYXlXaXRoLFxuICBleHBlY3QgYXMgZXhwZWN0Q0RLLFxuICBoYXZlUmVzb3VyY2UsXG4gIG9iamVjdExpa2UsXG4gIHN0cmluZ0xpa2UsXG4gIFN5bnRoVXRpbHMsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge1xuICBDb21wYXRpYmlsaXR5LFxuICBDb250YWluZXJEZWZpbml0aW9uLFxuICBDb250YWluZXJJbWFnZSxcbiAgVGFza0RlZmluaXRpb24sXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3MnO1xuaW1wb3J0IHtcbiAgQXBwLFxuICBDdXN0b21SZXNvdXJjZSxcbiAgU3RhY2ssXG4gIFRva2VuLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZSxcbiAgSVZlcnNpb24sXG4gIFJlbmRlclF1ZXVlSW1hZ2VzLFxuICBUaGlua2JveERvY2tlckltYWdlcyxcbiAgVGhpbmtib3hNYW5hZ2VkRGVhZGxpbmVEb2NrZXJSZWNpcGVzLFxuICBVc2FnZUJhc2VkTGljZW5zaW5nSW1hZ2VzLFxuICBWZXJzaW9uUXVlcnksXG59IGZyb20gJy4uL2xpYic7XG5cbmRlc2NyaWJlKCdUaGlua2JveERvY2tlclJlY2lwZXMnLCAoKSA9PiB7XG4gIGxldCBhcHA6IEFwcDtcbiAgbGV0IHN0YWNrOiBTdGFjaztcbiAgbGV0IGltYWdlczogVGhpbmtib3hEb2NrZXJJbWFnZXM7XG4gIGxldCB1c2VyQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZTogQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZTtcblxuICBkZXNjcmliZSgnZGVmYXVsdHMnLCAoKSA9PiB7XG4gICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgYXBwID0gbmV3IEFwcCgpO1xuICAgICAgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snKTtcbiAgICAgIHVzZXJBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlID0gQXdzVGhpbmtib3hFdWxhQWNjZXB0YW5jZS5VU0VSX0FDQ0VQVFNfQVdTX1RISU5LQk9YX0VVTEE7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGltYWdlcyA9IG5ldyBUaGlua2JveERvY2tlckltYWdlcyhzdGFjaywgJ0ltYWdlcycsIHtcbiAgICAgICAgdXNlckF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2UsXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2ZhaWxzIHZhbGlkYXRpb24gd2hlbiBFVUxBIGlzIG5vdCBhY2NlcHRlZCcsICgpID0+e1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG5ld1N0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ05ld1N0YWNrJyk7XG4gICAgICBjb25zdCBleHBlY3RlZEVycm9yID0gYFxuVGhlIFRoaW5rYm94RG9ja2VySW1hZ2VzIHdpbGwgaW5zdGFsbCBEZWFkbGluZSBvbnRvIG9uZSBvciBtb3JlIEVDMiBpbnN0YW5jZXMuXG5cbkRlYWRsaW5lIGlzIHByb3ZpZGVkIGJ5IEFXUyBUaGlua2JveCB1bmRlciB0aGUgQVdTIFRoaW5rYm94IEVuZCBVc2VyIExpY2Vuc2VcbkFncmVlbWVudCAoRVVMQSkuIEJ5IGluc3RhbGxpbmcgRGVhZGxpbmUsIHlvdSBhcmUgYWdyZWVpbmcgdG8gdGhlIHRlcm1zIG9mIHRoaXNcbmxpY2Vuc2UuIEZvbGxvdyB0aGUgbGluayBiZWxvdyB0byByZWFkIHRoZSB0ZXJtcyBvZiB0aGUgQVdTIFRoaW5rYm94IEVVTEEuXG5cbmh0dHBzOi8vd3d3LmF3c3RoaW5rYm94LmNvbS9lbmQtdXNlci1saWNlbnNlLWFncmVlbWVudFxuXG5CeSB1c2luZyB0aGUgVGhpbmtib3hEb2NrZXJJbWFnZXMgdG8gaW5zdGFsbCBEZWFkbGluZSB5b3UgYWdyZWUgdG8gdGhlIHRlcm1zIG9mXG50aGUgQVdTIFRoaW5rYm94IEVVTEEuXG5cblBsZWFzZSBzZXQgdGhlIHVzZXJBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlIHByb3BlcnR5IHRvXG5VU0VSX0FDQ0VQVFNfQVdTX1RISU5LQk9YX0VVTEEgdG8gc2lnbmlmeSB5b3VyIGFjY2VwdGFuY2Ugb2YgdGhlIHRlcm1zIG9mIHRoZVxuQVdTIFRoaW5rYm94IEVVTEEuXG5gO1xuICAgICAgdXNlckF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2UgPSBBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlLlVTRVJfUkVKRUNUU19BV1NfVEhJTktCT1hfRVVMQTtcbiAgICAgIG5ldyBUaGlua2JveERvY2tlckltYWdlcyhuZXdTdGFjaywgJ0ltYWdlcycsIHtcbiAgICAgICAgdXNlckF3c1RoaW5rYm94RXVsYUFjY2VwdGFuY2UsXG4gICAgICB9KTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgZnVuY3Rpb24gc3ludGgoKSB7XG4gICAgICAgIFN5bnRoVXRpbHMuc3ludGhlc2l6ZShuZXdTdGFjayk7XG4gICAgICB9XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzeW50aCkudG9UaHJvdyhleHBlY3RlZEVycm9yKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2NyZWF0ZXMgQ3VzdG9tOjpSRkRLX0VDUl9QUk9WSURFUicsICgpID0+IHtcbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlKCdDdXN0b206OlJGREtfRWNyUHJvdmlkZXInLCB7XG4gICAgICAgIEZvcmNlUnVuOiBzdHJpbmdMaWtlKCcqJyksXG4gICAgICB9KSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgncHJvdmlkZXMgY29udGFpbmVyIGltYWdlcyBmb3InLCAoKSA9PiB7XG4gICAgICB0ZXN0LmVhY2g8W3N0cmluZywgKCkgPT4gQ29udGFpbmVySW1hZ2UsIFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlc10+KFtcbiAgICAgICAgW1xuICAgICAgICAgICdSQ1MnLFxuICAgICAgICAgICgpID0+IHtcbiAgICAgICAgICAgIHJldHVybiBpbWFnZXMucmVtb3RlQ29ubmVjdGlvblNlcnZlcjtcbiAgICAgICAgICB9LFxuICAgICAgICAgIFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlcy5SRU1PVEVfQ09OTkVDVElPTl9TRVJWRVIsXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICAnTGljZW5zZSBGb3J3YXJkZXInLFxuICAgICAgICAgICgpID0+IHtcbiAgICAgICAgICAgIHJldHVybiBpbWFnZXMubGljZW5zZUZvcndhcmRlcjtcbiAgICAgICAgICB9LFxuICAgICAgICAgIFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlcy5MSUNFTlNFX0ZPUldBUkRFUixcbiAgICAgICAgXSxcbiAgICAgIF0pKCclcycsIChfbGFiZWwsIGltYWdlR2V0dGVyLCByZWNpcGUpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgdGFza0RlZlN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1Rhc2tEZWZTdGFjaycpO1xuICAgICAgICBjb25zdCBpbWFnZSA9IGltYWdlR2V0dGVyKCk7XG4gICAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IFRhc2tEZWZpbml0aW9uKHRhc2tEZWZTdGFjaywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgICAgY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eS5FQzIsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBlY3JQcm92aWRlciA9IGltYWdlcy5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDdXN0b21SZXNvdXJjZTtcbiAgICAgICAgY29uc3QgZXhwZWN0ZWRJbWFnZSA9IGAke2VjclByb3ZpZGVyLmdldEF0dCgnRWNyVVJJUHJlZml4Jyl9JHtyZWNpcGV9YDtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIG5ldyBDb250YWluZXJEZWZpbml0aW9uKHRhc2tEZWZTdGFjaywgJ0NvbnRhaW5lckRlZicsIHtcbiAgICAgICAgICBpbWFnZSxcbiAgICAgICAgICB0YXNrRGVmaW5pdGlvbixcbiAgICAgICAgICBtZW1vcnlSZXNlcnZhdGlvbk1pQjogMTAyNCxcbiAgICAgICAgfSk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICBleHBlY3RDREsodGFza0RlZlN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6RUNTOjpUYXNrRGVmaW5pdGlvbicsIHtcbiAgICAgICAgICBDb250YWluZXJEZWZpbml0aW9uczogYXJyYXlXaXRoKG9iamVjdExpa2Uoe1xuICAgICAgICAgICAgSW1hZ2U6IHRhc2tEZWZTdGFjay5yZXNvbHZlKGV4cGVjdGVkSW1hZ2UpLFxuICAgICAgICAgIH0pKSxcbiAgICAgICAgfSkpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnLmZvclJlbmRlclF1ZXVlKCknLCAoKSA9PiB7XG4gICAgICBsZXQgcmNzSW1hZ2U6IENvbnRhaW5lckltYWdlO1xuICAgICAgbGV0IHJxSW1hZ2VzOiBSZW5kZXJRdWV1ZUltYWdlcztcblxuICAgICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIHJjc0ltYWdlID0gaW1hZ2VzLnJlbW90ZUNvbm5lY3Rpb25TZXJ2ZXI7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICBycUltYWdlcyA9IGltYWdlcy5mb3JSZW5kZXJRdWV1ZSgpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ3JldHVybnMgY29ycmVjdCBSQ1MgaW1hZ2UnLCAoKSA9PiB7XG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHJxSW1hZ2VzLnJlbW90ZUNvbm5lY3Rpb25TZXJ2ZXIpLnRvQmUocmNzSW1hZ2UpO1xuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgnLmZvclVzYWdlQmFzZWRMaWNlbnNpbmcoKScsICgpID0+IHtcbiAgICAgIGxldCBsZkltYWdlOiBDb250YWluZXJJbWFnZTtcbiAgICAgIGxldCB1YmxJbWFnZXM6IFVzYWdlQmFzZWRMaWNlbnNpbmdJbWFnZXM7XG5cbiAgICAgIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgICAgICAvLyBHSVZFTlxuICAgICAgICBsZkltYWdlID0gaW1hZ2VzLmxpY2Vuc2VGb3J3YXJkZXI7XG5cbiAgICAgICAgLy8gV0hFTlxuICAgICAgICB1YmxJbWFnZXMgPSBpbWFnZXMuZm9yVXNhZ2VCYXNlZExpY2Vuc2luZygpO1xuICAgICAgfSk7XG5cbiAgICAgIHRlc3QoJ3JldHVybnMgY29ycmVjdCBSQ1MgaW1hZ2UnLCAoKSA9PiB7XG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgZXhwZWN0KHVibEltYWdlcy5saWNlbnNlRm9yd2FyZGVyKS50b0JlKGxmSW1hZ2UpO1xuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCd3aXRoIHZlcnNpb24nLCAoKSA9PiB7XG4gICAgbGV0IHZlcnNpb246IElWZXJzaW9uO1xuXG4gICAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgICAvLyBHSVZFTlxuICAgICAgYXBwID0gbmV3IEFwcCgpO1xuICAgICAgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2snKTtcbiAgICAgIHZlcnNpb24gPSBuZXcgVmVyc2lvblF1ZXJ5KHN0YWNrLCAnVmVyc2lvbicpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBpbWFnZXMgPSBuZXcgVGhpbmtib3hEb2NrZXJJbWFnZXMoc3RhY2ssICdJbWFnZXMnLCB7XG4gICAgICAgIHZlcnNpb24sXG4gICAgICAgIHVzZXJBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlLFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICBkZXNjcmliZSgncHJvdmlkZXMgY29udGFpbmVyIGltYWdlcyBmb3InLCAoKSA9PiB7XG4gICAgICB0ZXN0LmVhY2g8W3N0cmluZywgKCkgPT4gQ29udGFpbmVySW1hZ2UsIFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlc10+KFtcbiAgICAgICAgW1xuICAgICAgICAgICdSQ1MnLFxuICAgICAgICAgICgpID0+IHtcbiAgICAgICAgICAgIHJldHVybiBpbWFnZXMucmVtb3RlQ29ubmVjdGlvblNlcnZlcjtcbiAgICAgICAgICB9LFxuICAgICAgICAgIFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlcy5SRU1PVEVfQ09OTkVDVElPTl9TRVJWRVIsXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICAnTGljZW5zZSBGb3J3YXJkZXInLFxuICAgICAgICAgICgpID0+IHtcbiAgICAgICAgICAgIHJldHVybiBpbWFnZXMubGljZW5zZUZvcndhcmRlcjtcbiAgICAgICAgICB9LFxuICAgICAgICAgIFRoaW5rYm94TWFuYWdlZERlYWRsaW5lRG9ja2VyUmVjaXBlcy5MSUNFTlNFX0ZPUldBUkRFUixcbiAgICAgICAgXSxcbiAgICAgIF0pKCclcycsIChfbGFiZWwsIGltYWdlR2V0dGVyLCByZWNpcGUpID0+IHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgdGFza0RlZlN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1Rhc2tEZWZTdGFjaycpO1xuICAgICAgICBjb25zdCBpbWFnZSA9IGltYWdlR2V0dGVyKCk7XG4gICAgICAgIGNvbnN0IHRhc2tEZWZpbml0aW9uID0gbmV3IFRhc2tEZWZpbml0aW9uKHRhc2tEZWZTdGFjaywgJ1Rhc2tEZWYnLCB7XG4gICAgICAgICAgY29tcGF0aWJpbGl0eTogQ29tcGF0aWJpbGl0eS5FQzIsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBlY3JQcm92aWRlciA9IGltYWdlcy5ub2RlLmRlZmF1bHRDaGlsZCBhcyBDdXN0b21SZXNvdXJjZTtcbiAgICAgICAgY29uc3QgZXhwZWN0ZWRJbWFnZSA9IGAke2VjclByb3ZpZGVyLmdldEF0dCgnRWNyVVJJUHJlZml4Jyl9JHtyZWNpcGV9OiR7VG9rZW4uYXNTdHJpbmcodmVyc2lvbi5tYWpvclZlcnNpb24pfS4ke1Rva2VuLmFzU3RyaW5nKHZlcnNpb24ubWlub3JWZXJzaW9uKX0uJHtUb2tlbi5hc1N0cmluZyh2ZXJzaW9uLnJlbGVhc2VWZXJzaW9uKX1gO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgbmV3IENvbnRhaW5lckRlZmluaXRpb24odGFza0RlZlN0YWNrLCAnQ29udGFpbmVyRGVmJywge1xuICAgICAgICAgIGltYWdlLFxuICAgICAgICAgIHRhc2tEZWZpbml0aW9uLFxuICAgICAgICAgIG1lbW9yeVJlc2VydmF0aW9uTWlCOiAxMDI0LFxuICAgICAgICB9KTtcblxuICAgICAgICAvLyBUSEVOXG4gICAgICAgIGV4cGVjdENESyh0YXNrRGVmU3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uJywge1xuICAgICAgICAgIENvbnRhaW5lckRlZmluaXRpb25zOiBhcnJheVdpdGgob2JqZWN0TGlrZSh7XG4gICAgICAgICAgICBJbWFnZTogdGFza0RlZlN0YWNrLnJlc29sdmUoZXhwZWN0ZWRJbWFnZSksXG4gICAgICAgICAgfSkpLFxuICAgICAgICB9KSk7XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3ZhbGlkYXRlcyBWZXJzaW9uUXVlcnkgaXMgbm90IGluIGEgZGlmZmVyZW50IHN0YWNrJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IG5ld1N0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ05ld1N0YWNrJyk7XG4gICAgICBuZXcgVGhpbmtib3hEb2NrZXJJbWFnZXMobmV3U3RhY2ssICdJbWFnZXMnLCB7XG4gICAgICAgIHZlcnNpb24sXG4gICAgICAgIHVzZXJBd3NUaGlua2JveEV1bGFBY2NlcHRhbmNlLFxuICAgICAgfSk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGZ1bmN0aW9uIHN5bnRoKCkge1xuICAgICAgICBTeW50aFV0aWxzLnN5bnRoZXNpemUobmV3U3RhY2spO1xuICAgICAgfVxuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3Qoc3ludGgpLnRvVGhyb3coJ0EgVmVyc2lvblF1ZXJ5IGNhbiBub3QgYmUgc3VwcGxpZWQgZnJvbSBhIGRpZmZlcmVudCBzdGFjaycpO1xuICAgIH0pO1xuICB9KTtcbn0pO1xuIl19