"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_1 = require("../lib");
let app;
let stack;
let spotFleetStack;
let vpc;
let renderQueue;
let rcsImage;
const groupName = 'group_name';
const deadlineGroups = [
    groupName,
];
const workerMachineImage = new aws_ec2_1.GenericLinuxImage({
    'us-east-1': 'ami-any',
});
const instanceTypes = [
    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
];
const maxCapacity = 1;
describe('SpotEventPluginFleet', () => {
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'infraStack', {
            env: {
                region: 'us-east-1',
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
            }),
            version,
        });
        spotFleetStack = new core_1.Stack(app, 'SpotFleetStack', {
            env: {
                region: 'us-east-1',
            },
        });
    });
    describe('created with default values', () => {
        test('creates a security group', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.securityGroups).toBeDefined();
            expect(fleet.securityGroups.length).toBe(1);
            assert_1.expect(spotFleetStack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
        });
        test('allows connection to the render queue', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
                SourceSecurityGroupId: spotFleetStack.resolve(fleet.connections.securityGroups[0].securityGroupId),
            }));
        });
        test('creates a spot fleet instance role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('creates an instance profile', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.instanceProfile).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::InstanceProfile', {
                Roles: assert_1.arrayWith({
                    Ref: spotFleetStack.getLogicalId(fleet.fleetInstanceRole.node.defaultChild),
                }),
            }));
        });
        test('creates a spot fleet role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('adds group names to user data', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBeDefined();
            expect(renderedUserData).toMatch(groupName);
        });
        test('adds RFDK tags', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            expect(fleet.tags).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: rfdkTag.name,
                    Value: rfdkTag.value,
                })),
            }));
        });
        test('uses default LogGroup prefix %s', () => {
            // GIVEN
            const id = 'SpotFleet';
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: '/renderfarm/' + id,
            }));
        });
        test('sets default allocation strategy', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(lib_1.SpotFleetAllocationStrategy.LOWEST_PRICE);
        });
        test('does not set valid until property', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.validUntil).toBeUndefined();
        });
        test('does not set valid block devices', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.blockDevices).toBeUndefined();
        });
        test('does not set ssh key', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.keyName).toBeUndefined();
        });
    });
    describe('created with custom values', () => {
        test('uses provided required properties', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const imageConfig = workerMachineImage.getImage(fleet);
            // THEN
            expect(fleet.deadlineGroups).toBe(deadlineGroups);
            expect(fleet.instanceTypes).toBe(instanceTypes);
            expect(fleet.imageId).toBe(imageConfig.imageId);
            expect(fleet.osType).toBe(imageConfig.osType);
            expect(fleet.maxCapacity).toBe(maxCapacity);
        });
        test('uses provided security group', () => {
            // GIVEN
            const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(1);
            expect(fleet.securityGroups).toContainEqual(sg);
        });
        test('uses multiple provided security groups', () => {
            // GIVEN
            const sg1 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-123456789', {
                allowAllOutbound: false,
            });
            const sg2 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-987654321', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg1,
                    sg2,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(2);
            expect(fleet.securityGroups).toContainEqual(sg1);
            expect(fleet.securityGroups).toContainEqual(sg2);
        });
        test('adds to provided user data', () => {
            // GIVEN
            const originalCommands = 'original commands';
            const originalUserData = workerMachineImage.getImage(spotFleetStack).userData;
            originalUserData.addCommands(originalCommands);
            const renderedOriginalUser = originalUserData.render();
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userData: originalUserData,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBe(originalUserData);
            expect(renderedUserData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(originalCommands)));
            expect(renderedUserData).not.toEqual(renderedOriginalUser);
        });
        test('uses provided spot fleet instance role from the same stack', () => {
            // GIVEN
            const spotFleetInstanceRole = new aws_iam_1.Role(spotFleetStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                fleetInstanceRole: spotFleetInstanceRole,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBe(spotFleetInstanceRole);
            assert_1.expect(spotFleetStack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('throws if provided spot fleet instance role is not from the same stack', () => {
            // GIVEN
            const otherStack = new core_1.Stack(app, 'OtherStack', {
                env: { region: 'us-east-1' },
            });
            const spotFleetInstanceRole = new aws_iam_1.Role(otherStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            function createSpotEventPluginFleet() {
                new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    fleetInstanceRole: spotFleetInstanceRole,
                });
            }
            // THEN
            expect(createSpotEventPluginFleet).toThrowError('Fleet instance role should be created on the same stack as SpotEventPluginFleet to avoid circular dependencies.');
        });
        test('uses provided spot fleet role', () => {
            // GIVEN
            const fleetRole = new aws_iam_1.Role(stack, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                fleetRole: fleetRole,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBe(fleetRole);
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('tags resources deployed by CDK', () => {
            // GIVEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const tagName = 'name';
            const tagValue = 'tagValue';
            // WHEN
            core_1.Tags.of(fleet).add(tagName, tagValue);
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: tagName,
                    Value: tagValue,
                })),
            }));
        });
        test('uses provided subnets', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            const expectedSubnetId = stack.resolve(vpc.privateSubnets[0].subnetId);
            // THEN
            expect(stack.resolve(fleet.subnets.subnetIds)).toContainEqual(expectedSubnetId);
        });
        test('uses provided allocation strategy', () => {
            // GIVEN
            const allocationStartegy = lib_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                allocationStrategy: allocationStartegy,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(allocationStartegy);
        });
        test('adds deadline region to user data', () => {
            // GIVEN
            const deadlineRegion = 'someregion';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlineRegion: deadlineRegion,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(deadlineRegion);
        });
        test('adds deadline pools to user data', () => {
            // GIVEN
            const pool1 = 'pool1';
            const pool2 = 'pool2';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlinePools: [
                    pool1,
                    pool2,
                ],
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(pool1);
            expect(renderedUserData).toMatch(pool2);
        });
        test('uses provided ssh key name', () => {
            // GIVEN
            const keyName = 'test-key-name';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                keyName: keyName,
            });
            // THEN
            expect(fleet.keyName).toEqual(keyName);
        });
        test('UserData is added by UserDataProvider', () => {
            // WHEN
            class UserDataProvider extends lib_1.InstanceUserDataProvider {
                preCloudWatchAgent(host) {
                    host.userData.addCommands('echo preCloudWatchAgent');
                }
                preRenderQueueConfiguration(host) {
                    host.userData.addCommands('echo preRenderQueueConfiguration');
                }
                preWorkerConfiguration(host) {
                    host.userData.addCommands('echo preWorkerConfiguration');
                }
                postWorkerLaunch(host) {
                    host.userData.addCommands('echo postWorkerLaunch');
                }
            }
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userDataProvider: new UserDataProvider(spotFleetStack, 'UserDataProvider'),
            });
            const userData = fleet.userData.render();
            // THEN
            expect(userData).toContain('echo preCloudWatchAgent');
            expect(userData).toContain('echo preRenderQueueConfiguration');
            expect(userData).toContain('echo preWorkerConfiguration');
            expect(userData).toContain('echo postWorkerLaunch');
        });
        test.each([
            'test-prefix/',
            '',
        ])('uses custom LogGroup prefix %s', (testPrefix) => {
            // GIVEN
            const id = 'SpotFleet';
            const logGroupProps = {
                logGroupPrefix: testPrefix,
            };
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                logGroupProps: logGroupProps,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: testPrefix + id,
            }));
        });
    });
    describe('allowing remote control', () => {
        test('from CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('to CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('from SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('from other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
    });
    describe('validation with', () => {
        describe('instance types', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyInstanceTypes = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: emptyInstanceTypes,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one instance type is required for a Spot Fleet Request Configuration/);
            });
            test('passes with at least one', () => {
                // GIVEN
                const oneInstanceType = [aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL)];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: oneInstanceType,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('subnets', () => {
            test('error if no subnets provided', () => {
                // GIVEN
                const invalidSubnets = {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy zone'],
                };
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    vpcSubnets: invalidSubnets,
                });
                // THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.ERROR);
                expect(fleet.node.metadata[0].data).toMatch(/Did not find any subnets matching/);
            });
        });
        describe('groups', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyDeadlineGroups = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: emptyDeadlineGroups,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one Deadline Group is required for a Spot Fleet Request Configuration/);
            });
            test.each([
                'none',
                'with space',
                'group_*',
            ])('throws with %s', (invalidGroupName) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [invalidGroupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineGroups'/);
            });
            test('passes with valid group name', () => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [groupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('region', () => {
            test.each([
                'none',
                'all',
                'unrecognized',
                'none@123',
                'None',
            ])('throws with %s', (deadlineRegion) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineRegion'/);
            });
            test('passes with reserved name as substring', () => {
                // GIVEN
                const deadlineRegion = 'none123';
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('Block Device Tests', () => {
            test('Warning if no BlockDevices provided', () => {
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                });
                // THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
            });
            test('No Warnings if Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: '/dev/xvda',
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                        }],
                });
                //THEN
                expect(fleet.node.metadata).toHaveLength(0);
            });
            test('Warnings if non-Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                        }],
                });
                //THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('Warnings for BlockDevices without encryption specified', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                        }],
                });
                //THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('No warnings for Ephemeral blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                        }],
                });
                //THEN
                expect(fleet.node.metadata).toHaveLength(0);
            });
            test('No warnings for Suppressed blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                        }],
                });
                //THEN
                expect(fleet.node.metadata).toHaveLength(0);
            });
            test('throws if block devices without iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.IO1;
                // WHEN
                function createSpotEventPluginFleet() {
                    return new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups: [
                            groupName,
                        ],
                        instanceTypes: [
                            aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                        ],
                        workerMachineImage,
                        maxCapacity: 1,
                        blockDevices: [{
                                deviceName,
                                volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                    volumeType,
                                }),
                            }],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/iops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            });
            test('warning if block devices with iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const iops = 100;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups: [
                        groupName,
                    ],
                    instanceTypes: [
                        aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                    ],
                    workerMachineImage,
                    maxCapacity: 1,
                    blockDevices: [{
                            deviceName,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                iops,
                                volumeType,
                            }),
                        }],
                });
                // THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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