"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceConfiguration = exports.InstanceUserDataProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const version_1 = require("./version");
/**
 * Implementation of {@link IInstanceUserDataProvider}.
 *
 * Can be used as sub-class with override the desired methods
 * to add custom user data commands for WorkerInstanceFleet or WorkerInstanceConfiguration.
 *
 * @stability stable
 */
class InstanceUserDataProvider extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Method that is invoked before configuring the Cloud Watch Agent.
     *
     * @stability stable
     * @inheritdoc true
     */
    preCloudWatchAgent(_host) {
    }
    /**
     * Method that is invoked before the render queue configuration.
     *
     * @stability stable
     * @inheritdoc true
     */
    preRenderQueueConfiguration(_host) {
    }
    /**
     * Method that is invoked after configuring the connection to the render queue and before configuring the Deadline Worker.
     *
     * @stability stable
     * @inheritdoc true
     */
    preWorkerConfiguration(_host) {
    }
    /**
     * Method that is invoked after all configuration is done and worker started.
     *
     * @stability stable
     * @inheritdoc true
     */
    postWorkerLaunch(_host) {
    }
}
exports.InstanceUserDataProvider = InstanceUserDataProvider;
_a = JSII_RTTI_SYMBOL_1;
InstanceUserDataProvider[_a] = { fqn: "aws-rfdk.deadline.InstanceUserDataProvider", version: "0.27.0" };
/**
 * This construct can be used to configure Deadline Workers on an instance to connect to a RenderQueue, stream their log files to CloudWatch, and configure various settings of the Deadline Worker.
 *
 * The configuration happens on instance start-up using user data scripting.
 *
 * This configuration performs the following steps in order:
 * 1) Configure Cloud Watch Agent
 * 2) Configure Deadline Worker RenderQueue connection
 * 3) Configure Deadline Worker settings
 *
 * A `userDataProvider` can be specified that defines callback functions.
 * These callbacks can be used to inject user data commands at different points during the Worker instance configuration.
 *
 * Security Considerations
 * ------------------------
 * - The instances configured by this construct will download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 *
 * @stability stable
 */
class WorkerInstanceConfiguration extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        (_c = props.userDataProvider) === null || _c === void 0 ? void 0 : _c.preCloudWatchAgent(props.worker);
        if (props.cloudWatchLogSettings) {
            this.configureCloudWatchLogStream(props.worker, id, props.cloudWatchLogSettings, props.shouldInstallCloudWatchAgent);
        }
        (_d = props.userDataProvider) === null || _d === void 0 ? void 0 : _d.preRenderQueueConfiguration(props.worker);
        (_e = props.renderQueue) === null || _e === void 0 ? void 0 : _e.configureClientInstance({ host: props.worker });
        (_f = props.userDataProvider) === null || _f === void 0 ? void 0 : _f.preWorkerConfiguration(props.worker);
        this.listenerPort = (_h = (_g = props.workerSettings) === null || _g === void 0 ? void 0 : _g.listenerPort) !== null && _h !== void 0 ? _h : WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT;
        this.configureWorkerSettings(props.worker, id, props.workerSettings);
        (_j = props.userDataProvider) === null || _j === void 0 ? void 0 : _j.postWorkerLaunch(props.worker);
    }
    /**
     * This method can be used to configure a Deadline Worker instance to stream its logs to the AWS CloudWatch service.
     *
     * The logs that this configures to stream are:
     * - EC2 Instance UserData execution; this is the startup scripting that is run when the instance launches
     *    for the first time.
     * - Deadline Worker logs.
     * - Deadline Launcher logs.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param logGroupProps Configuration for the log group in CloudWatch.
     * @param shouldInstallAgent Boolean for if the worker's User Data should attempt to install the CloudWatch agent.
     * @stability stable
     */
    configureCloudWatchLogStream(worker, id, logGroupProps, shouldInstallAgent) {
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, id, logGroupProps);
        logGroup.grantWrite(worker);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        if (worker.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        }
        else {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        }
        new core_2.CloudWatchAgent(this, `${id}LogsConfig`, {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: worker,
            shouldInstallAgent,
        });
    }
    /**
     * This method can be used to set up the Deadline Worker application on an EC2 instance.
     *
     * From a practical
     * perspective, this is executing the script found in aws-rfdk/lib/deadline/scripts/[bash,powershell]/configureWorker.[sh,ps1]
     * to configure the Deadline Worker application.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param settings The Deadline Worker settings to apply.
     * @stability stable
     */
    configureWorkerSettings(worker, id, settings) {
        var _c, _d, _e, _f, _g;
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, `${id}ConfigScript`, {
            osType: worker.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        const configureWorkerPortAsset = new aws_s3_assets_1.Asset(this, `${id}WorkerListenerScript`, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'worker-listening-port.py'),
        });
        const configWorkerPortLocalPath = worker.userData.addS3DownloadCommand({
            bucket: configureWorkerPortAsset.bucket,
            bucketKey: configureWorkerPortAsset.s3ObjectKey,
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = (_d = (_c = settings === null || settings === void 0 ? void 0 : settings.groups) === null || _c === void 0 ? void 0 : _c.map(val => val.toLowerCase()).join(',')) !== null && _d !== void 0 ? _d : '';
        const pools = (_f = (_e = settings === null || settings === void 0 ? void 0 : settings.pools) === null || _e === void 0 ? void 0 : _e.map(val => val.toLowerCase()).join(',')) !== null && _f !== void 0 ? _f : '';
        configureWorkerScriptAsset.executeOn({
            host: worker,
            args: [
                `'${groups}'`,
                `'${pools}'`,
                `'${(_g = settings === null || settings === void 0 ? void 0 : settings.region) !== null && _g !== void 0 ? _g : ''}'`,
                `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                this.listenerPort.toString(),
                configWorkerPortLocalPath,
            ],
        });
    }
}
exports.WorkerInstanceConfiguration = WorkerInstanceConfiguration;
_b = JSII_RTTI_SYMBOL_1;
WorkerInstanceConfiguration[_b] = { fqn: "aws-rfdk.deadline.WorkerInstanceConfiguration", version: "0.27.0" };
/**
 * The default port to use for a worker to listen on for remote commands.
 */
WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT = 56032;
//# sourceMappingURL=data:application/json;base64,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