"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.testConstructTags = void 0;
const assert_1 = require("@aws-cdk/assert");
/**
 * The name of the tag that RFDK is expected to use
 */
const RFDK_TAG_NAME = 'aws-rfdk';
/**
 * The current version of the RFDK package
 */
const RFDK_VERSION = require('../../../package.json').version; // eslint-disable-line @typescript-eslint/no-require-imports
/**
 * Returns the resource properties for an expected RFDK tag for a given resource
 * type
 *
 * @param resourceType The L1 CloudFormation resource type (e.g. "AWS::EC2::Instance")
 * @param constructName The name of the L2 (or higher) RFDK construct class
 */
function getExpectedRfdkTagProperties(resourceType, constructName) {
    const expectedValue = `${RFDK_VERSION}:${constructName}`;
    if (resourceType === 'AWS::SSM::Parameter') {
        return {
            Tags: {
                [RFDK_TAG_NAME]: expectedValue,
            },
        };
    }
    else if (resourceType === 'AWS::AutoScaling::AutoScalingGroup') {
        return {
            Tags: assert_1.arrayWith({
                Key: RFDK_TAG_NAME,
                PropagateAtLaunch: true,
                Value: expectedValue,
            }),
        };
    }
    else {
        return {
            Tags: assert_1.arrayWith({
                Key: RFDK_TAG_NAME,
                Value: expectedValue,
            }),
        };
    }
}
/**
 * Creates tests for the specified resources created by a L2 (or higher construct) to ensure that the resources it
 * creates are created with the RFDK tagging convention.
 *
 * The convention is to create a tag named "aws-rfdk" whose value follows:
 *
 * <VERSION>:<CONSTRUCT_NAME>
 *
 * @param args Arguments to configure the creation of construct tagging tests
 */
/* eslint-disable-next-line jest/no-export */
function testConstructTags(args) {
    const { constructName, createConstruct, resourceTypeCounts } = args;
    const entries = Object.entries(resourceTypeCounts);
    test.each(entries)('tags %s x%d', (resourceType, expectedCount) => {
        // GIVEN
        const expectedProps = getExpectedRfdkTagProperties(resourceType, constructName);
        // WHEN
        const stack = createConstruct();
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike(resourceType, expectedCount, expectedProps));
    });
}
exports.testConstructTags = testConstructTags;
//# sourceMappingURL=data:application/json;base64,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