"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const mount_permissions_helper_1 = require("../lib/mount-permissions-helper");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MountableEFS', () => {
    let app;
    let stack;
    let vpc;
    let efsFS;
    let instance;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        efsFS = new efs.FileSystem(stack, 'EFS', { vpc });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the EFS's security group
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 2049,
            ToPort: 2049,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'InstanceInstanceSecurityGroupF0E2D5BE',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'EFSEfsSecurityGroup56F189CE',
                    'GroupId',
                ],
            },
        }));
        // Make sure we download the mountEfs script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/efs/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 r')));
    });
    describe.each([
        [undefined],
        [lib_1.MountPermissions.READONLY],
        [lib_1.MountPermissions.READWRITE],
    ])('access point with %s access permissions', (mountPermission) => {
        describe.each([
            [
                'unspecified POSIX user',
                {
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved non-root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '1000' },
                    expectedClientRootAccess: false,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '1000', gid: '0' },
                    expectedClientRootAccess: true,
                },
            ],
            [
                'resolved root POSIX user',
                {
                    posixUser: { uid: '0', gid: '1000' },
                    expectedClientRootAccess: true,
                },
            ],
        ])('%s', (_name, testCase) => {
            // GIVEN
            const { posixUser, expectedClientRootAccess } = testCase;
            const expectedActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mountPermission);
            if (expectedClientRootAccess) {
                expectedActions.push('elasticfilesystem:ClientRootAccess');
            }
            const mountPath = '/mnt/efs/fs1';
            let userData;
            let accessPoint;
            let expectedMountMode;
            beforeEach(() => {
                // GIVEN
                accessPoint = new efs.AccessPoint(stack, 'AccessPoint', {
                    fileSystem: efsFS,
                    posixUser,
                });
                const mount = new lib_1.MountableEfs(efsFS, {
                    filesystem: efsFS,
                    accessPoint,
                });
                expectedMountMode = (mountPermission === lib_1.MountPermissions.READONLY) ? 'ro' : 'rw';
                // WHEN
                mount.mountToLinuxInstance(instance, {
                    location: mountPath,
                    permissions: mountPermission,
                });
                userData = stack.resolve(instance.userData.render());
            });
            test('userdata specifies access point when mounting', () => {
                // THEN
                expect(userData).toEqual({
                    'Fn::Join': [
                        '',
                        expect.arrayContaining([
                            expect.stringMatching(new RegExp('(\\n|^)bash \\./mountEfs.sh $')),
                            stack.resolve(efsFS.fileSystemId),
                            ` ${mountPath} ${expectedMountMode},iam,accesspoint=`,
                            stack.resolve(accessPoint.accessPointId),
                            expect.stringMatching(/^\n/),
                        ]),
                    ],
                });
            });
            test('grants IAM access point permissions', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: assert_1.objectLike({
                        Statement: assert_1.arrayWith({
                            Action: expectedActions.length === 1 ? expectedActions[0] : expectedActions,
                            Condition: {
                                StringEquals: {
                                    'elasticfilesystem:AccessPointArn': stack.resolve(accessPoint.accessPointArn),
                                },
                            },
                            Effect: 'Allow',
                            Resource: stack.resolve(efsFS.node.defaultChild.attrArn),
                        }),
                        Version: '2012-10-17',
                    }),
                    Roles: assert_1.arrayWith(
                    // The Policy construct micro-optimizes the reference to a role in the same stack using its logical ID
                    stack.resolve(instance.role.node.defaultChild.ref)),
                }));
            });
        });
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 rw,option1,option2')));
    });
    test('asset is singleton', () => {
        var _a;
        // GIVEN
        const mount1 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        const mount2 = new lib_1.MountableEfs(efsFS, {
            filesystem: efsFS,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/efs/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = (_a = userData.match(regex)) !== null && _a !== void 0 ? _a : [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
});
//# sourceMappingURL=data:application/json;base64,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