"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Import cert', () => {
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
        subject: { cn: 'Server' },
    });
    const certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
    new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
        cert: secretCert.cert,
        certChain: secretCert.certChain,
        key: secretCert.key,
        passphrase: secretCert.passphrase,
    });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
        X509CertificatePem: {
            Cert: {
                'Fn::GetAtt': [
                    'Pem',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Pem',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
            CertChain: '',
        },
        Tags: [
            {
                Key: 'AcmCertImport-F4E2ABF9',
                Value: 'f4e2abf974443234fdb095fafcfa9ee2',
            },
            {
                Key: 'Name',
                Value: 'f4e2abf974443234fdb095fafcfa9ee2',
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.countResources('AWS::DynamoDB::Table', 2));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:ConditionCheckItem',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'Pem',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'Pem',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'acm:AddTagsToCertificate',
                        'acm:ImportCertificate',
                    ],
                    Condition: {
                        StringEquals: {
                            'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                        },
                    },
                    Resource: '*',
                },
                {
                    Action: [
                        'acm:DeleteCertificate',
                        'acm:DescribeCertificate',
                        'acm:GetCertificate',
                    ],
                    Resource: '*',
                },
            ],
        },
    }));
    // Expect Lambda for doing the cert importation to use the importCert() handler, openssl layer, and set DATABASE
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', (props, error) => {
        var _a, _b;
        if (!props.Handler || props.Handler !== 'x509-certificate.importCert') {
            error.failureReason = 'x509-certificate.importCert handler not found';
            error.resource = props.Handler;
            return false;
        }
        // Our test for the correct openssl lambda layer does not include the version, so we use a filter
        // function to do a partial match
        const filterOpensslArn = (value) => {
            return value.toString().includes('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:');
        };
        if (!props.Layers
            || !Array.isArray(props.Layers)
            || Array.of(props.Layers).filter(filterOpensslArn).length === 0) {
            error.failureReason = 'openssl Lambda Layer missing';
            error.resource = props.Layers;
            return false;
        }
        if (!props.Environment
            || !props.Environment.Variables
            || !props.Environment.Variables.DATABASE) {
            error.failureReason = 'DATABASE environment variable not set';
            error.resource = (_b = (_a = props.Environment) === null || _a === void 0 ? void 0 : _a.Variables) === null || _b === void 0 ? void 0 : _b.DATABASE;
            return false;
        }
        return true;
    }));
});
//# sourceMappingURL=data:application/json;base64,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