"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const tag_helpers_1 = require("./tag-helpers");
let app;
let infraStack;
let hmStack;
let wfStack;
let vpc;
let healthMonitor;
class TestMonitorableFleet extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            minCapacity: props.minCapacity,
        });
        this.connections = new aws_ec2_1.Connections();
        this.targetCapacity = parseInt(fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [fleet.autoScalingGroupArn],
                })],
        });
    }
}
describe('HealthMonitor', () => {
    beforeEach(() => {
        app = new core_1.App();
        infraStack = new core_1.Stack(app, 'infraStack');
        hmStack = new core_1.Stack(app, 'hmStack');
        wfStack = new core_1.Stack(app, 'wfStack');
        vpc = new aws_ec2_1.Vpc(infraStack, 'VPC');
    });
    test('validating default health monitor properties', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: 'kms:*',
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:GenerateDataKey',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: '*',
                    },
                ],
            },
            Description: `This key is used to encrypt SNS messages for ${healthMonitor.node.uniqueId}.`,
            EnableKeyRotation: true,
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: {
                            Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                        },
                        Sid: '0',
                    },
                ],
            },
            Topics: [
                {
                    Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                },
            ],
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: {
                'Fn::GetAtt': [
                    `${hmStack.getLogicalId(healthMonitor.node.findChild('SNSEncryptionKey').node.defaultChild)}`,
                    'Arn',
                ],
            },
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating health monitor properties while passing a key', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
            encryptionKey: aws_kms_1.Key.fromKeyArn(hmStack, 'importedKey', 'arn:aws:kms:us-west-2:123456789012:key/testarn'),
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::KMS::Key'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: 'arn:aws:kms:us-west-2:123456789012:key/testarn',
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating the target with default health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(hmStack).notTo((assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: assert_1.arrayWith(assert_1.deepObjectLike({
                CidrIp: '0.0.0.0/0',
                FromPort: 8081,
                IpProtocol: 'tcp',
                ToPort: 8081,
            })),
        })));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '8081',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('validating the target with custom health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {
            port: 7171,
        });
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('2 ASG gets registered to same LB', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, { port: 7171 });
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, { port: 7171 });
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 8,
            ActionsEnabled: true,
            DatapointsToAlarm: 8,
            Threshold: 0,
            TreatMissingData: 'notBreaching',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            ActionsEnabled: true,
            DatapointsToAlarm: 1,
            Threshold: 35,
            TreatMissingData: 'notBreaching',
        }));
    });
    test('validating LB target limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 50,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating LB listener limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'listeners-per-application-load-balancer',
                    max: 1,
                }, {
                    name: 'target-groups-per-action-on-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target group limit per lb', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'target-groups-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target limit exhaustion', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 2,
        });
        expect(() => {
            healthMonitor.registerFleet(fleet, {});
        }).toThrowError(/AWS service limit \"targets-per-application-load-balancer\" reached. Limit: 1/);
    });
    test('validating deletion protection', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            deletionProtection: false,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    test('specifying a subnet', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        // Make sure it has the public subnets
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assert_1.stringLike('*PublicSubnet*') },
                { 'Fn::ImportValue': assert_1.stringLike('*PublicSubnet*') },
            ],
        }));
        // Make sure the private subnets aren't present
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            Subnets: [
                { 'Fn::ImportValue': assert_1.notMatching(assert_1.stringLike('*PrivateSubnet*')) },
                { 'Fn::ImportValue': assert_1.notMatching(assert_1.stringLike('*PrivateSubnet*')) },
            ],
        }));
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'HealthMonitor',
            createConstruct: () => {
                // GIVEN
                const fleetStack = new core_1.Stack(app, 'FleetStack');
                const fleet = new TestMonitorableFleet(fleetStack, 'workerFleet', {
                    vpc,
                });
                // WHEN
                healthMonitor = new lib_1.HealthMonitor(hmStack, 'HealthMonitor', {
                    vpc,
                });
                healthMonitor.registerFleet(fleet, {});
                return hmStack;
            },
            resourceTypeCounts: {
                'AWS::KMS::Key': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::EC2::SecurityGroup': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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