"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const efs = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto an instance.
 *
 * An optional EFS access point can be specified for mounting the EFS file-system. For more information on using EFS
 * Access Points, see https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html. For this to work properly, the
 * EFS mount helper is required. The EFS Mount helper comes pre-installed on Amazon Linux 2. For other Linux
 * distributions, the host machine must have the Amazon EFS client installed. We advise installing the Amazon EFS Client
 * when building your AMI. For instructions on installing the Amazon EFS client for other distributions, see
 * https://docs.aws.amazon.com/efs/latest/ug/installing-amazon-efs-utils.html#installing-other-distro.
 *
 * NOTE: Without an EFS access point, the file-system is writeable only by the root user.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class MountableEfs {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        this.fileSystem = props.filesystem;
        this.accessPoint = props.accessPoint;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        if (core_1.Construct.isConstruct(target)) {
            target.node.addDependency(this.props.filesystem.mountTargetsAvailable);
        }
        if (this.props.accessPoint) {
            const grantActions = mount_permissions_helper_1.MountPermissionsHelper.toEfsIAMActions(mount === null || mount === void 0 ? void 0 : mount.permissions);
            if (this.accessPointRequiresClientRootAccess(this.props.accessPoint)) {
                grantActions.push('elasticfilesystem:ClientRootAccess');
            }
            target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                resources: [
                    this.props.filesystem.node.defaultChild.attrArn,
                ],
                actions: grantActions,
                conditions: {
                    StringEquals: {
                        'elasticfilesystem:AccessPointArn': this.props.accessPoint.accessPointArn,
                    },
                },
            }));
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton();
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.accessPoint) {
            mountOptions.push('iam', `accesspoint=${this.props.accessPoint.accessPointId}`);
        }
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Uses a CDK escape-hatch to fetch the UID/GID of the access point POSIX user.
     *
     * @param accessPoint The access point to obtain the POSIX user for
     */
    getAccessPointPosixUser(accessPoint) {
        const accessPointResource = accessPoint.node.defaultChild;
        return accessPointResource.posixUser;
    }
    /**
     * Uses a synthesis-time check to determine whether an access point is setting its UID/GID to 0 (root). Mounting such
     * an access point requires the `ClientRootAccess` IAM permission.
     *
     * If this introspection is possible and the access point is determined to require root access, the method returns
     * true.
     *
     * If there is no information at synthesis-time, the method returns false as a secure default.
     *
     * @param accessPoint The access point to introspect
     */
    accessPointRequiresClientRootAccess(accessPoint) {
        if (accessPoint instanceof efs.AccessPoint) {
            const posixUser = this.getAccessPointPosixUser(accessPoint);
            // The following code path is cannot be triggered using the L2 construct for EFS Access Points. It currently
            // accepts a PosixUser struct. We will skip coverage for the time-being.
            /* istanbul ignore next */
            if (core_1.isResolvableObject(posixUser)) {
                // We can't know at synthesis time whether this POSIX user is root. Use secure defaults.
                return false;
            }
            if (!posixUser) {
                // No POSIX user specified we will not grant ClientRootAccess permission to opt on the side of secure defaults.
                return false;
            }
            // We have synthesis-time values for the UID/GID being set in the access point. Return true if either is 0 (root).
            return Number(posixUser.uid) === 0 || Number(posixUser.gid) === 0;
        }
        else {
            // This code path is for imported or custom-implementations of efs.AccessPoint
            // We cannot introspect the access point, so we will impose secure defaults and not grant ClientRootAccess.
            return false;
        }
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton() {
        var _b;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableEfs = MountableEfs;
_a = JSII_RTTI_SYMBOL_1;
MountableEfs[_a] = { fqn: "aws-rfdk.MountableEfs", version: "0.27.0" };
//# sourceMappingURL=data:application/json;base64,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