"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchAgent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("./script-assets");
/**
 * This construct is a thin wrapper that provides the ability to install and configure the CloudWatchAgent ( https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html ) on one or more EC2 instances during instance startup.
 *
 * It accomplishes this by downloading and executing the configuration script on the instance.
 * The script will download the CloudWatch Agent installer,
 * optionally verify the installer, and finally install the CloudWatch Agent.
 * The installer is downloaded via the Amazon S3 API, thus, this construct can be used
 * on instances that have no access to the internet as long as the VPC contains
 * an VPC Gateway Endpoint for S3 ( https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-s3.html ).
 *
 * {@link CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR} - Context variable to skip validation
 * of the downloaded CloudWatch Agent installer if set to 'TRUE'.
 * WARNING: Only use this if your deployments are failing due to a validation failure,
 * but you have verified that the failure is benign.
 *
 * Resources Deployed
 * ------------------------
 * - String SSM Parameter in Systems Manager Parameter Store to store the cloudwatch agent configuration;
 * - A script Asset which is uploaded to S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class CloudWatchAgent extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        const shouldInstallAgent = (_b = props.shouldInstallAgent) !== null && _b !== void 0 ? _b : true;
        // Create the asset for the configuration script
        this.configurationScript = script_assets_1.ScriptAsset.fromPathConvention(scope, 'CloudWatchConfigurationScriptAsset', {
            osType: props.host.osType,
            baseName: 'configureCloudWatchAgent',
            rootDir: path.join(__dirname, '../scripts/'),
        });
        // Create a new SSM Parameter holding the json configuration
        this.ssmParameterForConfig = new aws_ssm_1.StringParameter(scope, 'StringParameter', {
            description: 'config file for Repository logs config',
            stringValue: props.cloudWatchConfig,
        });
        this.grantRead(props.host);
        this.configure(props.host, shouldInstallAgent, this.node.tryGetContext(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR) === 'TRUE');
    }
    /**
     * Grants read permissions to the principal on the assets bucket and parameter store.
     */
    grantRead(grantee) {
        this.configurationScript.grantRead(grantee);
        this.ssmParameterForConfig.grantRead(grantee);
    }
    /**
     * Configures the CloudWatch Agent on the target host.
     *
     * This is done by adding UserData commands to the target host.
     *
     * @param host The host to configure the CloudWatch agent on
     * @param shouldInstallAgent Attempts to install the CloudWatch agent on the instance if set to true.
     * @param skipValidation Skips the validation of the CloudWatch agent installer if set to true.
     */
    configure(host, shouldInstallAgent, skipValidation) {
        const region = core_1.Stack.of(this).region;
        if (shouldInstallAgent) {
            // Grant access to the required CloudWatch Agent and GPG installer files.
            const cloudWatchAgentBucket = aws_s3_1.Bucket.fromBucketArn(this, 'CloudWatchAgentBucket', `arn:aws:s3:::amazoncloudwatch-agent-${region}`);
            cloudWatchAgentBucket.grantRead(host);
            const gpgBucket = aws_s3_1.Bucket.fromBucketArn(this, 'GpgBucket', `arn:aws:s3:::rfdk-external-dependencies-${region}`);
            gpgBucket.grantRead(host);
        }
        const scriptArgs = [];
        // Flags must be set before positional arguments for some scripts
        if (shouldInstallAgent) {
            scriptArgs.push(CloudWatchAgent.INSTALL_CWAGENT_FLAG);
        }
        if (skipValidation) {
            scriptArgs.push(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG);
        }
        // This assumes that the CloudWatch agent construct is always put in the same region as the instance or ASG
        // using it, which should always hold true.
        scriptArgs.push(region);
        scriptArgs.push(this.ssmParameterForConfig.parameterName);
        this.configurationScript.executeOn({
            host,
            args: scriptArgs,
        });
    }
}
exports.CloudWatchAgent = CloudWatchAgent;
_a = JSII_RTTI_SYMBOL_1;
CloudWatchAgent[_a] = { fqn: "aws-rfdk.CloudWatchAgent", version: "0.27.0" };
/**
 * The context variable to indicate that CloudWatch agent installer validation should be skipped.
 *
 * @stability stable
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR = 'RFDK_SKIP_CWAGENT_VALIDATION';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG = '-s';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.INSTALL_CWAGENT_FLAG = '-i';
//# sourceMappingURL=data:application/json;base64,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