/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.rugged.raster;

import java.lang.reflect.Array;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathUtils;
import org.hipparchus.util.Precision;
import org.orekit.bodies.GeodeticPoint;
import org.orekit.rugged.errors.DumpManager;
import org.orekit.rugged.errors.RuggedException;
import org.orekit.rugged.errors.RuggedMessages;
import org.orekit.rugged.raster.EarthHemisphere;
import org.orekit.rugged.raster.Tile;
import org.orekit.rugged.raster.TileFactory;
import org.orekit.rugged.raster.TileUpdater;
import org.orekit.rugged.raster.UpdatableTile;

public class TilesCache<T extends Tile> {
    private static double STEP_EQUALITY = 5.0 * Precision.EPSILON;
    private final TileFactory<T> factory;
    private final TileUpdater updater;
    private final boolean isOverlapping;
    private final T[] tiles;

    public TilesCache(TileFactory<T> factory, TileUpdater updater, int maxTiles, boolean isOverlappingTiles) {
        this.factory = factory;
        this.updater = updater;
        this.isOverlapping = isOverlappingTiles;
        Tile[] array = (Tile[])Array.newInstance(Tile.class, maxTiles);
        this.tiles = array;
    }

    public T getTile(double latitude, double longitude) {
        int i;
        for (i = 0; i < this.tiles.length; ++i) {
            T tile = this.tiles[i];
            if (tile == null || tile.getLocation(latitude, longitude) != Tile.Location.HAS_INTERPOLATION_NEIGHBORS) continue;
            while (i > 0) {
                this.tiles[i] = this.tiles[i - 1];
                --i;
            }
            this.tiles[0] = tile;
            return tile;
        }
        for (i = this.tiles.length - 1; i > 0; --i) {
            this.tiles[i] = this.tiles[i - 1];
        }
        T tile = this.createTile(latitude, longitude);
        if (!this.isOverlapping) {
            Tile.Location pointLocation = tile.getLocation(latitude, longitude);
            if (pointLocation != Tile.Location.HAS_INTERPOLATION_NEIGHBORS) {
                return this.createZipperTile(tile, latitude, longitude, pointLocation);
            }
            this.tiles[0] = tile;
            return tile;
        }
        if (tile.getLocation(latitude, longitude) != Tile.Location.HAS_INTERPOLATION_NEIGHBORS) {
            throw new RuggedException(RuggedMessages.TILE_WITHOUT_REQUIRED_NEIGHBORS_SELECTED, FastMath.toDegrees((double)latitude), FastMath.toDegrees((double)longitude));
        }
        this.tiles[0] = tile;
        return tile;
    }

    private T createTile(double latitude, double longitude) {
        T tile = this.factory.createTile();
        Boolean wasSuspended = DumpManager.suspend();
        this.updater.updateTile(latitude, longitude, (UpdatableTile)tile);
        DumpManager.resume(wasSuspended);
        tile.tileUpdateCompleted();
        return tile;
    }

    private T createZipperTile(T currentTile, double latitude, double longitude, Tile.Location pointLocation) {
        T zipperTile = null;
        switch (pointLocation) {
            case NORTH: {
                zipperTile = this.createZipperNorthOrSouth(EarthHemisphere.NORTH, longitude, currentTile);
                break;
            }
            case SOUTH: {
                zipperTile = this.createZipperNorthOrSouth(EarthHemisphere.SOUTH, longitude, currentTile);
                break;
            }
            case WEST: {
                zipperTile = this.createZipperWestOrEast(EarthHemisphere.WEST, latitude, currentTile);
                break;
            }
            case EAST: {
                zipperTile = this.createZipperWestOrEast(EarthHemisphere.EAST, latitude, currentTile);
                break;
            }
            case NORTH_WEST: {
                zipperTile = this.createCornerZipper(EarthHemisphere.NORTH, EarthHemisphere.WEST, latitude, longitude, currentTile);
                break;
            }
            case NORTH_EAST: {
                zipperTile = this.createCornerZipper(EarthHemisphere.NORTH, EarthHemisphere.EAST, latitude, longitude, currentTile);
                break;
            }
            case SOUTH_WEST: {
                zipperTile = this.createCornerZipper(EarthHemisphere.SOUTH, EarthHemisphere.WEST, latitude, longitude, currentTile);
                break;
            }
            case SOUTH_EAST: {
                zipperTile = this.createCornerZipper(EarthHemisphere.SOUTH, EarthHemisphere.EAST, latitude, longitude, currentTile);
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        for (int i = this.tiles.length - 1; i > 0; --i) {
            this.tiles[i] = this.tiles[i - 1];
        }
        this.tiles[1] = currentTile;
        this.tiles[0] = zipperTile;
        return zipperTile;
    }

    private T initializeZipperTile(double zipperLatMin, double zipperLonMin, double zipperLatStep, double zipperLonStep, int zipperLatRows, int zipperLonCols, double[][] zipperElevations) {
        T zipperTile = this.factory.createTile();
        zipperTile.setGeometry(zipperLatMin, zipperLonMin, zipperLatStep, zipperLonStep, zipperLatRows, zipperLonCols);
        for (int iLat = 0; iLat < zipperLatRows; ++iLat) {
            for (int jLon = 0; jLon < zipperLonCols; ++jLon) {
                zipperTile.setElevation(iLat, jLon, zipperElevations[iLat][jLon]);
            }
        }
        zipperTile.tileUpdateCompleted();
        return zipperTile;
    }

    private T createZipperNorthOrSouth(EarthHemisphere latitudeHemisphere, double longitude, T currentTile) {
        double[][] elevations;
        double zipperLonMin;
        double zipperLatMin;
        int currentTileLatRows = currentTile.getLatitudeRows();
        int currentTileLonCols = currentTile.getLongitudeColumns();
        double currentTileLatStep = currentTile.getLatitudeStep();
        double currentTileLonStep = currentTile.getLongitudeStep();
        double currentTileMinLat = currentTile.getMinimumLatitude();
        double currentTileMinLon = currentTile.getMinimumLongitude();
        T tileNorthOrSouth = this.createNorthOrSouthTile(latitudeHemisphere, longitude, currentTileMinLat, currentTileLatStep, currentTileLatRows);
        int zipperLatRows = 4;
        double zipperLatStep = currentTileLatStep;
        boolean isSameStepLat = true;
        double northSouthLatitudeStep = tileNorthOrSouth.getLatitudeStep();
        if (!(Math.abs(currentTileLatStep - northSouthLatitudeStep) < STEP_EQUALITY)) {
            isSameStepLat = false;
            if (northSouthLatitudeStep < currentTileLatStep) {
                zipperLatStep = northSouthLatitudeStep;
            }
        }
        double zipperLonStep = currentTileLonStep;
        int zipperLonCols = currentTileLonCols;
        boolean isSameStepLon = true;
        double northSouthLongitudeStep = tileNorthOrSouth.getLongitudeStep();
        if (!(Math.abs(currentTileLonStep - northSouthLongitudeStep) < STEP_EQUALITY)) {
            isSameStepLon = false;
            if (northSouthLongitudeStep < currentTileLonStep) {
                zipperLonStep = northSouthLongitudeStep;
                zipperLonCols = tileNorthOrSouth.getLongitudeColumns();
            }
        }
        switch (latitudeHemisphere) {
            case NORTH: {
                double currentTileNorthernLatitude = currentTileMinLat - 0.5 * currentTileLatStep + (double)currentTileLatRows * currentTileLatStep;
                zipperLatMin = currentTileNorthernLatitude - 2.0 * zipperLatStep + 0.5 * zipperLatStep;
                zipperLonMin = currentTileMinLon;
                elevations = this.getZipperNorthSouthElevations(zipperLonCols, tileNorthOrSouth, currentTile, isSameStepLat, isSameStepLon, zipperLatMin, zipperLonMin, zipperLatStep, zipperLonStep);
                break;
            }
            case SOUTH: {
                double currentTileSouthernLatitude = currentTileMinLat - 0.5 * currentTileLatStep;
                zipperLatMin = currentTileSouthernLatitude - 2.0 * zipperLatStep + 0.5 * zipperLatStep;
                zipperLonMin = currentTileMinLon;
                elevations = this.getZipperNorthSouthElevations(zipperLonCols, currentTile, tileNorthOrSouth, isSameStepLat, isSameStepLon, zipperLatMin, zipperLonMin, zipperLatStep, zipperLonStep);
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        T zipperNorthOrSouth = this.initializeZipperTile(zipperLatMin, zipperLonMin, zipperLatStep, zipperLonStep, 4, zipperLonCols, elevations);
        return zipperNorthOrSouth;
    }

    private T createZipperWestOrEast(EarthHemisphere longitudeHemisphere, double latitude, T currentTile) {
        double[][] elevations;
        double zipperLonMin;
        int currentTileLatRows = currentTile.getLatitudeRows();
        int currentTileLonCols = currentTile.getLongitudeColumns();
        double currentTileLatStep = currentTile.getLatitudeStep();
        double currentTileLonStep = currentTile.getLongitudeStep();
        double currentTileMinLon = currentTile.getMinimumLongitude();
        T tileWestOrEast = this.createEastOrWestTile(longitudeHemisphere, latitude, currentTileMinLon, currentTileLonStep, currentTileLonCols);
        if (!(Math.abs(currentTileLatStep - tileWestOrEast.getLatitudeStep()) < STEP_EQUALITY) || !(Math.abs(currentTileLonStep - tileWestOrEast.getLongitudeStep()) < STEP_EQUALITY)) {
            throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
        }
        double zipperLonStep = currentTileLonStep;
        int zipperLatRows = currentTileLatRows;
        int zipperLonCols = 4;
        switch (longitudeHemisphere) {
            case WEST: {
                zipperLonMin = currentTileMinLon - 2.0 * currentTileLonStep;
                elevations = this.getZipperEastWestElevations(zipperLatRows, currentTile, tileWestOrEast);
                break;
            }
            case EAST: {
                zipperLonMin = currentTileMinLon + (double)(currentTileLonCols - 2) * currentTileLonStep;
                elevations = this.getZipperEastWestElevations(zipperLatRows, tileWestOrEast, currentTile);
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        T zipperWestOrEast = this.initializeZipperTile(currentTile.getMinimumLatitude(), zipperLonMin, currentTileLatStep, zipperLonStep, zipperLatRows, 4, elevations);
        return zipperWestOrEast;
    }

    private int computeLatitudeIndex(double latitude, double latitudeMin, double latitudeStep, int latitudeRows) {
        double doubleLatitudeIndex = (latitude - (latitudeMin - 0.5 * latitudeStep)) / latitudeStep;
        return FastMath.max((int)0, (int)FastMath.min((int)(latitudeRows - 1), (int)((int)FastMath.floor((double)doubleLatitudeIndex))));
    }

    private int computeLongitudeIndex(double longitude, double longitudeMin, double longitudeStep, int longitudeColumns) {
        double doubleLongitudeIndex = (longitude - (longitudeMin - 0.5 * longitudeStep)) / longitudeStep;
        return FastMath.max((int)0, (int)FastMath.min((int)(longitudeColumns - 1), (int)((int)FastMath.floor((double)doubleLongitudeIndex))));
    }

    private double[][] getZipperNorthSouthElevations(int zipperLonCols, T northernTile, T southernTile, boolean isSameStepLat, boolean isSameStepLon, double zipperLatMin, double zipperLonMin, double zipperLatStep, double zipperLonStep) {
        double[][] elevations = new double[4][zipperLonCols];
        if (isSameStepLat && isSameStepLon) {
            for (int jLon = 0; jLon < zipperLonCols; ++jLon) {
                boolean lat3 = true;
                elevations[3][jLon] = northernTile.getElevationAtIndices(1, jLon);
                boolean lat2 = false;
                elevations[2][jLon] = northernTile.getElevationAtIndices(0, jLon);
                int lat1 = southernTile.getLatitudeRows() - 1;
                elevations[1][jLon] = southernTile.getElevationAtIndices(lat1, jLon);
                int lat0 = southernTile.getLatitudeRows() - 2;
                elevations[0][jLon] = southernTile.getElevationAtIndices(lat0, jLon);
            }
        } else {
            double deltaLon = 0.1 * zipperLonStep;
            double zipperLonMax = MathUtils.normalizeAngle((double)(zipperLonMin + (double)(zipperLonCols - 1) * zipperLonStep), (double)0.0);
            double northernMinLat = northernTile.getMinimumLatitude();
            double northernLatStep = northernTile.getLatitudeStep();
            int northernLatRows = northernTile.getLatitudeRows();
            double northernMinLon = northernTile.getMinimumLongitude();
            double northernLonStep = northernTile.getLongitudeStep();
            int northernLonCols = northernTile.getLongitudeColumns();
            double southernMinLat = southernTile.getMinimumLatitude();
            double southernLatStep = southernTile.getLatitudeStep();
            int southernLatRows = southernTile.getLatitudeRows();
            double southernMinLon = southernTile.getMinimumLongitude();
            double southernLonStep = southernTile.getLongitudeStep();
            int southernLonCols = southernTile.getLongitudeColumns();
            for (double zipperLonCurrent = zipperLonMin + deltaLon; zipperLonCurrent <= zipperLonMax + 2.0 * deltaLon; zipperLonCurrent += zipperLonStep) {
                int zipperLonIndex = this.computeLongitudeIndex(zipperLonCurrent, zipperLonMin, zipperLonStep, zipperLonCols);
                int northenLongitudeIndex = this.computeLongitudeIndex(zipperLonCurrent, northernMinLon, northernLonStep, northernLonCols);
                double zipperLat3 = zipperLatMin + 3.1 * zipperLatStep;
                int lat3Index = this.computeLatitudeIndex(zipperLat3, northernMinLat, northernLatStep, northernLatRows);
                elevations[3][zipperLonIndex] = northernTile.getElevationAtIndices(lat3Index, northenLongitudeIndex);
                boolean lat2Index = false;
                elevations[2][zipperLonIndex] = northernTile.getElevationAtIndices(0, northenLongitudeIndex);
                int southernLongitudeIndex = this.computeLongitudeIndex(zipperLonCurrent, southernMinLon, southernLonStep, southernLonCols);
                int lat1Index = southernTile.getLatitudeRows() - 1;
                elevations[1][zipperLonIndex] = southernTile.getElevationAtIndices(lat1Index, southernLongitudeIndex);
                double zipperLat0 = zipperLatMin + 0.1 * zipperLatStep;
                int lat0Index = this.computeLatitudeIndex(zipperLat0, southernMinLat, southernLatStep, southernLatRows);
                elevations[0][zipperLonIndex] = southernTile.getElevationAtIndices(lat0Index, southernLongitudeIndex);
            }
        }
        return elevations;
    }

    private double[][] getZipperEastWestElevations(int zipperLatRows, T easternTile, T westernTile) {
        double[][] elevations = new double[zipperLatRows][4];
        for (int iLat = 0; iLat < zipperLatRows; ++iLat) {
            boolean lon3 = true;
            elevations[iLat][3] = easternTile.getElevationAtIndices(iLat, 1);
            boolean lon2 = false;
            elevations[iLat][2] = easternTile.getElevationAtIndices(iLat, 0);
            int lon1 = westernTile.getLongitudeColumns() - 1;
            elevations[iLat][1] = westernTile.getElevationAtIndices(iLat, lon1);
            int lon0 = westernTile.getLongitudeColumns() - 2;
            elevations[iLat][0] = westernTile.getElevationAtIndices(iLat, lon0);
        }
        return elevations;
    }

    private T createCornerZipper(EarthHemisphere latitudeHemisphere, EarthHemisphere longitudeHemisphere, double latitude, double longitude, T currentTile) {
        T aboveRightTile;
        T aboveLeftTile;
        T belowRightTile;
        T belowLeftTile;
        int currentTileLatRows = currentTile.getLatitudeRows();
        int currentTileLonCols = currentTile.getLongitudeColumns();
        double currentTileLatStep = currentTile.getLatitudeStep();
        double currentTileLonStep = currentTile.getLongitudeStep();
        double currentTileLonMin = currentTile.getMinimumLongitude();
        double currentTileLatMin = currentTile.getMinimumLatitude();
        block0 : switch (latitudeHemisphere) {
            case NORTH: {
                T tileWest;
                switch (longitudeHemisphere) {
                    case WEST: {
                        tileWest = this.createEastOrWestTile(EarthHemisphere.WEST, latitude, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        T tileNorth = this.createNorthOrSouthTile(EarthHemisphere.NORTH, longitude, currentTileLatMin, currentTileLatStep, currentTileLatRows);
                        T tileNorthWest = this.createIntercardinalTile(EarthHemisphere.NORTH, currentTileLatMin, currentTileLatStep, currentTileLatRows, EarthHemisphere.WEST, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        belowLeftTile = tileWest;
                        belowRightTile = currentTile;
                        aboveLeftTile = tileNorthWest;
                        aboveRightTile = tileNorth;
                        break block0;
                    }
                    case EAST: {
                        T tileEast = this.createEastOrWestTile(EarthHemisphere.EAST, latitude, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        T tileNorth = this.createNorthOrSouthTile(EarthHemisphere.NORTH, longitude, currentTileLatMin, currentTileLatStep, currentTileLatRows);
                        T tileNorthEast = this.createIntercardinalTile(EarthHemisphere.NORTH, currentTileLatMin, currentTileLatStep, currentTileLatRows, EarthHemisphere.EAST, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        belowLeftTile = currentTile;
                        belowRightTile = tileEast;
                        aboveLeftTile = tileNorth;
                        aboveRightTile = tileNorthEast;
                        break block0;
                    }
                }
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
            case SOUTH: {
                T tileWest;
                switch (longitudeHemisphere) {
                    case WEST: {
                        tileWest = this.createEastOrWestTile(EarthHemisphere.WEST, latitude, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        T tileSouth = this.createNorthOrSouthTile(EarthHemisphere.SOUTH, longitude, currentTileLatMin, currentTileLatStep, currentTileLatRows);
                        T tileSouthhWest = this.createIntercardinalTile(EarthHemisphere.SOUTH, currentTileLatMin, currentTileLatStep, currentTileLatRows, EarthHemisphere.WEST, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        belowLeftTile = tileSouthhWest;
                        belowRightTile = tileSouth;
                        aboveLeftTile = tileWest;
                        aboveRightTile = currentTile;
                        break block0;
                    }
                    case EAST: {
                        T tileEast = this.createEastOrWestTile(EarthHemisphere.EAST, latitude, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        T tileSouth = this.createNorthOrSouthTile(EarthHemisphere.SOUTH, longitude, currentTileLatMin, currentTileLatStep, currentTileLatRows);
                        T tileSouthhEast = this.createIntercardinalTile(EarthHemisphere.SOUTH, currentTileLatMin, currentTileLatStep, currentTileLatRows, EarthHemisphere.EAST, currentTileLonMin, currentTileLonStep, currentTileLonCols);
                        belowLeftTile = tileSouth;
                        belowRightTile = tileSouthhEast;
                        aboveLeftTile = currentTile;
                        aboveRightTile = tileEast;
                        break block0;
                    }
                }
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        if (!(Math.abs(belowLeftTile.getLatitudeStep() - belowRightTile.getLatitudeStep()) < STEP_EQUALITY && Math.abs(belowLeftTile.getLongitudeStep() - belowRightTile.getLongitudeStep()) < STEP_EQUALITY && Math.abs(aboveLeftTile.getLatitudeStep() - aboveRightTile.getLatitudeStep()) < STEP_EQUALITY && Math.abs(aboveLeftTile.getLongitudeStep() - aboveRightTile.getLongitudeStep()) < STEP_EQUALITY)) {
            throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
        }
        double belowLatitudeStep = belowRightTile.getLatitudeStep();
        double aboveLatitudeStep = aboveRightTile.getLatitudeStep();
        double zipperLatStep = belowLatitudeStep;
        boolean isSameStepLat = true;
        if (!(Math.abs(belowLatitudeStep - aboveLatitudeStep) < STEP_EQUALITY)) {
            isSameStepLat = false;
            if (aboveLatitudeStep < belowLatitudeStep) {
                zipperLatStep = aboveLatitudeStep;
            }
        }
        double belowLongitudeStep = belowRightTile.getLongitudeStep();
        double aboveLongitudeStep = aboveRightTile.getLongitudeStep();
        double zipperLonStep = belowLongitudeStep;
        boolean isSameStepLon = true;
        if (!(Math.abs(belowLongitudeStep - aboveLongitudeStep) < STEP_EQUALITY)) {
            isSameStepLon = false;
            if (aboveLongitudeStep < belowLongitudeStep) {
                zipperLonStep = aboveLongitudeStep;
            }
        }
        GeodeticPoint zipperCorner = this.computeCornerZipperOrigin(zipperLatStep, zipperLonStep, latitudeHemisphere, currentTileLatMin, currentTileLatStep, currentTileLatRows, longitudeHemisphere, currentTileLonMin, currentTileLonStep, currentTileLonCols);
        return this.initializeCornerZipperTile(zipperCorner.getLatitude(), zipperCorner.getLongitude(), zipperLatStep, zipperLonStep, belowLeftTile, aboveLeftTile, belowRightTile, aboveRightTile, isSameStepLat, isSameStepLon);
    }

    private T initializeCornerZipperTile(double zipperLatMin, double zipperLonMin, double zipperLatStep, double zipperLonStep, T belowLeftTile, T aboveLeftTile, T belowRightTile, T aboveRightTile, boolean isSameStepLat, boolean isSameStepLon) {
        int zipperLatRows = 4;
        int zipperLonCols = 4;
        double[][] elevations = new double[4][4];
        if (isSameStepLat && isSameStepLon) {
            elevations[0][0] = belowLeftTile.getElevationAtIndices(belowLeftTile.getLatitudeRows() - 2, belowLeftTile.getLongitudeColumns() - 2);
            elevations[0][1] = belowLeftTile.getElevationAtIndices(belowLeftTile.getLatitudeRows() - 2, belowLeftTile.getLongitudeColumns() - 1);
            elevations[0][2] = belowRightTile.getElevationAtIndices(belowRightTile.getLatitudeRows() - 2, 0);
            elevations[0][3] = belowRightTile.getElevationAtIndices(belowRightTile.getLatitudeRows() - 2, 1);
            elevations[1][0] = belowLeftTile.getElevationAtIndices(belowLeftTile.getLatitudeRows() - 1, belowLeftTile.getLongitudeColumns() - 2);
            elevations[1][1] = belowLeftTile.getElevationAtIndices(belowLeftTile.getLatitudeRows() - 1, belowLeftTile.getLongitudeColumns() - 1);
            elevations[1][2] = belowRightTile.getElevationAtIndices(belowRightTile.getLatitudeRows() - 1, 0);
            elevations[1][3] = belowRightTile.getElevationAtIndices(belowRightTile.getLatitudeRows() - 1, 1);
            elevations[2][0] = aboveLeftTile.getElevationAtIndices(0, aboveLeftTile.getLongitudeColumns() - 2);
            elevations[2][1] = aboveLeftTile.getElevationAtIndices(0, aboveLeftTile.getLongitudeColumns() - 1);
            elevations[2][2] = aboveRightTile.getElevationAtIndices(0, 0);
            elevations[2][3] = aboveRightTile.getElevationAtIndices(0, 1);
            elevations[3][0] = aboveLeftTile.getElevationAtIndices(1, aboveLeftTile.getLongitudeColumns() - 2);
            elevations[3][1] = aboveLeftTile.getElevationAtIndices(1, aboveLeftTile.getLongitudeColumns() - 1);
            elevations[3][2] = aboveRightTile.getElevationAtIndices(1, 0);
            elevations[3][3] = aboveRightTile.getElevationAtIndices(1, 1);
        } else {
            double zipperLat0 = zipperLatMin + 0.1 * zipperLatStep;
            double zipperLat1 = zipperLat0 + zipperLatStep;
            double zipperLat2 = zipperLat1 + zipperLatStep;
            double zipperLat3 = zipperLat2 + zipperLatStep;
            double zipperLon0 = zipperLonMin + 0.1 * zipperLonStep;
            double zipperLon1 = zipperLon0 + zipperLonStep;
            double zipperLon2 = zipperLon1 + zipperLonStep;
            double zipperLon3 = zipperLon2 + zipperLonStep;
            int belowLeftLatitudeIndex0 = this.computeLatitudeIndex(zipperLat0, belowLeftTile.getMinimumLatitude(), belowLeftTile.getLatitudeStep(), belowLeftTile.getLatitudeRows());
            int belowLeftLatitudeIndex1 = this.computeLatitudeIndex(zipperLat1, belowLeftTile.getMinimumLatitude(), belowLeftTile.getLatitudeStep(), belowLeftTile.getLatitudeRows());
            int belowRightLatitudeIndex0 = this.computeLatitudeIndex(zipperLat0, belowRightTile.getMinimumLatitude(), belowRightTile.getLatitudeStep(), belowRightTile.getLatitudeRows());
            int belowRightLatitudeIndex1 = this.computeLatitudeIndex(zipperLat1, belowRightTile.getMinimumLatitude(), belowRightTile.getLatitudeStep(), belowRightTile.getLatitudeRows());
            int aboveLeftLatitudeIndex2 = this.computeLatitudeIndex(zipperLat2, aboveLeftTile.getMinimumLatitude(), aboveLeftTile.getLatitudeStep(), aboveLeftTile.getLatitudeRows());
            int aboveLeftLatitudeIndex3 = this.computeLatitudeIndex(zipperLat3, aboveLeftTile.getMinimumLatitude(), aboveLeftTile.getLatitudeStep(), aboveLeftTile.getLatitudeRows());
            int aboveRightLatitudeIndex2 = this.computeLatitudeIndex(zipperLat2, aboveRightTile.getMinimumLatitude(), aboveRightTile.getLatitudeStep(), aboveRightTile.getLatitudeRows());
            int aboveRightLatitudeIndex3 = this.computeLatitudeIndex(zipperLat3, aboveRightTile.getMinimumLatitude(), aboveRightTile.getLatitudeStep(), aboveRightTile.getLatitudeRows());
            int belowLeftLongitudeIndex0 = this.computeLongitudeIndex(zipperLon0, belowLeftTile.getMinimumLongitude(), belowLeftTile.getLongitudeStep(), belowLeftTile.getLongitudeColumns());
            int belowLeftLongitudeIndex1 = this.computeLongitudeIndex(zipperLon1, belowLeftTile.getMinimumLongitude(), belowLeftTile.getLongitudeStep(), belowLeftTile.getLongitudeColumns());
            int belowRightLongitudeIndex2 = this.computeLongitudeIndex(zipperLon2, belowRightTile.getMinimumLongitude(), belowRightTile.getLongitudeStep(), belowRightTile.getLongitudeColumns());
            int belowRightLongitudeIndex3 = this.computeLongitudeIndex(zipperLon3, belowRightTile.getMinimumLongitude(), belowRightTile.getLongitudeStep(), belowRightTile.getLongitudeColumns());
            int aboveLeftLongitudeIndex0 = this.computeLongitudeIndex(zipperLon0, aboveLeftTile.getMinimumLongitude(), aboveLeftTile.getLongitudeStep(), aboveLeftTile.getLongitudeColumns());
            int aboveLeftLongitudeIndex1 = this.computeLongitudeIndex(zipperLon1, aboveLeftTile.getMinimumLongitude(), aboveLeftTile.getLongitudeStep(), aboveLeftTile.getLongitudeColumns());
            int aboveRightLongitudeIndex2 = this.computeLongitudeIndex(zipperLon2, aboveRightTile.getMinimumLongitude(), aboveRightTile.getLongitudeStep(), aboveRightTile.getLongitudeColumns());
            int aboveRightLongitudeIndex3 = this.computeLongitudeIndex(zipperLon3, aboveRightTile.getMinimumLongitude(), aboveRightTile.getLongitudeStep(), aboveRightTile.getLongitudeColumns());
            elevations[0][0] = belowLeftTile.getElevationAtIndices(belowLeftLatitudeIndex0, belowLeftLongitudeIndex0);
            elevations[0][1] = belowLeftTile.getElevationAtIndices(belowLeftLatitudeIndex0, belowLeftLongitudeIndex1);
            elevations[0][2] = belowRightTile.getElevationAtIndices(belowRightLatitudeIndex0, belowRightLongitudeIndex2);
            elevations[0][3] = belowRightTile.getElevationAtIndices(belowRightLatitudeIndex0, belowRightLongitudeIndex3);
            elevations[1][0] = belowLeftTile.getElevationAtIndices(belowLeftLatitudeIndex1, belowLeftLongitudeIndex0);
            elevations[1][1] = belowLeftTile.getElevationAtIndices(belowLeftLatitudeIndex1, belowLeftLongitudeIndex1);
            elevations[1][2] = belowRightTile.getElevationAtIndices(belowRightLatitudeIndex1, belowRightLongitudeIndex2);
            elevations[1][3] = belowRightTile.getElevationAtIndices(belowRightLatitudeIndex1, belowRightLongitudeIndex3);
            elevations[2][0] = aboveLeftTile.getElevationAtIndices(aboveLeftLatitudeIndex2, aboveLeftLongitudeIndex0);
            elevations[2][1] = aboveLeftTile.getElevationAtIndices(aboveLeftLatitudeIndex2, aboveLeftLongitudeIndex1);
            elevations[2][2] = aboveRightTile.getElevationAtIndices(aboveRightLatitudeIndex2, aboveRightLongitudeIndex2);
            elevations[2][3] = aboveRightTile.getElevationAtIndices(aboveRightLatitudeIndex2, aboveRightLongitudeIndex3);
            elevations[3][0] = aboveLeftTile.getElevationAtIndices(aboveLeftLatitudeIndex3, aboveLeftLongitudeIndex0);
            elevations[3][1] = aboveLeftTile.getElevationAtIndices(aboveLeftLatitudeIndex3, aboveLeftLongitudeIndex1);
            elevations[3][2] = aboveRightTile.getElevationAtIndices(aboveRightLatitudeIndex3, aboveRightLongitudeIndex2);
            elevations[3][3] = aboveRightTile.getElevationAtIndices(aboveRightLatitudeIndex3, aboveRightLongitudeIndex3);
        }
        T cornerZipperTile = this.initializeZipperTile(zipperLatMin, zipperLonMin, zipperLatStep, zipperLonStep, 4, 4, elevations);
        return cornerZipperTile;
    }

    private T createIntercardinalTile(EarthHemisphere latitudeHemisphere, double latitudeMin, double latitudeStep, int latitudeRows, EarthHemisphere longitudeHemisphere, double longitudeMin, double longitudeStep, int longitudeCols) {
        int latHemisphere;
        int lonHemisphere;
        switch (longitudeHemisphere) {
            case EAST: {
                lonHemisphere = 1;
                break;
            }
            case WEST: {
                lonHemisphere = -1;
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        switch (latitudeHemisphere) {
            case NORTH: {
                latHemisphere = 1;
                break;
            }
            case SOUTH: {
                latHemisphere = -1;
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        double latToGetIntercardinalTile = latitudeMin + (double)(latHemisphere * latitudeRows) * latitudeStep;
        double lonToGetIntercardinalTile = longitudeMin + (double)(lonHemisphere * longitudeCols) * longitudeStep;
        T intercardinalTile = this.createTile(latToGetIntercardinalTile, lonToGetIntercardinalTile);
        return intercardinalTile;
    }

    private GeodeticPoint computeCornerZipperOrigin(double zipperLatStep, double zipperLonStep, EarthHemisphere latitudeHemisphere, double currentTileLatMin, double currentTileLatStep, int currentTileLatRows, EarthHemisphere longitudeHemisphere, double currentTileLonMin, double currentTileLonStep, int currentTileLonCols) {
        double zipperLonMin;
        double zipperLatMin;
        block0 : switch (latitudeHemisphere) {
            case NORTH: {
                switch (longitudeHemisphere) {
                    case WEST: {
                        zipperLatMin = currentTileLatMin + (double)currentTileLatRows * currentTileLatStep - 2.0 * zipperLatStep;
                        zipperLonMin = currentTileLonMin - 2.0 * zipperLonStep;
                        break block0;
                    }
                    case EAST: {
                        zipperLatMin = currentTileLatMin + (double)currentTileLatRows * currentTileLatStep - 2.0 * zipperLatStep;
                        zipperLonMin = currentTileLonMin + (double)currentTileLonCols * currentTileLonStep - 2.0 * zipperLonStep;
                        break block0;
                    }
                }
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
            case SOUTH: {
                switch (longitudeHemisphere) {
                    case WEST: {
                        zipperLatMin = currentTileLatMin - 2.0 * zipperLatStep;
                        zipperLonMin = currentTileLonMin - 2.0 * zipperLonStep;
                        break block0;
                    }
                    case EAST: {
                        zipperLatMin = currentTileLatMin - 2.0 * zipperLatStep;
                        zipperLonMin = currentTileLonMin + (double)currentTileLonCols * currentTileLonStep - 2.0 * zipperLonStep;
                        break block0;
                    }
                }
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        return new GeodeticPoint(zipperLatMin, zipperLonMin, 0.0);
    }

    private T createNorthOrSouthTile(EarthHemisphere latitudeHemisphere, double longitude, double latitudeMin, double latitudeStep, int latitudeRows) {
        int hemisphere;
        switch (latitudeHemisphere) {
            case NORTH: {
                hemisphere = 1;
                break;
            }
            case SOUTH: {
                hemisphere = -1;
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        double latToGetNewTile = latitudeMin + (double)(hemisphere * latitudeRows) * latitudeStep;
        return this.createTile(latToGetNewTile, longitude);
    }

    private T createEastOrWestTile(EarthHemisphere longitudeHemisphere, double latitude, double longitudeMin, double longitudeStep, int longitudeCols) {
        int hemisphere;
        switch (longitudeHemisphere) {
            case EAST: {
                hemisphere = 1;
                break;
            }
            case WEST: {
                hemisphere = -1;
                break;
            }
            default: {
                throw new RuggedException(RuggedMessages.INTERNAL_ERROR, new Object[0]);
            }
        }
        double lonToGetNewTile = longitudeMin + (double)(hemisphere * longitudeCols) * longitudeStep;
        return this.createTile(latitude, lonToGetNewTile);
    }
}

