/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.rugged.linesensor;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;
import org.hipparchus.analysis.UnivariateFunction;
import org.hipparchus.analysis.solvers.BracketingNthOrderBrentSolver;
import org.hipparchus.exception.MathIllegalArgumentException;
import org.hipparchus.exception.MathIllegalStateException;
import org.hipparchus.geometry.Point;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.linear.Array2DRowRealMatrix;
import org.hipparchus.linear.ArrayRealVector;
import org.hipparchus.linear.DecompositionSolver;
import org.hipparchus.linear.MatrixUtils;
import org.hipparchus.linear.QRDecomposition;
import org.hipparchus.linear.RealMatrix;
import org.hipparchus.linear.RealVector;
import org.hipparchus.linear.SingularValueDecomposition;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.Precision;
import org.orekit.frames.Transform;
import org.orekit.rugged.errors.RuggedException;
import org.orekit.rugged.errors.RuggedInternalError;
import org.orekit.rugged.linesensor.LineSensor;
import org.orekit.rugged.utils.SpacecraftToObservedBody;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;

public class SensorMeanPlaneCrossing {
    private static final int CACHED_RESULTS = 6;
    private final SpacecraftToObservedBody scToBody;
    private final double midLine;
    private final Transform midBodyToInert;
    private final Transform midScToInert;
    private final int minLine;
    private final int maxLine;
    private final boolean lightTimeCorrection;
    private final boolean aberrationOfLightCorrection;
    private final LineSensor sensor;
    private final Vector3D meanPlaneNormal;
    private final int maxEval;
    private final double accuracy;
    private final List<CrossingResult> cachedResults;

    public SensorMeanPlaneCrossing(LineSensor sensor, SpacecraftToObservedBody scToBody, int minLine, int maxLine, boolean lightTimeCorrection, boolean aberrationOfLightCorrection, int maxEval, double accuracy) {
        this(sensor, scToBody, minLine, maxLine, lightTimeCorrection, aberrationOfLightCorrection, maxEval, accuracy, SensorMeanPlaneCrossing.computeMeanPlaneNormal(sensor, minLine, maxLine), Stream.empty());
    }

    public SensorMeanPlaneCrossing(LineSensor sensor, SpacecraftToObservedBody scToBody, int minLine, int maxLine, boolean lightTimeCorrection, boolean aberrationOfLightCorrection, int maxEval, double accuracy, Vector3D meanPlaneNormal, Stream<CrossingResult> cachedResults) {
        this.sensor = sensor;
        this.minLine = minLine;
        this.maxLine = maxLine;
        this.lightTimeCorrection = lightTimeCorrection;
        this.aberrationOfLightCorrection = aberrationOfLightCorrection;
        this.maxEval = maxEval;
        this.accuracy = accuracy;
        this.scToBody = scToBody;
        this.midLine = 0.5 * (double)(minLine + maxLine);
        AbsoluteDate midDate = sensor.getDate(this.midLine);
        this.midBodyToInert = scToBody.getBodyToInertial(midDate);
        this.midScToInert = scToBody.getScToInertial(midDate);
        this.meanPlaneNormal = meanPlaneNormal;
        this.cachedResults = new ArrayList<CrossingResult>(6);
        cachedResults.forEach(crossingResult -> {
            if (crossingResult != null && this.cachedResults.size() < 6) {
                this.cachedResults.add((CrossingResult)crossingResult);
            }
        });
    }

    private static Vector3D computeMeanPlaneNormal(LineSensor sensor, int minLine, int maxLine) {
        Vector3D last;
        Vector3D first;
        AbsoluteDate midDate = sensor.getDate(0.5 * (double)(minLine + maxLine));
        RealMatrix matrix = MatrixUtils.createRealMatrix((int)3, (int)(2 * sensor.getNbPixels()));
        for (int i = 0; i < sensor.getNbPixels(); ++i) {
            Vector3D l = sensor.getLOS(midDate, i);
            matrix.setEntry(0, 2 * i, l.getX());
            matrix.setEntry(1, 2 * i, l.getY());
            matrix.setEntry(2, 2 * i, l.getZ());
            matrix.setEntry(0, 2 * i + 1, -l.getX());
            matrix.setEntry(1, 2 * i + 1, -l.getY());
            matrix.setEntry(2, 2 * i + 1, -l.getZ());
        }
        SingularValueDecomposition svd = new SingularValueDecomposition(matrix);
        Vector3D singularVector = (Vector3D)new Vector3D(svd.getU().getColumn(2)).normalize();
        if (Vector3D.dotProduct((Vector3D)singularVector, (Vector3D)Vector3D.crossProduct((Vector3D)(first = sensor.getLOS(midDate, 0)), (Vector3D)(last = sensor.getLOS(midDate, sensor.getNbPixels() - 1)))) >= 0.0) {
            return singularVector;
        }
        return singularVector.negate();
    }

    public LineSensor getSensor() {
        return this.sensor;
    }

    public SpacecraftToObservedBody getScToBody() {
        return this.scToBody;
    }

    public int getMinLine() {
        return this.minLine;
    }

    public int getMaxLine() {
        return this.maxLine;
    }

    public int getMaxEval() {
        return this.maxEval;
    }

    public double getAccuracy() {
        return this.accuracy;
    }

    public Vector3D getMeanPlaneNormal() {
        return this.meanPlaneNormal;
    }

    public Stream<CrossingResult> getCachedResults() {
        return this.cachedResults.stream();
    }

    public CrossingResult find(Vector3D target) {
        double guessedCrossingLine;
        double crossingLine = this.midLine;
        Transform bodyToInert = this.midBodyToInert;
        Transform scToInert = this.midScToInert;
        if (this.cachedResults.size() >= 4 && (guessedCrossingLine = this.guessStartLine(target)) >= (double)this.minLine && guessedCrossingLine <= (double)this.maxLine) {
            crossingLine = guessedCrossingLine;
            AbsoluteDate date = this.sensor.getDate(crossingLine);
            bodyToInert = this.scToBody.getBodyToInertial(date);
            scToInert = this.scToBody.getScToInertial(date);
        }
        PVCoordinates targetPV = new PVCoordinates(target, Vector3D.ZERO);
        double[] crossingLineHistory = new double[this.maxEval];
        double[] betaHistory = new double[this.maxEval];
        double[] betaDerHistory = new double[this.maxEval];
        boolean atMin = false;
        boolean atMax = false;
        for (int i = 0; i < this.maxEval; ++i) {
            double deltaL;
            crossingLineHistory[i] = crossingLine;
            Vector3D[] targetDirection = this.evaluateLine(crossingLine, targetPV, bodyToInert, scToInert);
            betaHistory[i] = FastMath.acos((double)Vector3D.dotProduct((Vector3D)targetDirection[0], (Vector3D)this.meanPlaneNormal));
            double s = Vector3D.dotProduct((Vector3D)targetDirection[1], (Vector3D)this.meanPlaneNormal);
            betaDerHistory[i] = -s / FastMath.sqrt((double)(1.0 - s * s));
            if (i == 0) {
                deltaL = (1.5707963267948966 - betaHistory[i]) / betaDerHistory[i];
                crossingLine += deltaL;
            } else {
                double a0 = betaHistory[i - 1] - 1.5707963267948966;
                double l0 = crossingLineHistory[i - 1];
                double d0 = betaDerHistory[i - 1];
                double a1 = betaHistory[i] - 1.5707963267948966;
                double l1 = crossingLineHistory[i];
                double d1 = betaDerHistory[i];
                double a1Ma0 = a1 - a0;
                crossingLine = ((l0 * (a1 - 3.0 * a0) - a0 * a1Ma0 / d0) * a1 * a1 + (l1 * (3.0 * a1 - a0) - a1 * a1Ma0 / d1) * a0 * a0) / (a1Ma0 * a1Ma0 * a1Ma0);
                deltaL = crossingLine - l1;
            }
            if (FastMath.abs((double)deltaL) <= this.accuracy) {
                CrossingResult crossingResult = new CrossingResult(this.sensor.getDate(crossingLine), crossingLine, target, targetDirection[0], targetDirection[1]);
                boolean isNew = true;
                for (CrossingResult existing : this.cachedResults) {
                    isNew = isNew && FastMath.abs((double)(crossingLine - existing.crossingLine)) > this.accuracy;
                }
                if (isNew) {
                    if (this.cachedResults.size() >= 6) {
                        this.cachedResults.remove(this.cachedResults.size() - 1);
                    }
                    this.cachedResults.add(0, crossingResult);
                }
                return crossingResult;
            }
            for (int j = 0; j < i; ++j) {
                if (!(FastMath.abs((double)(crossingLine - crossingLineHistory[j])) <= 1.0)) continue;
                CrossingResult slowResult = this.slowFind(targetPV, crossingLine);
                if (slowResult == null) {
                    return null;
                }
                if (this.cachedResults.size() >= 6) {
                    this.cachedResults.remove(this.cachedResults.size() - 1);
                }
                this.cachedResults.add(0, slowResult);
                return this.cachedResults.get(0);
            }
            if (crossingLine < (double)this.minLine) {
                if (atMin) {
                    return null;
                }
                atMin = true;
                crossingLine = this.minLine;
            } else if (crossingLine > (double)this.maxLine) {
                if (atMax) {
                    return null;
                }
                atMax = true;
                crossingLine = this.maxLine;
            } else {
                atMin = false;
                atMax = false;
            }
            AbsoluteDate date = this.sensor.getDate(crossingLine);
            bodyToInert = this.scToBody.getBodyToInertial(date);
            scToInert = this.scToBody.getScToInertial(date);
        }
        return null;
    }

    private double guessStartLine(Vector3D target) {
        try {
            int n = this.cachedResults.size();
            Array2DRowRealMatrix m = new Array2DRowRealMatrix(n, 4);
            ArrayRealVector v = new ArrayRealVector(n);
            for (int i = 0; i < n; ++i) {
                CrossingResult crossingResult = this.cachedResults.get(i);
                m.setEntry(i, 0, crossingResult.getTarget().getX());
                m.setEntry(i, 1, crossingResult.getTarget().getY());
                m.setEntry(i, 2, crossingResult.getTarget().getZ());
                m.setEntry(i, 3, 1.0);
                v.setEntry(i, crossingResult.getLine());
            }
            DecompositionSolver solver = new QRDecomposition((RealMatrix)m, Precision.SAFE_MIN).getSolver();
            RealVector coefficients = solver.solve((RealVector)v);
            return target.getX() * coefficients.getEntry(0) + target.getY() * coefficients.getEntry(1) + target.getZ() * coefficients.getEntry(2) + coefficients.getEntry(3);
        }
        catch (MathIllegalStateException sme) {
            return Double.NaN;
        }
    }

    private CrossingResult slowFind(final PVCoordinates targetPV, double initialGuess) {
        try {
            double startValue = initialGuess < (double)this.minLine || initialGuess > (double)this.maxLine ? this.midLine : initialGuess;
            BracketingNthOrderBrentSolver solver = new BracketingNthOrderBrentSolver(this.accuracy, 5);
            double crossingLine = solver.solve(this.maxEval, new UnivariateFunction(){

                public double value(double x) {
                    try {
                        AbsoluteDate date = SensorMeanPlaneCrossing.this.sensor.getDate(x);
                        Vector3D[] targetDirection = SensorMeanPlaneCrossing.this.evaluateLine(x, targetPV, SensorMeanPlaneCrossing.this.scToBody.getBodyToInertial(date), SensorMeanPlaneCrossing.this.scToBody.getScToInertial(date));
                        return 1.5707963267948966 - FastMath.acos((double)Vector3D.dotProduct((Vector3D)targetDirection[0], (Vector3D)SensorMeanPlaneCrossing.this.meanPlaneNormal));
                    }
                    catch (RuggedException re) {
                        throw new RuggedInternalError(re);
                    }
                }
            }, (double)this.minLine, (double)this.maxLine, startValue);
            AbsoluteDate date = this.sensor.getDate(crossingLine);
            Vector3D[] targetDirection = this.evaluateLine(crossingLine, targetPV, this.scToBody.getBodyToInertial(date), this.scToBody.getScToInertial(date));
            return new CrossingResult(this.sensor.getDate(crossingLine), crossingLine, targetPV.getPosition(), targetDirection[0], targetDirection[1]);
        }
        catch (MathIllegalArgumentException nbe) {
            return null;
        }
    }

    private Vector3D[] evaluateLine(double lineNumber, PVCoordinates targetPV, Transform bodyToInert, Transform scToInert) {
        Vector3D obsLInertDot;
        Vector3D obsLInert;
        PVCoordinates targetInert;
        PVCoordinates refInert = scToInert.transformPVCoordinates(new PVCoordinates(this.sensor.getPosition(), Vector3D.ZERO));
        if (this.lightTimeCorrection) {
            Vector3D iT = bodyToInert.transformPosition(targetPV.getPosition());
            double deltaT = refInert.getPosition().distance((Point)iT) / 2.99792458E8;
            targetInert = bodyToInert.shiftedBy(-deltaT).transformPVCoordinates(targetPV);
        } else {
            targetInert = bodyToInert.transformPVCoordinates(targetPV);
        }
        PVCoordinates lInert = new PVCoordinates(refInert, targetInert);
        Transform inertToSc = scToInert.getInverse();
        if (this.aberrationOfLightCorrection) {
            PVCoordinates spacecraftPV = scToInert.transformPVCoordinates(PVCoordinates.ZERO);
            Vector3D l = (Vector3D)lInert.getPosition().normalize();
            Vector3D lDot = this.normalizedDot(lInert.getPosition(), lInert.getVelocity());
            Vector3D kObs = new Vector3D(2.99792458E8, l, 1.0, spacecraftPV.getVelocity());
            obsLInert = (Vector3D)kObs.normalize();
            obsLInertDot = this.normalizedDot(kObs, new Vector3D(2.99792458E8, lDot));
        } else {
            obsLInert = (Vector3D)lInert.getPosition().normalize();
            obsLInertDot = this.normalizedDot(lInert.getPosition(), lInert.getVelocity());
        }
        Vector3D dir = inertToSc.transformVector(obsLInert);
        Vector3D dirDot = new Vector3D(1.0, inertToSc.transformVector(obsLInertDot), -1.0, Vector3D.crossProduct((Vector3D)inertToSc.getRotationRate(), (Vector3D)dir));
        double rate = this.sensor.getRate(lineNumber);
        return new Vector3D[]{dir, dirDot.scalarMultiply(1.0 / rate)};
    }

    private Vector3D normalizedDot(Vector3D u, Vector3D uDot) {
        double n = u.getNorm();
        return new Vector3D(1.0 / n, uDot, -Vector3D.dotProduct((Vector3D)u, (Vector3D)uDot) / (n * n * n), u);
    }

    public static class CrossingResult {
        private final AbsoluteDate crossingDate;
        private final double crossingLine;
        private final Vector3D target;
        private final Vector3D targetDirection;
        private final Vector3D targetDirectionDerivative;

        public CrossingResult(AbsoluteDate crossingDate, double crossingLine, Vector3D target, Vector3D targetDirection, Vector3D targetDirectionDerivative) {
            this.crossingDate = crossingDate;
            this.crossingLine = crossingLine;
            this.target = target;
            this.targetDirection = targetDirection;
            this.targetDirectionDerivative = targetDirectionDerivative;
        }

        public AbsoluteDate getDate() {
            return this.crossingDate;
        }

        public double getLine() {
            return this.crossingLine;
        }

        public Vector3D getTarget() {
            return this.target;
        }

        public Vector3D getTargetDirection() {
            return this.targetDirection;
        }

        public Vector3D getTargetDirectionDerivative() {
            return this.targetDirectionDerivative;
        }
    }
}

