/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.util.List;
import java.util.stream.Stream;
import org.hipparchus.analysis.interpolation.HermiteInterpolator;
import org.hipparchus.util.MathUtils;
import org.orekit.errors.OrekitInternalError;
import org.orekit.frames.Frame;
import org.orekit.orbits.AbstractOrbitInterpolator;
import org.orekit.orbits.CartesianOrbit;
import org.orekit.orbits.CircularOrbit;
import org.orekit.orbits.EquinoctialOrbit;
import org.orekit.orbits.KeplerianOrbit;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngleType;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.AbstractTimeInterpolator;
import org.orekit.utils.CartesianDerivativesFilter;
import org.orekit.utils.TimeStampedPVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinatesHermiteInterpolator;

public class OrbitHermiteInterpolator
extends AbstractOrbitInterpolator {
    private final CartesianDerivativesFilter pvaFilter;

    public OrbitHermiteInterpolator(Frame outputInertialFrame) {
        this(2, outputInertialFrame);
    }

    public OrbitHermiteInterpolator(int interpolationPoints, Frame outputInertialFrame) {
        this(interpolationPoints, outputInertialFrame, CartesianDerivativesFilter.USE_PVA);
    }

    public OrbitHermiteInterpolator(int interpolationPoints, Frame outputInertialFrame, CartesianDerivativesFilter pvaFilter) {
        this(interpolationPoints, 0.001, outputInertialFrame, pvaFilter);
    }

    public OrbitHermiteInterpolator(int interpolationPoints, double extrapolationThreshold, Frame outputInertialFrame, CartesianDerivativesFilter pvaFilter) {
        super(interpolationPoints, extrapolationThreshold, outputInertialFrame);
        this.pvaFilter = pvaFilter;
    }

    public CartesianDerivativesFilter getPVAFilter() {
        return this.pvaFilter;
    }

    @Override
    protected Orbit interpolate(AbstractTimeInterpolator.InterpolationData interpolationData) {
        List<Orbit> sample = interpolationData.getNeighborList();
        OrbitType orbitType = ((Orbit)sample.get(0)).getType();
        if (orbitType == OrbitType.CARTESIAN) {
            return this.interpolateCartesian(interpolationData.getInterpolationDate(), sample);
        }
        return this.interpolateCommon(interpolationData.getInterpolationDate(), sample, orbitType);
    }

    private CartesianOrbit interpolateCartesian(AbsoluteDate interpolationDate, List<Orbit> sample) {
        TimeStampedPVCoordinatesHermiteInterpolator interpolator = new TimeStampedPVCoordinatesHermiteInterpolator(this.getNbInterpolationPoints(), this.getExtrapolationThreshold(), this.pvaFilter);
        Stream sampleStream = sample.stream();
        Stream<TimeStampedPVCoordinates> sampleTimeStampedPV = sampleStream.map(Orbit::getPVCoordinates);
        TimeStampedPVCoordinates interpolated = interpolator.interpolate(interpolationDate, sampleTimeStampedPV);
        double mu = sample.get(0).getMu();
        return new CartesianOrbit(interpolated, this.getOutputInertialFrame(), interpolationDate, mu);
    }

    private Orbit interpolateCommon(AbsoluteDate interpolationDate, List<Orbit> orbits, OrbitType orbitType) {
        boolean useDerivatives = true;
        for (Orbit orbit : orbits) {
            useDerivatives = useDerivatives && orbit.hasDerivatives();
        }
        double mu = orbits.get(0).getMu();
        switch (orbitType) {
            case CIRCULAR: {
                double[][] interpolated = this.interpolateCircular(interpolationDate, orbits, useDerivatives);
                return new CircularOrbit(interpolated[0][0], interpolated[0][1], interpolated[0][2], interpolated[0][3], interpolated[0][4], interpolated[0][5], interpolated[1][0], interpolated[1][1], interpolated[1][2], interpolated[1][3], interpolated[1][4], interpolated[1][5], PositionAngleType.MEAN, this.getOutputInertialFrame(), interpolationDate, mu);
            }
            case KEPLERIAN: {
                double[][] interpolated = this.interpolateKeplerian(interpolationDate, orbits, useDerivatives);
                return new KeplerianOrbit(interpolated[0][0], interpolated[0][1], interpolated[0][2], interpolated[0][3], interpolated[0][4], interpolated[0][5], interpolated[1][0], interpolated[1][1], interpolated[1][2], interpolated[1][3], interpolated[1][4], interpolated[1][5], PositionAngleType.MEAN, this.getOutputInertialFrame(), interpolationDate, mu);
            }
            case EQUINOCTIAL: {
                double[][] interpolated = this.interpolateEquinoctial(interpolationDate, orbits, useDerivatives);
                return new EquinoctialOrbit(interpolated[0][0], interpolated[0][1], interpolated[0][2], interpolated[0][3], interpolated[0][4], interpolated[0][5], interpolated[1][0], interpolated[1][1], interpolated[1][2], interpolated[1][3], interpolated[1][4], interpolated[1][5], PositionAngleType.MEAN, this.getOutputInertialFrame(), interpolationDate, mu);
            }
        }
        throw new OrekitInternalError(null);
    }

    private double[][] interpolateCircular(AbsoluteDate interpolationDate, List<Orbit> orbits, boolean useDerivatives) {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousRAAN = Double.NaN;
        double previousAlphaM = Double.NaN;
        for (Orbit orbit : orbits) {
            double continuousAlphaM;
            double continuousRAAN;
            CircularOrbit circ = (CircularOrbit)OrbitType.CIRCULAR.convertType(orbit);
            if (previousDate == null) {
                continuousRAAN = circ.getRightAscensionOfAscendingNode();
                continuousAlphaM = circ.getAlphaM();
            } else {
                double dt = circ.getDate().durationFrom(previousDate);
                double keplerAM = previousAlphaM + circ.getKeplerianMeanMotion() * dt;
                continuousRAAN = MathUtils.normalizeAngle((double)circ.getRightAscensionOfAscendingNode(), (double)previousRAAN);
                continuousAlphaM = MathUtils.normalizeAngle((double)circ.getAlphaM(), (double)keplerAM);
            }
            previousDate = circ.getDate();
            previousRAAN = continuousRAAN;
            previousAlphaM = continuousAlphaM;
            if (useDerivatives) {
                interpolator.addSamplePoint(circ.getDate().durationFrom(interpolationDate), (double[][])new double[][]{{circ.getA(), circ.getCircularEx(), circ.getCircularEy(), circ.getI(), continuousRAAN, continuousAlphaM}, {circ.getADot(), circ.getCircularExDot(), circ.getCircularEyDot(), circ.getIDot(), circ.getRightAscensionOfAscendingNodeDot(), circ.getAlphaMDot()}});
                continue;
            }
            interpolator.addSamplePoint(circ.getDate().durationFrom(interpolationDate), (double[][])new double[][]{{circ.getA(), circ.getCircularEx(), circ.getCircularEy(), circ.getI(), continuousRAAN, continuousAlphaM}});
        }
        return interpolator.derivatives(0.0, 1);
    }

    private double[][] interpolateKeplerian(AbsoluteDate interpolationDate, List<Orbit> orbits, boolean useDerivatives) {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousPA = Double.NaN;
        double previousRAAN = Double.NaN;
        double previousM = Double.NaN;
        for (Orbit orbit : orbits) {
            double continuousM;
            double continuousRAAN;
            double continuousPA;
            KeplerianOrbit kep = (KeplerianOrbit)OrbitType.KEPLERIAN.convertType(orbit);
            if (previousDate == null) {
                continuousPA = kep.getPerigeeArgument();
                continuousRAAN = kep.getRightAscensionOfAscendingNode();
                continuousM = kep.getMeanAnomaly();
            } else {
                double dt = kep.getDate().durationFrom(previousDate);
                double keplerM = previousM + kep.getKeplerianMeanMotion() * dt;
                continuousPA = MathUtils.normalizeAngle((double)kep.getPerigeeArgument(), (double)previousPA);
                continuousRAAN = MathUtils.normalizeAngle((double)kep.getRightAscensionOfAscendingNode(), (double)previousRAAN);
                continuousM = MathUtils.normalizeAngle((double)kep.getMeanAnomaly(), (double)keplerM);
            }
            previousDate = kep.getDate();
            previousPA = continuousPA;
            previousRAAN = continuousRAAN;
            previousM = continuousM;
            if (useDerivatives) {
                interpolator.addSamplePoint(kep.getDate().durationFrom(interpolationDate), (double[][])new double[][]{{kep.getA(), kep.getE(), kep.getI(), continuousPA, continuousRAAN, continuousM}, {kep.getADot(), kep.getEDot(), kep.getIDot(), kep.getPerigeeArgumentDot(), kep.getRightAscensionOfAscendingNodeDot(), kep.getMeanAnomalyDot()}});
                continue;
            }
            interpolator.addSamplePoint(kep.getDate().durationFrom(interpolationDate), (double[][])new double[][]{{kep.getA(), kep.getE(), kep.getI(), continuousPA, continuousRAAN, continuousM}});
        }
        return interpolator.derivatives(0.0, 1);
    }

    private double[][] interpolateEquinoctial(AbsoluteDate interpolationDate, List<Orbit> orbits, boolean useDerivatives) {
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousLm = Double.NaN;
        for (Orbit orbit : orbits) {
            double continuousLm;
            EquinoctialOrbit equi = (EquinoctialOrbit)OrbitType.EQUINOCTIAL.convertType(orbit);
            if (previousDate == null) {
                continuousLm = equi.getLM();
            } else {
                double dt = equi.getDate().durationFrom(previousDate);
                double keplerLm = previousLm + equi.getKeplerianMeanMotion() * dt;
                continuousLm = MathUtils.normalizeAngle((double)equi.getLM(), (double)keplerLm);
            }
            previousDate = equi.getDate();
            previousLm = continuousLm;
            if (useDerivatives) {
                interpolator.addSamplePoint(equi.getDate().durationFrom(interpolationDate), (double[][])new double[][]{{equi.getA(), equi.getEquinoctialEx(), equi.getEquinoctialEy(), equi.getHx(), equi.getHy(), continuousLm}, {equi.getADot(), equi.getEquinoctialExDot(), equi.getEquinoctialEyDot(), equi.getHxDot(), equi.getHyDot(), equi.getLMDot()}});
                continue;
            }
            interpolator.addSamplePoint(equi.getDate().durationFrom(interpolationDate), (double[][])new double[][]{{equi.getA(), equi.getEquinoctialEx(), equi.getEquinoctialEy(), equi.getHx(), equi.getHy(), continuousLm}});
        }
        return interpolator.derivatives(0.0, 1);
    }
}

