/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import org.hipparchus.analysis.differentiation.UnivariateDerivative1;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.Vector;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.SinCos;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitInternalError;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.FieldKeplerianAnomalyUtility;
import org.orekit.orbits.KeplerianAnomalyUtility;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngleBased;
import org.orekit.orbits.PositionAngleType;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class KeplerianOrbit
extends Orbit
implements PositionAngleBased {
    private static final long serialVersionUID = 20231217L;
    private static final String ECCENTRICITY = "eccentricity";
    private final double a;
    private final double e;
    private final double i;
    private final double pa;
    private final double raan;
    private final double cachedAnomaly;
    private final double aDot;
    private final double eDot;
    private final double iDot;
    private final double paDot;
    private final double raanDot;
    private final double cachedAnomalyDot;
    private final PositionAngleType cachedPositionAngleType;
    private transient PVCoordinates partialPV;

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, PositionAngleType type, PositionAngleType cachedPositionAngleType, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, e, i, pa, raan, anomaly, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, type, cachedPositionAngleType, frame, date, mu);
    }

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, PositionAngleType type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, e, i, pa, raan, anomaly, type, type, frame, date, mu);
    }

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, double aDot, double eDot, double iDot, double paDot, double raanDot, double anomalyDot, PositionAngleType type, PositionAngleType cachedPositionAngleType, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        double trueAnomaly;
        this.cachedPositionAngleType = cachedPositionAngleType;
        if (a * (1.0 - e) < 0.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.ORBIT_A_E_MISMATCH_WITH_CONIC_TYPE, a, e);
        }
        this.checkParameterRangeInclusive(ECCENTRICITY, e, 0.0, Double.POSITIVE_INFINITY);
        this.a = a;
        this.aDot = aDot;
        this.e = e;
        this.eDot = eDot;
        this.i = i;
        this.iDot = iDot;
        this.pa = pa;
        this.paDot = paDot;
        this.raan = raan;
        this.raanDot = raanDot;
        if (this.hasDerivatives()) {
            UnivariateDerivative1 cachedAnomalyUD = this.initializeCachedAnomaly(anomaly, anomalyDot, type);
            this.cachedAnomaly = cachedAnomalyUD.getValue();
            this.cachedAnomalyDot = cachedAnomalyUD.getFirstDerivative();
        } else {
            this.cachedAnomaly = this.initializeCachedAnomaly(anomaly, type);
            this.cachedAnomalyDot = Double.NaN;
        }
        if (!this.isElliptical() && 1.0 + e * FastMath.cos((double)(trueAnomaly = this.getTrueAnomaly())) <= 0.0) {
            double vMax = FastMath.acos((double)(-1.0 / e));
            throw new OrekitIllegalArgumentException(OrekitMessages.ORBIT_ANOMALY_OUT_OF_HYPERBOLIC_RANGE, trueAnomaly, e, -vMax, vMax);
        }
        this.partialPV = null;
    }

    public KeplerianOrbit(double a, double e, double i, double pa, double raan, double anomaly, double aDot, double eDot, double iDot, double paDot, double raanDot, double anomalyDot, PositionAngleType type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, e, i, pa, raan, anomaly, aDot, eDot, iDot, paDot, raanDot, anomalyDot, type, type, frame, date, mu);
    }

    public KeplerianOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        this(pvCoordinates, frame, mu, KeplerianOrbit.hasNonKeplerianAcceleration(pvCoordinates, mu));
    }

    private KeplerianOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu, boolean reliableAcceleration) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D momentum = pvCoordinates.getMomentum();
        double m2 = momentum.getNormSq();
        this.i = Vector3D.angle((Vector3D)momentum, (Vector3D)Vector3D.PLUS_K);
        this.raan = Vector3D.crossProduct((Vector3D)Vector3D.PLUS_K, (Vector3D)momentum).getAlpha();
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        Vector3D pvA = pvCoordinates.getAcceleration();
        double r2 = pvP.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        this.a = r / (2.0 - rV2OnMu);
        double muA = mu * this.a;
        if (this.isElliptical()) {
            double eSE = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)muA);
            double eCE = rV2OnMu - 1.0;
            this.e = FastMath.sqrt((double)(eSE * eSE + eCE * eCE));
            this.cachedPositionAngleType = PositionAngleType.ECCENTRIC;
            this.cachedAnomaly = FastMath.atan2((double)eSE, (double)eCE);
        } else {
            double eSH = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)(-muA));
            double eCH = rV2OnMu - 1.0;
            this.e = FastMath.sqrt((double)(1.0 - m2 / muA));
            this.cachedPositionAngleType = PositionAngleType.TRUE;
            this.cachedAnomaly = KeplerianAnomalyUtility.hyperbolicEccentricToTrue(this.e, FastMath.log((double)((eCH + eSH) / (eCH - eSH))) / 2.0);
        }
        this.checkParameterRangeInclusive(ECCENTRICITY, this.e, 0.0, Double.POSITIVE_INFINITY);
        Vector3D node = new Vector3D(this.raan, 0.0);
        double px = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)node);
        double py = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)Vector3D.crossProduct((Vector3D)momentum, (Vector3D)node)) / FastMath.sqrt((double)m2);
        this.pa = FastMath.atan2((double)py, (double)px) - this.getTrueAnomaly();
        this.partialPV = pvCoordinates;
        if (reliableAcceleration) {
            double[][] jacobian = new double[6][6];
            this.getJacobianWrtCartesian(PositionAngleType.MEAN, jacobian);
            Vector3D keplerianAcceleration = new Vector3D(-mu / (r * r2), pvP);
            Vector3D nonKeplerianAcceleration = pvA.subtract((Vector)keplerianAcceleration);
            double aX = nonKeplerianAcceleration.getX();
            double aY = nonKeplerianAcceleration.getY();
            double aZ = nonKeplerianAcceleration.getZ();
            this.aDot = jacobian[0][3] * aX + jacobian[0][4] * aY + jacobian[0][5] * aZ;
            this.eDot = jacobian[1][3] * aX + jacobian[1][4] * aY + jacobian[1][5] * aZ;
            this.iDot = jacobian[2][3] * aX + jacobian[2][4] * aY + jacobian[2][5] * aZ;
            this.paDot = jacobian[3][3] * aX + jacobian[3][4] * aY + jacobian[3][5] * aZ;
            this.raanDot = jacobian[4][3] * aX + jacobian[4][4] * aY + jacobian[4][5] * aZ;
            double MDot = this.getKeplerianMeanMotion() + jacobian[5][3] * aX + jacobian[5][4] * aY + jacobian[5][5] * aZ;
            UnivariateDerivative1 eUD = new UnivariateDerivative1(this.e, this.eDot);
            UnivariateDerivative1 MUD = new UnivariateDerivative1(this.getMeanAnomaly(), MDot);
            if (this.cachedPositionAngleType == PositionAngleType.ECCENTRIC) {
                UnivariateDerivative1 EUD = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicMeanToEccentric(eUD, MUD) : FieldKeplerianAnomalyUtility.ellipticMeanToEccentric(eUD, MUD);
                this.cachedAnomalyDot = EUD.getFirstDerivative();
            } else {
                UnivariateDerivative1 vUD = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicMeanToTrue(eUD, MUD) : FieldKeplerianAnomalyUtility.ellipticMeanToTrue(eUD, MUD);
                this.cachedAnomalyDot = vUD.getFirstDerivative();
            }
        } else {
            this.aDot = Double.NaN;
            this.eDot = Double.NaN;
            this.iDot = Double.NaN;
            this.paDot = Double.NaN;
            this.raanDot = Double.NaN;
            this.cachedAnomalyDot = Double.NaN;
        }
    }

    public KeplerianOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public KeplerianOrbit(Orbit op) {
        this(op.getPVCoordinates(), op.getFrame(), op.getMu(), op.hasDerivatives());
    }

    @Override
    public OrbitType getType() {
        return OrbitType.KEPLERIAN;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getADot() {
        return this.aDot;
    }

    @Override
    public double getE() {
        return this.e;
    }

    @Override
    public double getEDot() {
        return this.eDot;
    }

    @Override
    public double getI() {
        return this.i;
    }

    @Override
    public double getIDot() {
        return this.iDot;
    }

    public double getPerigeeArgument() {
        return this.pa;
    }

    public double getPerigeeArgumentDot() {
        return this.paDot;
    }

    public double getRightAscensionOfAscendingNode() {
        return this.raan;
    }

    public double getRightAscensionOfAscendingNodeDot() {
        return this.raanDot;
    }

    public double getTrueAnomaly() {
        switch (this.cachedPositionAngleType) {
            case MEAN: {
                return this.a < 0.0 ? KeplerianAnomalyUtility.hyperbolicMeanToTrue(this.e, this.cachedAnomaly) : KeplerianAnomalyUtility.ellipticMeanToTrue(this.e, this.cachedAnomaly);
            }
            case TRUE: {
                return this.cachedAnomaly;
            }
            case ECCENTRIC: {
                return this.a < 0.0 ? KeplerianAnomalyUtility.hyperbolicEccentricToTrue(this.e, this.cachedAnomaly) : KeplerianAnomalyUtility.ellipticEccentricToTrue(this.e, this.cachedAnomaly);
            }
        }
        throw new OrekitInternalError(null);
    }

    public double getTrueAnomalyDot() {
        if (this.hasDerivatives()) {
            switch (this.cachedPositionAngleType) {
                case MEAN: {
                    UnivariateDerivative1 eUD = new UnivariateDerivative1(this.e, this.eDot);
                    UnivariateDerivative1 MUD = new UnivariateDerivative1(this.cachedAnomaly, this.cachedAnomalyDot);
                    UnivariateDerivative1 vUD = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicMeanToTrue(eUD, MUD) : FieldKeplerianAnomalyUtility.ellipticMeanToTrue(eUD, MUD);
                    return vUD.getFirstDerivative();
                }
                case TRUE: {
                    return this.cachedAnomalyDot;
                }
                case ECCENTRIC: {
                    UnivariateDerivative1 eUD2 = new UnivariateDerivative1(this.e, this.eDot);
                    UnivariateDerivative1 EUD = new UnivariateDerivative1(this.cachedAnomaly, this.cachedAnomalyDot);
                    UnivariateDerivative1 vUD2 = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicEccentricToTrue(eUD2, EUD) : FieldKeplerianAnomalyUtility.ellipticEccentricToTrue(eUD2, EUD);
                    return vUD2.getFirstDerivative();
                }
            }
            throw new OrekitInternalError(null);
        }
        return Double.NaN;
    }

    public double getEccentricAnomaly() {
        switch (this.cachedPositionAngleType) {
            case MEAN: {
                return this.a < 0.0 ? KeplerianAnomalyUtility.hyperbolicMeanToEccentric(this.e, this.cachedAnomaly) : KeplerianAnomalyUtility.ellipticMeanToEccentric(this.e, this.cachedAnomaly);
            }
            case ECCENTRIC: {
                return this.cachedAnomaly;
            }
            case TRUE: {
                return this.a < 0.0 ? KeplerianAnomalyUtility.hyperbolicTrueToEccentric(this.e, this.cachedAnomaly) : KeplerianAnomalyUtility.ellipticTrueToEccentric(this.e, this.cachedAnomaly);
            }
        }
        throw new OrekitInternalError(null);
    }

    public double getEccentricAnomalyDot() {
        if (this.hasDerivatives()) {
            switch (this.cachedPositionAngleType) {
                case ECCENTRIC: {
                    return this.cachedAnomalyDot;
                }
                case TRUE: {
                    UnivariateDerivative1 eUD = new UnivariateDerivative1(this.e, this.eDot);
                    UnivariateDerivative1 vUD = new UnivariateDerivative1(this.cachedAnomaly, this.cachedAnomalyDot);
                    UnivariateDerivative1 EUD = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicTrueToEccentric(eUD, vUD) : FieldKeplerianAnomalyUtility.ellipticTrueToEccentric(eUD, vUD);
                    return EUD.getFirstDerivative();
                }
                case MEAN: {
                    UnivariateDerivative1 eUD2 = new UnivariateDerivative1(this.e, this.eDot);
                    UnivariateDerivative1 MUD = new UnivariateDerivative1(this.cachedAnomaly, this.cachedAnomalyDot);
                    UnivariateDerivative1 EUD2 = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicMeanToEccentric(eUD2, MUD) : FieldKeplerianAnomalyUtility.ellipticMeanToEccentric(eUD2, MUD);
                    return EUD2.getFirstDerivative();
                }
            }
            throw new OrekitInternalError(null);
        }
        return Double.NaN;
    }

    public double getMeanAnomaly() {
        switch (this.cachedPositionAngleType) {
            case ECCENTRIC: {
                return this.a < 0.0 ? KeplerianAnomalyUtility.hyperbolicEccentricToMean(this.e, this.cachedAnomaly) : KeplerianAnomalyUtility.ellipticEccentricToMean(this.e, this.cachedAnomaly);
            }
            case MEAN: {
                return this.cachedAnomaly;
            }
            case TRUE: {
                return this.a < 0.0 ? KeplerianAnomalyUtility.hyperbolicTrueToMean(this.e, this.cachedAnomaly) : KeplerianAnomalyUtility.ellipticTrueToMean(this.e, this.cachedAnomaly);
            }
        }
        throw new OrekitInternalError(null);
    }

    public double getMeanAnomalyDot() {
        if (this.hasDerivatives()) {
            switch (this.cachedPositionAngleType) {
                case MEAN: {
                    return this.cachedAnomalyDot;
                }
                case ECCENTRIC: {
                    UnivariateDerivative1 eUD = new UnivariateDerivative1(this.e, this.eDot);
                    UnivariateDerivative1 EUD = new UnivariateDerivative1(this.cachedAnomaly, this.cachedAnomalyDot);
                    UnivariateDerivative1 MUD = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicEccentricToMean(eUD, EUD) : FieldKeplerianAnomalyUtility.ellipticEccentricToMean(eUD, EUD);
                    return MUD.getFirstDerivative();
                }
                case TRUE: {
                    UnivariateDerivative1 eUD2 = new UnivariateDerivative1(this.e, this.eDot);
                    UnivariateDerivative1 vUD = new UnivariateDerivative1(this.cachedAnomaly, this.cachedAnomalyDot);
                    UnivariateDerivative1 MUD2 = this.a < 0.0 ? FieldKeplerianAnomalyUtility.hyperbolicTrueToMean(eUD2, vUD) : FieldKeplerianAnomalyUtility.ellipticTrueToMean(eUD2, vUD);
                    return MUD2.getFirstDerivative();
                }
            }
            throw new OrekitInternalError(null);
        }
        return Double.NaN;
    }

    public double getAnomaly(PositionAngleType type) {
        return type == PositionAngleType.MEAN ? this.getMeanAnomaly() : (type == PositionAngleType.ECCENTRIC ? this.getEccentricAnomaly() : this.getTrueAnomaly());
    }

    public double getAnomalyDot(PositionAngleType type) {
        return type == PositionAngleType.MEAN ? this.getMeanAnomalyDot() : (type == PositionAngleType.ECCENTRIC ? this.getEccentricAnomalyDot() : this.getTrueAnomalyDot());
    }

    @Override
    public double getEquinoctialEx() {
        return this.e * FastMath.cos((double)(this.pa + this.raan));
    }

    @Override
    public double getEquinoctialExDot() {
        double paPraan = this.pa + this.raan;
        SinCos sc = FastMath.sinCos((double)paPraan);
        return this.eDot * sc.cos() - this.e * sc.sin() * (this.paDot + this.raanDot);
    }

    @Override
    public double getEquinoctialEy() {
        return this.e * FastMath.sin((double)(this.pa + this.raan));
    }

    @Override
    public double getEquinoctialEyDot() {
        double paPraan = this.pa + this.raan;
        SinCos sc = FastMath.sinCos((double)paPraan);
        return this.eDot * sc.sin() + this.e * sc.cos() * (this.paDot + this.raanDot);
    }

    @Override
    public double getHx() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.cos((double)this.raan) * FastMath.tan((double)(0.5 * this.i));
    }

    @Override
    public double getHxDot() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        SinCos sc = FastMath.sinCos((double)this.raan);
        double tan = FastMath.tan((double)(0.5 * this.i));
        return 0.5 * (1.0 + tan * tan) * sc.cos() * this.iDot - tan * sc.sin() * this.raanDot;
    }

    @Override
    public double getHy() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.sin((double)this.raan) * FastMath.tan((double)(0.5 * this.i));
    }

    @Override
    public double getHyDot() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        SinCos sc = FastMath.sinCos((double)this.raan);
        double tan = FastMath.tan((double)(0.5 * this.i));
        return 0.5 * (1.0 + tan * tan) * sc.sin() * this.iDot + tan * sc.cos() * this.raanDot;
    }

    @Override
    public double getLv() {
        return this.pa + this.raan + this.getTrueAnomaly();
    }

    @Override
    public double getLvDot() {
        return this.paDot + this.raanDot + this.getTrueAnomalyDot();
    }

    @Override
    public double getLE() {
        return this.pa + this.raan + this.getEccentricAnomaly();
    }

    @Override
    public double getLEDot() {
        return this.paDot + this.raanDot + this.getEccentricAnomalyDot();
    }

    @Override
    public double getLM() {
        return this.pa + this.raan + this.getMeanAnomaly();
    }

    @Override
    public double getLMDot() {
        return this.paDot + this.raanDot + this.getMeanAnomalyDot();
    }

    private UnivariateDerivative1 initializeCachedAnomaly(double anomaly, double anomalyDot, PositionAngleType inputType) {
        if (this.cachedPositionAngleType == inputType) {
            return new UnivariateDerivative1(anomaly, anomalyDot);
        }
        UnivariateDerivative1 eUD = new UnivariateDerivative1(this.e, this.eDot);
        UnivariateDerivative1 anomalyUD = new UnivariateDerivative1(anomaly, anomalyDot);
        if (this.a < 0.0) {
            switch (this.cachedPositionAngleType) {
                case MEAN: {
                    if (inputType == PositionAngleType.ECCENTRIC) {
                        return FieldKeplerianAnomalyUtility.hyperbolicEccentricToMean(eUD, anomalyUD);
                    }
                    return FieldKeplerianAnomalyUtility.hyperbolicTrueToMean(eUD, anomalyUD);
                }
                case ECCENTRIC: {
                    if (inputType == PositionAngleType.MEAN) {
                        return FieldKeplerianAnomalyUtility.hyperbolicMeanToEccentric(eUD, anomalyUD);
                    }
                    return FieldKeplerianAnomalyUtility.hyperbolicTrueToEccentric(eUD, anomalyUD);
                }
                case TRUE: {
                    if (inputType == PositionAngleType.MEAN) {
                        return FieldKeplerianAnomalyUtility.hyperbolicMeanToTrue(eUD, anomalyUD);
                    }
                    return FieldKeplerianAnomalyUtility.hyperbolicEccentricToTrue(eUD, anomalyUD);
                }
            }
        } else {
            switch (this.cachedPositionAngleType) {
                case MEAN: {
                    if (inputType == PositionAngleType.ECCENTRIC) {
                        return FieldKeplerianAnomalyUtility.ellipticEccentricToMean(eUD, anomalyUD);
                    }
                    return FieldKeplerianAnomalyUtility.ellipticTrueToMean(eUD, anomalyUD);
                }
                case ECCENTRIC: {
                    if (inputType == PositionAngleType.MEAN) {
                        return FieldKeplerianAnomalyUtility.ellipticMeanToEccentric(eUD, anomalyUD);
                    }
                    return FieldKeplerianAnomalyUtility.ellipticTrueToEccentric(eUD, anomalyUD);
                }
                case TRUE: {
                    if (inputType == PositionAngleType.MEAN) {
                        return FieldKeplerianAnomalyUtility.ellipticMeanToTrue(eUD, anomalyUD);
                    }
                    return FieldKeplerianAnomalyUtility.ellipticEccentricToTrue(eUD, anomalyUD);
                }
            }
        }
        throw new OrekitInternalError(null);
    }

    private double initializeCachedAnomaly(double anomaly, PositionAngleType inputType) {
        return KeplerianAnomalyUtility.convertAnomaly(inputType, anomaly, this.e, this.cachedPositionAngleType);
    }

    private Vector3D[] referenceAxes() {
        SinCos scRaan = FastMath.sinCos((double)this.raan);
        SinCos scPa = FastMath.sinCos((double)this.pa);
        SinCos scI = FastMath.sinCos((double)this.i);
        double cosRaan = scRaan.cos();
        double sinRaan = scRaan.sin();
        double cosPa = scPa.cos();
        double sinPa = scPa.sin();
        double cosI = scI.cos();
        double sinI = scI.sin();
        double crcp = cosRaan * cosPa;
        double crsp = cosRaan * sinPa;
        double srcp = sinRaan * cosPa;
        double srsp = sinRaan * sinPa;
        return new Vector3D[]{new Vector3D(crcp - cosI * srsp, srcp + cosI * crsp, sinI * sinPa), new Vector3D(-crsp - cosI * srcp, -srsp + cosI * crcp, sinI * cosPa)};
    }

    private void computePVWithoutA() {
        if (this.partialPV != null) {
            return;
        }
        Vector3D[] axes = this.referenceAxes();
        if (this.isElliptical()) {
            double uME2 = (1.0 - this.e) * (1.0 + this.e);
            double s1Me2 = FastMath.sqrt((double)uME2);
            SinCos scE = FastMath.sinCos((double)this.getEccentricAnomaly());
            double cosE = scE.cos();
            double sinE = scE.sin();
            double x = this.a * (cosE - this.e);
            double y = this.a * sinE * s1Me2;
            double factor = FastMath.sqrt((double)(this.getMu() / this.a)) / (1.0 - this.e * cosE);
            double xDot = -sinE * factor;
            double yDot = cosE * s1Me2 * factor;
            Vector3D position = new Vector3D(x, axes[0], y, axes[1]);
            Vector3D velocity = new Vector3D(xDot, axes[0], yDot, axes[1]);
            this.partialPV = new PVCoordinates(position, velocity);
        } else {
            SinCos scV = FastMath.sinCos((double)this.getTrueAnomaly());
            double sinV = scV.sin();
            double cosV = scV.cos();
            double f = this.a * (1.0 - this.e * this.e);
            double posFactor = f / (1.0 + this.e * cosV);
            double velFactor = FastMath.sqrt((double)(this.getMu() / f));
            double x = posFactor * cosV;
            double y = posFactor * sinV;
            double xDot = -velFactor * sinV;
            double yDot = velFactor * (this.e + cosV);
            Vector3D position = new Vector3D(x, axes[0], y, axes[1]);
            Vector3D velocity = new Vector3D(xDot, axes[0], yDot, axes[1]);
            this.partialPV = new PVCoordinates(position, velocity);
        }
    }

    private Vector3D nonKeplerianAcceleration() {
        double[][] dCdP = new double[6][6];
        this.getJacobianWrtParameters(PositionAngleType.MEAN, dCdP);
        double nonKeplerianMeanMotion = this.getMeanAnomalyDot() - this.getKeplerianMeanMotion();
        double nonKeplerianAx = dCdP[3][0] * this.aDot + dCdP[3][1] * this.eDot + dCdP[3][2] * this.iDot + dCdP[3][3] * this.paDot + dCdP[3][4] * this.raanDot + dCdP[3][5] * nonKeplerianMeanMotion;
        double nonKeplerianAy = dCdP[4][0] * this.aDot + dCdP[4][1] * this.eDot + dCdP[4][2] * this.iDot + dCdP[4][3] * this.paDot + dCdP[4][4] * this.raanDot + dCdP[4][5] * nonKeplerianMeanMotion;
        double nonKeplerianAz = dCdP[5][0] * this.aDot + dCdP[5][1] * this.eDot + dCdP[5][2] * this.iDot + dCdP[5][3] * this.paDot + dCdP[5][4] * this.raanDot + dCdP[5][5] * nonKeplerianMeanMotion;
        return new Vector3D(nonKeplerianAx, nonKeplerianAy, nonKeplerianAz);
    }

    @Override
    protected Vector3D initPosition() {
        Vector3D[] axes = this.referenceAxes();
        if (this.isElliptical()) {
            double uME2 = (1.0 - this.e) * (1.0 + this.e);
            double s1Me2 = FastMath.sqrt((double)uME2);
            SinCos scE = FastMath.sinCos((double)this.getEccentricAnomaly());
            double cosE = scE.cos();
            double sinE = scE.sin();
            return new Vector3D(this.a * (cosE - this.e), axes[0], this.a * sinE * s1Me2, axes[1]);
        }
        SinCos scV = FastMath.sinCos((double)this.getTrueAnomaly());
        double sinV = scV.sin();
        double cosV = scV.cos();
        double f = this.a * (1.0 - this.e * this.e);
        double posFactor = f / (1.0 + this.e * cosV);
        return new Vector3D(posFactor * cosV, axes[0], posFactor * sinV, axes[1]);
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        this.computePVWithoutA();
        double r2 = this.partialPV.getPosition().getNormSq();
        Vector3D keplerianAcceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt((double)r2)), this.partialPV.getPosition());
        Vector3D acceleration = this.hasDerivatives() ? keplerianAcceleration.add((Vector)this.nonKeplerianAcceleration()) : keplerianAcceleration;
        return new TimeStampedPVCoordinates(this.getDate(), this.partialPV.getPosition(), this.partialPV.getVelocity(), acceleration);
    }

    @Override
    public KeplerianOrbit shiftedBy(double dt) {
        KeplerianOrbit keplerianShifted = new KeplerianOrbit(this.a, this.e, this.i, this.pa, this.raan, this.getMeanAnomaly() + this.getKeplerianMeanMotion() * dt, PositionAngleType.MEAN, this.cachedPositionAngleType, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
        if (this.hasDerivatives()) {
            Vector3D nonKeplerianAcceleration = this.nonKeplerianAcceleration();
            keplerianShifted.computePVWithoutA();
            Vector3D fixedP = new Vector3D(1.0, keplerianShifted.partialPV.getPosition(), 0.5 * dt * dt, nonKeplerianAcceleration);
            double fixedR2 = fixedP.getNormSq();
            double fixedR = FastMath.sqrt((double)fixedR2);
            Vector3D fixedV = new Vector3D(1.0, keplerianShifted.partialPV.getVelocity(), dt, nonKeplerianAcceleration);
            Vector3D fixedA = new Vector3D(-this.getMu() / (fixedR2 * fixedR), keplerianShifted.partialPV.getPosition(), 1.0, nonKeplerianAcceleration);
            return new KeplerianOrbit(new TimeStampedPVCoordinates(keplerianShifted.getDate(), fixedP, fixedV, fixedA), keplerianShifted.getFrame(), keplerianShifted.getMu());
        }
        return keplerianShifted;
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        if (this.isElliptical()) {
            return this.computeJacobianMeanWrtCartesianElliptical();
        }
        return this.computeJacobianMeanWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianMeanWrtCartesianElliptical() {
        double factorI1;
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        Vector3D momentum = this.partialPV.getMomentum();
        double v2 = velocity.getNormSq();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double r3 = r * r2;
        double px = position.getX();
        double py = position.getY();
        double pz = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double mx = momentum.getX();
        double my = momentum.getY();
        double mz = momentum.getZ();
        double mu = this.getMu();
        double sqrtMuA = FastMath.sqrt((double)(this.a * mu));
        double sqrtAoMu = FastMath.sqrt((double)(this.a / mu));
        double a2 = this.a * this.a;
        double twoA = 2.0 * this.a;
        double rOnA = r / this.a;
        double oMe2 = 1.0 - this.e * this.e;
        double epsilon = FastMath.sqrt((double)oMe2);
        double sqrtRec = 1.0 / epsilon;
        SinCos scI = FastMath.sinCos((double)this.i);
        SinCos scPA = FastMath.sinCos((double)this.pa);
        double cosI = scI.cos();
        double sinI = scI.sin();
        double cosPA = scPA.cos();
        double sinPA = scPA.sin();
        double pv = Vector3D.dotProduct((Vector3D)position, (Vector3D)velocity);
        double cosE = (this.a - r) / (this.a * this.e);
        double sinE = pv / (this.e * sqrtMuA);
        Vector3D vectorAR = new Vector3D(2.0 * a2 / r3, position);
        Vector3D vectorARDot = velocity.scalarMultiply(2.0 * a2 / mu);
        KeplerianOrbit.fillHalfRow(1.0, vectorAR, jacobian[0], 0);
        KeplerianOrbit.fillHalfRow(1.0, vectorARDot, jacobian[0], 3);
        double factorER3 = pv / twoA;
        Vector3D vectorER = new Vector3D(cosE * v2 / (r * mu), position, sinE / sqrtMuA, velocity, -factorER3 * sinE / sqrtMuA, vectorAR);
        Vector3D vectorERDot = new Vector3D(sinE / sqrtMuA, position, cosE * 2.0 * r / mu, velocity, -factorER3 * sinE / sqrtMuA, vectorARDot);
        KeplerianOrbit.fillHalfRow(1.0, vectorER, jacobian[1], 0);
        KeplerianOrbit.fillHalfRow(1.0, vectorERDot, jacobian[1], 3);
        double coefE = cosE / (this.e * sqrtMuA);
        Vector3D vectorEAnR = new Vector3D(-sinE * v2 / (this.e * r * mu), position, coefE, velocity, -factorER3 * coefE, vectorAR);
        Vector3D vectorEAnRDot = new Vector3D(-sinE * 2.0 * r / (this.e * mu), velocity, coefE, position, -factorER3 * coefE, vectorARDot);
        double s1 = -sinE * pz / r - cosE * vz * sqrtAoMu;
        double s2 = -cosE * pz / r3;
        double s3 = -sinE * vz / (2.0 * sqrtMuA);
        double t1 = sqrtRec * (cosE * pz / r - sinE * vz * sqrtAoMu);
        double t2 = sqrtRec * (-sinE * pz / r3);
        double t3 = sqrtRec * (cosE - this.e) * vz / (2.0 * sqrtMuA);
        double t4 = sqrtRec * (this.e * sinI * cosPA * sqrtRec - vz * sqrtAoMu);
        Vector3D s = new Vector3D(cosE / r, Vector3D.PLUS_K, s1, vectorEAnR, s2, position, s3, vectorAR);
        Vector3D sDot = new Vector3D(-sinE * sqrtAoMu, Vector3D.PLUS_K, s1, vectorEAnRDot, s3, vectorARDot);
        Vector3D t = new Vector3D(sqrtRec * sinE / r, Vector3D.PLUS_K).add((Vector)new Vector3D(t1, vectorEAnR, t2, position, t3, vectorAR, t4, vectorER));
        Vector3D tDot = new Vector3D(sqrtRec * (cosE - this.e) * sqrtAoMu, Vector3D.PLUS_K, t1, vectorEAnRDot, t3, vectorARDot, t4, vectorERDot);
        double i1 = factorI1 = -sinI * sqrtRec / sqrtMuA;
        double i2 = -factorI1 * mz / twoA;
        double i3 = factorI1 * mz * this.e / oMe2;
        double i4 = cosI * sinPA;
        double i5 = cosI * cosPA;
        KeplerianOrbit.fillHalfRow(i1, new Vector3D(vy, -vx, 0.0), i2, vectorAR, i3, vectorER, i4, s, i5, t, jacobian[2], 0);
        KeplerianOrbit.fillHalfRow(i1, new Vector3D(-py, px, 0.0), i2, vectorARDot, i3, vectorERDot, i4, sDot, i5, tDot, jacobian[2], 3);
        KeplerianOrbit.fillHalfRow(cosPA / sinI, s, -sinPA / sinI, t, jacobian[3], 0);
        KeplerianOrbit.fillHalfRow(cosPA / sinI, sDot, -sinPA / sinI, tDot, jacobian[3], 3);
        double factorRaanR = 1.0 / (mu * this.a * oMe2 * sinI * sinI);
        KeplerianOrbit.fillHalfRow(-factorRaanR * my, new Vector3D(0.0, vz, -vy), factorRaanR * mx, new Vector3D(-vz, 0.0, vx), jacobian[4], 0);
        KeplerianOrbit.fillHalfRow(-factorRaanR * my, new Vector3D(0.0, -pz, py), factorRaanR * mx, new Vector3D(pz, 0.0, -px), jacobian[4], 3);
        KeplerianOrbit.fillHalfRow(rOnA, vectorEAnR, -sinE, vectorER, jacobian[5], 0);
        KeplerianOrbit.fillHalfRow(rOnA, vectorEAnRDot, -sinE, vectorERDot, jacobian[5], 3);
        return jacobian;
    }

    private double[][] computeJacobianMeanWrtCartesianHyperbolic() {
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        Vector3D momentum = this.partialPV.getMomentum();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double r3 = r * r2;
        double x = position.getX();
        double y = position.getY();
        double z = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double mx = momentum.getX();
        double my = momentum.getY();
        double mz = momentum.getZ();
        double mu = this.getMu();
        double absA = -this.a;
        double sqrtMuA = FastMath.sqrt((double)(absA * mu));
        double a2 = this.a * this.a;
        double rOa = r / absA;
        SinCos scI = FastMath.sinCos((double)this.i);
        double cosI = scI.cos();
        double sinI = scI.sin();
        double pv = Vector3D.dotProduct((Vector3D)position, (Vector3D)velocity);
        Vector3D vectorAR = new Vector3D(-2.0 * a2 / r3, position);
        Vector3D vectorARDot = velocity.scalarMultiply(-2.0 * a2 / mu);
        KeplerianOrbit.fillHalfRow(-1.0, vectorAR, jacobian[0], 0);
        KeplerianOrbit.fillHalfRow(-1.0, vectorARDot, jacobian[0], 3);
        double m = momentum.getNorm();
        double oOm = 1.0 / m;
        Vector3D dcXP = new Vector3D(0.0, vz, -vy);
        Vector3D dcYP = new Vector3D(-vz, 0.0, vx);
        Vector3D dcZP = new Vector3D(vy, -vx, 0.0);
        Vector3D dcXV = new Vector3D(0.0, -z, y);
        Vector3D dcYV = new Vector3D(z, 0.0, -x);
        Vector3D dcZV = new Vector3D(-y, x, 0.0);
        Vector3D dCP = new Vector3D(mx * oOm, dcXP, my * oOm, dcYP, mz * oOm, dcZP);
        Vector3D dCV = new Vector3D(mx * oOm, dcXV, my * oOm, dcYV, mz * oOm, dcZV);
        double mOMu = m / mu;
        Vector3D dpP = new Vector3D(2.0 * mOMu, dCP);
        Vector3D dpV = new Vector3D(2.0 * mOMu, dCV);
        double p = m * mOMu;
        double moO2ae = 1.0 / (2.0 * absA * this.e);
        double m2OaMu = -p / absA;
        KeplerianOrbit.fillHalfRow(moO2ae, dpP, m2OaMu * moO2ae, vectorAR, jacobian[1], 0);
        KeplerianOrbit.fillHalfRow(moO2ae, dpV, m2OaMu * moO2ae, vectorARDot, jacobian[1], 3);
        double cI1 = 1.0 / (m * sinI);
        double cI2 = cosI * cI1;
        KeplerianOrbit.fillHalfRow(cI2, dCP, -cI1, dcZP, jacobian[2], 0);
        KeplerianOrbit.fillHalfRow(cI2, dCV, -cI1, dcZV, jacobian[2], 3);
        double cP1 = y * oOm;
        double cP2 = -x * oOm;
        double cP3 = -(mx * cP1 + my * cP2);
        double cP4 = cP3 * oOm;
        double cP5 = -1.0 / (r2 * sinI * sinI);
        double cP6 = z * cP5;
        double cP7 = cP3 * cP5;
        Vector3D dacP = new Vector3D(cP1, dcXP, cP2, dcYP, cP4, dCP, oOm, new Vector3D(-my, mx, 0.0));
        Vector3D dacV = new Vector3D(cP1, dcXV, cP2, dcYV, cP4, dCV);
        Vector3D dpoP = new Vector3D(cP6, dacP, cP7, Vector3D.PLUS_K);
        Vector3D dpoV = new Vector3D(cP6, dacV);
        double re2 = r2 * this.e * this.e;
        double recOre2 = (p - r) / re2;
        double resOre2 = pv * mOMu / re2;
        Vector3D dreP = new Vector3D(mOMu, velocity, pv / mu, dCP);
        Vector3D dreV = new Vector3D(mOMu, position, pv / mu, dCV);
        Vector3D davP = new Vector3D(-resOre2, dpP, recOre2, dreP, resOre2 / r, position);
        Vector3D davV = new Vector3D(-resOre2, dpV, recOre2, dreV);
        KeplerianOrbit.fillHalfRow(1.0, dpoP, -1.0, davP, jacobian[3], 0);
        KeplerianOrbit.fillHalfRow(1.0, dpoV, -1.0, davV, jacobian[3], 3);
        double cO0 = cI1 * cI1;
        double cO1 = mx * cO0;
        double cO2 = -my * cO0;
        KeplerianOrbit.fillHalfRow(cO1, dcYP, cO2, dcXP, jacobian[4], 0);
        KeplerianOrbit.fillHalfRow(cO1, dcYV, cO2, dcXV, jacobian[4], 3);
        double s2a = pv / (2.0 * absA);
        double oObux = 1.0 / FastMath.sqrt((double)(m * m + mu * absA));
        double scasbu = pv * oObux;
        Vector3D dauP = new Vector3D(1.0 / sqrtMuA, velocity, -s2a / sqrtMuA, vectorAR);
        Vector3D dauV = new Vector3D(1.0 / sqrtMuA, position, -s2a / sqrtMuA, vectorARDot);
        Vector3D dbuP = new Vector3D(oObux * mu / 2.0, vectorAR, m * oObux, dCP);
        Vector3D dbuV = new Vector3D(oObux * mu / 2.0, vectorARDot, m * oObux, dCV);
        Vector3D dcuP = new Vector3D(oObux, velocity, -scasbu * oObux, dbuP);
        Vector3D dcuV = new Vector3D(oObux, position, -scasbu * oObux, dbuV);
        KeplerianOrbit.fillHalfRow(1.0, dauP, -this.e / (1.0 + rOa), dcuP, jacobian[5], 0);
        KeplerianOrbit.fillHalfRow(1.0, dauV, -this.e / (1.0 + rOa), dcuV, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        if (this.isElliptical()) {
            return this.computeJacobianEccentricWrtCartesianElliptical();
        }
        return this.computeJacobianEccentricWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianEccentricWrtCartesianElliptical() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesianElliptical();
        SinCos scE = FastMath.sinCos((double)this.getEccentricAnomaly());
        double aOr = 1.0 / (1.0 - this.e * scE.cos());
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aOr * (anomalyRow[j] + scE.sin() * eRow[j]);
        }
        return jacobian;
    }

    private double[][] computeJacobianEccentricWrtCartesianHyperbolic() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesianHyperbolic();
        double H = this.getEccentricAnomaly();
        double coshH = FastMath.cosh((double)H);
        double sinhH = FastMath.sinh((double)H);
        double absaOr = 1.0 / (this.e * coshH - 1.0);
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = absaOr * (anomalyRow[j] - sinhH * eRow[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        if (this.isElliptical()) {
            return this.computeJacobianTrueWrtCartesianElliptical();
        }
        return this.computeJacobianTrueWrtCartesianHyperbolic();
    }

    private double[][] computeJacobianTrueWrtCartesianElliptical() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesianElliptical();
        double e2 = this.e * this.e;
        double oMe2 = 1.0 - e2;
        double epsilon = FastMath.sqrt((double)oMe2);
        SinCos scE = FastMath.sinCos((double)this.getEccentricAnomaly());
        double aOr = 1.0 / (1.0 - this.e * scE.cos());
        double aFactor = epsilon * aOr;
        double eFactor = scE.sin() * aOr / epsilon;
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aFactor * anomalyRow[j] + eFactor * eRow[j];
        }
        return jacobian;
    }

    private double[][] computeJacobianTrueWrtCartesianHyperbolic() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesianHyperbolic();
        double e2 = this.e * this.e;
        double e2Mo = e2 - 1.0;
        double epsilon = FastMath.sqrt((double)e2Mo);
        double H = this.getEccentricAnomaly();
        double coshH = FastMath.cosh((double)H);
        double sinhH = FastMath.sinh((double)H);
        double aOr = 1.0 / (this.e * coshH - 1.0);
        double aFactor = epsilon * aOr;
        double eFactor = sinhH * aOr / epsilon;
        double[] eRow = jacobian[1];
        double[] anomalyRow = jacobian[5];
        for (int j = 0; j < anomalyRow.length; ++j) {
            anomalyRow[j] = aFactor * anomalyRow[j] - eFactor * eRow[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngleType type, double gm, double[] pDot) {
        pDot[5] = pDot[5] + KeplerianOrbit.computeKeplerianAnomalyDot(type, this.a, this.e, gm, this.cachedAnomaly, this.cachedPositionAngleType);
    }

    private static double computeKeplerianAnomalyDot(PositionAngleType type, double a, double e, double mu, double anomaly, PositionAngleType cachedType) {
        double absA = FastMath.abs((double)a);
        double n = FastMath.sqrt((double)(mu / absA)) / absA;
        if (type == PositionAngleType.MEAN) {
            return n;
        }
        double oMe2 = FastMath.abs((double)(1.0 - e * e));
        double ksi = 1.0 + e * FastMath.cos((double)KeplerianAnomalyUtility.convertAnomaly(cachedType, anomaly, e, PositionAngleType.TRUE));
        if (type == PositionAngleType.ECCENTRIC) {
            return n * ksi / oMe2;
        }
        return n * ksi * ksi / (oMe2 * FastMath.sqrt((double)oMe2));
    }

    public String toString() {
        return "Keplerian parameters: " + '{' + "a: " + this.a + "; e: " + this.e + "; i: " + FastMath.toDegrees((double)this.i) + "; pa: " + FastMath.toDegrees((double)this.pa) + "; raan: " + FastMath.toDegrees((double)this.raan) + "; v: " + FastMath.toDegrees((double)this.getTrueAnomaly()) + ";}";
    }

    @Override
    public PositionAngleType getCachedPositionAngleType() {
        return this.cachedPositionAngleType;
    }

    @Override
    public boolean hasRates() {
        return this.hasDerivatives();
    }

    @Override
    public KeplerianOrbit removeRates() {
        PositionAngleType positionAngleType = this.getCachedPositionAngleType();
        return new KeplerianOrbit(this.a, this.e, this.i, this.pa, this.raan, this.cachedAnomaly, positionAngleType, positionAngleType, this.getFrame(), this.getDate(), this.getMu());
    }

    private void checkParameterRangeInclusive(String parameterName, double parameter, double lowerBound, double upperBound) {
        if (parameter < lowerBound || parameter > upperBound) {
            throw new OrekitException((Localizable)OrekitMessages.INVALID_PARAMETER_RANGE, parameterName, parameter, lowerBound, upperBound);
        }
    }

    @DefaultDataContext
    private Object writeReplace() {
        return new DTO(this);
    }

    @DefaultDataContext
    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20231217L;
        private final double[] d;
        private final PositionAngleType positionAngleType;
        private final Frame frame;

        private DTO(KeplerianOrbit orbit) {
            TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
            this.positionAngleType = orbit.cachedPositionAngleType;
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            double epoch = FastMath.floor((double)pv.getDate().durationFrom(j2000Epoch));
            double offset = pv.getDate().durationFrom(j2000Epoch.shiftedBy(epoch));
            this.d = orbit.hasDerivatives() ? new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.e, orbit.i, orbit.pa, orbit.raan, orbit.cachedAnomaly, orbit.aDot, orbit.eDot, orbit.iDot, orbit.paDot, orbit.raanDot, orbit.cachedAnomalyDot} : new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.e, orbit.i, orbit.pa, orbit.raan, orbit.cachedAnomaly};
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            if (this.d.length >= 15) {
                return new KeplerianOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.d[9], this.d[10], this.d[11], this.d[12], this.d[13], this.d[14], this.positionAngleType, this.positionAngleType, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
            }
            return new KeplerianOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.positionAngleType, this.positionAngleType, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

