/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import org.hipparchus.analysis.differentiation.UnivariateDerivative1;
import org.hipparchus.geometry.Vector;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.SinCos;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitInternalError;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.EquinoctialLongitudeArgumentUtility;
import org.orekit.orbits.FieldEquinoctialLongitudeArgumentUtility;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngleBased;
import org.orekit.orbits.PositionAngleType;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class EquinoctialOrbit
extends Orbit
implements PositionAngleBased {
    private static final long serialVersionUID = 20170414L;
    private final double a;
    private final double ex;
    private final double ey;
    private final double hx;
    private final double hy;
    private final double cachedL;
    private final PositionAngleType cachedPositionAngleType;
    private final double aDot;
    private final double exDot;
    private final double eyDot;
    private final double hxDot;
    private final double hyDot;
    private final double cachedLDot;
    private transient PVCoordinates partialPV;

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, PositionAngleType type, PositionAngleType cachedPositionAngleType, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, ex, ey, hx, hy, l, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, type, cachedPositionAngleType, frame, date, mu);
    }

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, PositionAngleType type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, ex, ey, hx, hy, l, type, type, frame, date, mu);
    }

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, double aDot, double exDot, double eyDot, double hxDot, double hyDot, double lDot, PositionAngleType type, PositionAngleType cachedPositionAngleType, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        if (ex * ex + ey * ey >= 1.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.cachedPositionAngleType = cachedPositionAngleType;
        this.a = a;
        this.aDot = aDot;
        this.ex = ex;
        this.exDot = exDot;
        this.ey = ey;
        this.eyDot = eyDot;
        this.hx = hx;
        this.hxDot = hxDot;
        this.hy = hy;
        this.hyDot = hyDot;
        if (this.hasDerivatives()) {
            UnivariateDerivative1 lUD = this.initializeCachedL(l, lDot, type);
            this.cachedL = lUD.getValue();
            this.cachedLDot = lUD.getFirstDerivative();
        } else {
            this.cachedL = this.initializeCachedL(l, type);
            this.cachedLDot = Double.NaN;
        }
        this.partialPV = null;
    }

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, double aDot, double exDot, double eyDot, double hxDot, double hyDot, double lDot, PositionAngleType type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, ex, ey, hx, hy, l, aDot, exDot, eyDot, hxDot, hyDot, lDot, type, type, frame, date, mu);
    }

    public EquinoctialOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        Vector3D pvA = pvCoordinates.getAcceleration();
        double r2 = pvP.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        this.a = r / (2.0 - rV2OnMu);
        if (!this.isElliptical()) {
            throw new OrekitIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        Vector3D w = (Vector3D)pvCoordinates.getMomentum().normalize();
        double d = 1.0 / (1.0 + w.getZ());
        this.hx = -d * w.getY();
        this.hy = d * w.getX();
        this.cachedPositionAngleType = PositionAngleType.TRUE;
        double cLv = (pvP.getX() - d * pvP.getZ() * w.getX()) / r;
        double sLv = (pvP.getY() - d * pvP.getZ() * w.getY()) / r;
        this.cachedL = FastMath.atan2((double)sLv, (double)cLv);
        double eSE = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)(mu * this.a));
        double eCE = rV2OnMu - 1.0;
        double e2 = eCE * eCE + eSE * eSE;
        double f = eCE - e2;
        double g = FastMath.sqrt((double)(1.0 - e2)) * eSE;
        this.ex = this.a * (f * cLv + g * sLv) / r;
        this.ey = this.a * (f * sLv - g * cLv) / r;
        this.partialPV = pvCoordinates;
        if (EquinoctialOrbit.hasNonKeplerianAcceleration(pvCoordinates, mu)) {
            double[][] jacobian = new double[6][6];
            this.getJacobianWrtCartesian(PositionAngleType.MEAN, jacobian);
            Vector3D keplerianAcceleration = new Vector3D(-mu / (r * r2), pvP);
            Vector3D nonKeplerianAcceleration = pvA.subtract((Vector)keplerianAcceleration);
            double aX = nonKeplerianAcceleration.getX();
            double aY = nonKeplerianAcceleration.getY();
            double aZ = nonKeplerianAcceleration.getZ();
            this.aDot = jacobian[0][3] * aX + jacobian[0][4] * aY + jacobian[0][5] * aZ;
            this.exDot = jacobian[1][3] * aX + jacobian[1][4] * aY + jacobian[1][5] * aZ;
            this.eyDot = jacobian[2][3] * aX + jacobian[2][4] * aY + jacobian[2][5] * aZ;
            this.hxDot = jacobian[3][3] * aX + jacobian[3][4] * aY + jacobian[3][5] * aZ;
            this.hyDot = jacobian[4][3] * aX + jacobian[4][4] * aY + jacobian[4][5] * aZ;
            double lMDot = this.getKeplerianMeanMotion() + jacobian[5][3] * aX + jacobian[5][4] * aY + jacobian[5][5] * aZ;
            UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
            UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
            UnivariateDerivative1 lMUD = new UnivariateDerivative1(this.getLM(), lMDot);
            UnivariateDerivative1 lvUD = FieldEquinoctialLongitudeArgumentUtility.meanToTrue(exUD, eyUD, lMUD);
            this.cachedLDot = lvUD.getFirstDerivative();
        } else {
            this.aDot = Double.NaN;
            this.exDot = Double.NaN;
            this.eyDot = Double.NaN;
            this.hxDot = Double.NaN;
            this.hyDot = Double.NaN;
            this.cachedLDot = Double.NaN;
        }
    }

    public EquinoctialOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public EquinoctialOrbit(Orbit op) {
        super(op.getFrame(), op.getDate(), op.getMu());
        this.a = op.getA();
        this.aDot = op.getADot();
        this.ex = op.getEquinoctialEx();
        this.exDot = op.getEquinoctialExDot();
        this.ey = op.getEquinoctialEy();
        this.eyDot = op.getEquinoctialEyDot();
        this.hx = op.getHx();
        this.hxDot = op.getHxDot();
        this.hy = op.getHy();
        this.hyDot = op.getHyDot();
        this.cachedPositionAngleType = PositionAngleType.TRUE;
        this.cachedL = op.getLv();
        this.cachedLDot = op.getLvDot();
        this.partialPV = null;
    }

    @Override
    public OrbitType getType() {
        return OrbitType.EQUINOCTIAL;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getADot() {
        return this.aDot;
    }

    @Override
    public double getEquinoctialEx() {
        return this.ex;
    }

    @Override
    public double getEquinoctialExDot() {
        return this.exDot;
    }

    @Override
    public double getEquinoctialEy() {
        return this.ey;
    }

    @Override
    public double getEquinoctialEyDot() {
        return this.eyDot;
    }

    @Override
    public double getHx() {
        return this.hx;
    }

    @Override
    public double getHxDot() {
        return this.hxDot;
    }

    @Override
    public double getHy() {
        return this.hy;
    }

    @Override
    public double getHyDot() {
        return this.hyDot;
    }

    @Override
    public double getLv() {
        switch (this.cachedPositionAngleType) {
            case TRUE: {
                return this.cachedL;
            }
            case ECCENTRIC: {
                return EquinoctialLongitudeArgumentUtility.eccentricToTrue(this.ex, this.ey, this.cachedL);
            }
            case MEAN: {
                return EquinoctialLongitudeArgumentUtility.meanToTrue(this.ex, this.ey, this.cachedL);
            }
        }
        throw new OrekitInternalError(null);
    }

    @Override
    public double getLvDot() {
        switch (this.cachedPositionAngleType) {
            case ECCENTRIC: {
                UnivariateDerivative1 lEUD = new UnivariateDerivative1(this.cachedL, this.cachedLDot);
                UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
                UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
                UnivariateDerivative1 lvUD = FieldEquinoctialLongitudeArgumentUtility.eccentricToTrue(exUD, eyUD, lEUD);
                return lvUD.getFirstDerivative();
            }
            case TRUE: {
                return this.cachedLDot;
            }
            case MEAN: {
                UnivariateDerivative1 lMUD = new UnivariateDerivative1(this.cachedL, this.cachedLDot);
                UnivariateDerivative1 exUD2 = new UnivariateDerivative1(this.ex, this.exDot);
                UnivariateDerivative1 eyUD2 = new UnivariateDerivative1(this.ey, this.eyDot);
                UnivariateDerivative1 lvUD2 = FieldEquinoctialLongitudeArgumentUtility.meanToTrue(exUD2, eyUD2, lMUD);
                return lvUD2.getFirstDerivative();
            }
        }
        throw new OrekitInternalError(null);
    }

    @Override
    public double getLE() {
        switch (this.cachedPositionAngleType) {
            case TRUE: {
                return EquinoctialLongitudeArgumentUtility.trueToEccentric(this.ex, this.ey, this.cachedL);
            }
            case ECCENTRIC: {
                return this.cachedL;
            }
            case MEAN: {
                return EquinoctialLongitudeArgumentUtility.meanToEccentric(this.ex, this.ey, this.cachedL);
            }
        }
        throw new OrekitInternalError(null);
    }

    @Override
    public double getLEDot() {
        switch (this.cachedPositionAngleType) {
            case TRUE: {
                UnivariateDerivative1 lvUD = new UnivariateDerivative1(this.cachedL, this.cachedLDot);
                UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
                UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
                UnivariateDerivative1 lEUD = FieldEquinoctialLongitudeArgumentUtility.trueToEccentric(exUD, eyUD, lvUD);
                return lEUD.getFirstDerivative();
            }
            case ECCENTRIC: {
                return this.cachedLDot;
            }
            case MEAN: {
                UnivariateDerivative1 lMUD = new UnivariateDerivative1(this.cachedL, this.cachedLDot);
                UnivariateDerivative1 exUD2 = new UnivariateDerivative1(this.ex, this.exDot);
                UnivariateDerivative1 eyUD2 = new UnivariateDerivative1(this.ey, this.eyDot);
                UnivariateDerivative1 lEUD2 = FieldEquinoctialLongitudeArgumentUtility.meanToEccentric(exUD2, eyUD2, lMUD);
                return lEUD2.getFirstDerivative();
            }
        }
        throw new OrekitInternalError(null);
    }

    @Override
    public double getLM() {
        switch (this.cachedPositionAngleType) {
            case TRUE: {
                return EquinoctialLongitudeArgumentUtility.trueToMean(this.ex, this.ey, this.cachedL);
            }
            case MEAN: {
                return this.cachedL;
            }
            case ECCENTRIC: {
                return EquinoctialLongitudeArgumentUtility.eccentricToMean(this.ex, this.ey, this.cachedL);
            }
        }
        throw new OrekitInternalError(null);
    }

    @Override
    public double getLMDot() {
        switch (this.cachedPositionAngleType) {
            case TRUE: {
                UnivariateDerivative1 lvUD = new UnivariateDerivative1(this.cachedL, this.cachedLDot);
                UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
                UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
                UnivariateDerivative1 lMUD = FieldEquinoctialLongitudeArgumentUtility.trueToMean(exUD, eyUD, lvUD);
                return lMUD.getFirstDerivative();
            }
            case MEAN: {
                return this.cachedLDot;
            }
            case ECCENTRIC: {
                UnivariateDerivative1 lEUD = new UnivariateDerivative1(this.cachedL, this.cachedLDot);
                UnivariateDerivative1 exUD2 = new UnivariateDerivative1(this.ex, this.exDot);
                UnivariateDerivative1 eyUD2 = new UnivariateDerivative1(this.ey, this.eyDot);
                UnivariateDerivative1 lMUD2 = FieldEquinoctialLongitudeArgumentUtility.eccentricToMean(exUD2, eyUD2, lEUD);
                return lMUD2.getFirstDerivative();
            }
        }
        throw new OrekitInternalError(null);
    }

    public double getL(PositionAngleType type) {
        return type == PositionAngleType.MEAN ? this.getLM() : (type == PositionAngleType.ECCENTRIC ? this.getLE() : this.getLv());
    }

    public double getLDot(PositionAngleType type) {
        return type == PositionAngleType.MEAN ? this.getLMDot() : (type == PositionAngleType.ECCENTRIC ? this.getLEDot() : this.getLvDot());
    }

    @Deprecated
    public static double eccentricToTrue(double lE, double ex, double ey) {
        return EquinoctialLongitudeArgumentUtility.eccentricToTrue(ex, ey, lE);
    }

    @Deprecated
    public static double trueToEccentric(double lv, double ex, double ey) {
        return EquinoctialLongitudeArgumentUtility.trueToEccentric(ex, ey, lv);
    }

    @Deprecated
    public static double meanToEccentric(double lM, double ex, double ey) {
        return EquinoctialLongitudeArgumentUtility.meanToEccentric(ex, ey, lM);
    }

    @Deprecated
    public static double eccentricToMean(double lE, double ex, double ey) {
        return EquinoctialLongitudeArgumentUtility.eccentricToMean(ex, ey, lE);
    }

    @Override
    public double getE() {
        return FastMath.sqrt((double)(this.ex * this.ex + this.ey * this.ey));
    }

    @Override
    public double getEDot() {
        return (this.ex * this.exDot + this.ey * this.eyDot) / FastMath.sqrt((double)(this.ex * this.ex + this.ey * this.ey));
    }

    @Override
    public double getI() {
        return 2.0 * FastMath.atan((double)FastMath.sqrt((double)(this.hx * this.hx + this.hy * this.hy)));
    }

    @Override
    public double getIDot() {
        double h2 = this.hx * this.hx + this.hy * this.hy;
        double h = FastMath.sqrt((double)h2);
        return 2.0 * (this.hx * this.hxDot + this.hy * this.hyDot) / (h * (1.0 + h2));
    }

    private void computePVWithoutA() {
        if (this.partialPV != null) {
            return;
        }
        double lE = this.getLE();
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double factH = 1.0 / (1.0 + hx2 + hy2);
        double ux = (1.0 + hx2 - hy2) * factH;
        double uy = 2.0 * this.hx * this.hy * factH;
        double uz = -2.0 * this.hy * factH;
        double vx = uy;
        double vy = (1.0 - hx2 + hy2) * factH;
        double vz = 2.0 * this.hx * factH;
        double exey = this.ex * this.ey;
        double ex2 = this.ex * this.ex;
        double ey2 = this.ey * this.ey;
        double e2 = ex2 + ey2;
        double eta = 1.0 + FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / eta;
        SinCos scLe = FastMath.sinCos((double)lE);
        double cLe = scLe.cos();
        double sLe = scLe.sin();
        double exCeyS = this.ex * cLe + this.ey * sLe;
        double x = this.a * ((1.0 - beta * ey2) * cLe + beta * exey * sLe - this.ex);
        double y = this.a * ((1.0 - beta * ex2) * sLe + beta * exey * cLe - this.ey);
        double factor = FastMath.sqrt((double)(this.getMu() / this.a)) / (1.0 - exCeyS);
        double xdot = factor * (-sLe + beta * this.ey * exCeyS);
        double ydot = factor * (cLe - beta * this.ex * exCeyS);
        Vector3D position = new Vector3D(x * ux + y * vx, x * uy + y * vy, x * uz + y * vz);
        Vector3D velocity = new Vector3D(xdot * ux + ydot * vx, xdot * uy + ydot * vy, xdot * uz + ydot * vz);
        this.partialPV = new PVCoordinates(position, velocity);
    }

    private UnivariateDerivative1 initializeCachedL(double l, double lDot, PositionAngleType inputType) {
        if (this.cachedPositionAngleType == inputType) {
            return new UnivariateDerivative1(l, lDot);
        }
        UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
        UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
        UnivariateDerivative1 lUD = new UnivariateDerivative1(l, lDot);
        switch (this.cachedPositionAngleType) {
            case ECCENTRIC: {
                if (inputType == PositionAngleType.MEAN) {
                    return FieldEquinoctialLongitudeArgumentUtility.meanToEccentric(exUD, eyUD, lUD);
                }
                return FieldEquinoctialLongitudeArgumentUtility.trueToEccentric(exUD, eyUD, lUD);
            }
            case TRUE: {
                if (inputType == PositionAngleType.MEAN) {
                    return FieldEquinoctialLongitudeArgumentUtility.meanToTrue(exUD, eyUD, lUD);
                }
                return FieldEquinoctialLongitudeArgumentUtility.eccentricToTrue(exUD, eyUD, lUD);
            }
            case MEAN: {
                if (inputType == PositionAngleType.TRUE) {
                    return FieldEquinoctialLongitudeArgumentUtility.trueToMean(exUD, eyUD, lUD);
                }
                return FieldEquinoctialLongitudeArgumentUtility.eccentricToMean(exUD, eyUD, lUD);
            }
        }
        throw new OrekitInternalError(null);
    }

    private double initializeCachedL(double l, PositionAngleType positionAngleType) {
        return EquinoctialLongitudeArgumentUtility.convertL(positionAngleType, l, this.ex, this.ey, this.cachedPositionAngleType);
    }

    private Vector3D nonKeplerianAcceleration() {
        double[][] dCdP = new double[6][6];
        this.getJacobianWrtParameters(PositionAngleType.MEAN, dCdP);
        double nonKeplerianMeanMotion = this.getLMDot() - this.getKeplerianMeanMotion();
        double nonKeplerianAx = dCdP[3][0] * this.aDot + dCdP[3][1] * this.exDot + dCdP[3][2] * this.eyDot + dCdP[3][3] * this.hxDot + dCdP[3][4] * this.hyDot + dCdP[3][5] * nonKeplerianMeanMotion;
        double nonKeplerianAy = dCdP[4][0] * this.aDot + dCdP[4][1] * this.exDot + dCdP[4][2] * this.eyDot + dCdP[4][3] * this.hxDot + dCdP[4][4] * this.hyDot + dCdP[4][5] * nonKeplerianMeanMotion;
        double nonKeplerianAz = dCdP[5][0] * this.aDot + dCdP[5][1] * this.exDot + dCdP[5][2] * this.eyDot + dCdP[5][3] * this.hxDot + dCdP[5][4] * this.hyDot + dCdP[5][5] * nonKeplerianMeanMotion;
        return new Vector3D(nonKeplerianAx, nonKeplerianAy, nonKeplerianAz);
    }

    @Override
    protected Vector3D initPosition() {
        double lE = this.getLE();
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double factH = 1.0 / (1.0 + hx2 + hy2);
        double ux = (1.0 + hx2 - hy2) * factH;
        double uy = 2.0 * this.hx * this.hy * factH;
        double uz = -2.0 * this.hy * factH;
        double vx = uy;
        double vy = (1.0 - hx2 + hy2) * factH;
        double vz = 2.0 * this.hx * factH;
        double exey = this.ex * this.ey;
        double ex2 = this.ex * this.ex;
        double ey2 = this.ey * this.ey;
        double e2 = ex2 + ey2;
        double eta = 1.0 + FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / eta;
        SinCos scLe = FastMath.sinCos((double)lE);
        double cLe = scLe.cos();
        double sLe = scLe.sin();
        double x = this.a * ((1.0 - beta * ey2) * cLe + beta * exey * sLe - this.ex);
        double y = this.a * ((1.0 - beta * ex2) * sLe + beta * exey * cLe - this.ey);
        return new Vector3D(x * ux + y * vx, x * uy + y * vy, x * uz + y * vz);
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        this.computePVWithoutA();
        double r2 = this.partialPV.getPosition().getNormSq();
        Vector3D keplerianAcceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt((double)r2)), this.partialPV.getPosition());
        Vector3D acceleration = this.hasDerivatives() ? keplerianAcceleration.add((Vector)this.nonKeplerianAcceleration()) : keplerianAcceleration;
        return new TimeStampedPVCoordinates(this.getDate(), this.partialPV.getPosition(), this.partialPV.getVelocity(), acceleration);
    }

    @Override
    public EquinoctialOrbit shiftedBy(double dt) {
        EquinoctialOrbit keplerianShifted = new EquinoctialOrbit(this.a, this.ex, this.ey, this.hx, this.hy, this.getLM() + this.getKeplerianMeanMotion() * dt, PositionAngleType.MEAN, this.cachedPositionAngleType, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
        if (this.hasDerivatives()) {
            Vector3D nonKeplerianAcceleration = this.nonKeplerianAcceleration();
            keplerianShifted.computePVWithoutA();
            Vector3D fixedP = new Vector3D(1.0, keplerianShifted.partialPV.getPosition(), 0.5 * dt * dt, nonKeplerianAcceleration);
            double fixedR2 = fixedP.getNormSq();
            double fixedR = FastMath.sqrt((double)fixedR2);
            Vector3D fixedV = new Vector3D(1.0, keplerianShifted.partialPV.getVelocity(), dt, nonKeplerianAcceleration);
            Vector3D fixedA = new Vector3D(-this.getMu() / (fixedR2 * fixedR), keplerianShifted.partialPV.getPosition(), 1.0, nonKeplerianAcceleration);
            return new EquinoctialOrbit(new TimeStampedPVCoordinates(keplerianShifted.getDate(), fixedP, fixedV, fixedA), keplerianShifted.getFrame(), keplerianShifted.getMu());
        }
        return keplerianShifted;
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double r3 = r * r2;
        double mu = this.getMu();
        double sqrtMuA = FastMath.sqrt((double)(this.a * mu));
        double a2 = this.a * this.a;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double oMe2 = 1.0 - e2;
        double epsilon = FastMath.sqrt((double)oMe2);
        double beta = 1.0 / (1.0 + epsilon);
        double ratio = epsilon * beta;
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double hxhy = this.hx * this.hy;
        Vector3D f = (Vector3D)new Vector3D(1.0 - hy2 + hx2, 2.0 * hxhy, -2.0 * this.hy).normalize();
        Vector3D g = (Vector3D)new Vector3D(2.0 * hxhy, 1.0 + hy2 - hx2, 2.0 * this.hx).normalize();
        Vector3D w = (Vector3D)Vector3D.crossProduct((Vector3D)position, (Vector3D)velocity).normalize();
        double x = Vector3D.dotProduct((Vector3D)position, (Vector3D)f);
        double y = Vector3D.dotProduct((Vector3D)position, (Vector3D)g);
        double xDot = Vector3D.dotProduct((Vector3D)velocity, (Vector3D)f);
        double yDot = Vector3D.dotProduct((Vector3D)velocity, (Vector3D)g);
        double c1 = this.a / (sqrtMuA * epsilon);
        double c2 = this.a * sqrtMuA * beta / r3;
        double c3 = sqrtMuA / (r3 * epsilon);
        Vector3D drDotSdEx = new Vector3D(c1 * xDot * yDot - c2 * this.ey * x - c3 * x * y, f, -c1 * xDot * xDot - c2 * this.ey * y + c3 * x * x, g);
        Vector3D drDotSdEy = new Vector3D(c1 * yDot * yDot + c2 * this.ex * x - c3 * y * y, f, -c1 * xDot * yDot + c2 * this.ex * y + c3 * x * y, g);
        Vector3D vectorAR = new Vector3D(2.0 * a2 / r3, position);
        Vector3D vectorARDot = new Vector3D(2.0 * a2 / mu, velocity);
        EquinoctialOrbit.fillHalfRow(1.0, vectorAR, jacobian[0], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorARDot, jacobian[0], 3);
        double d1 = -this.a * ratio / r3;
        double d2 = (this.hy * xDot - this.hx * yDot) / (sqrtMuA * epsilon);
        double d3 = (this.hx * y - this.hy * x) / sqrtMuA;
        Vector3D vectorExRDot = new Vector3D((2.0 * x * yDot - xDot * y) / mu, g, -y * yDot / mu, f, -this.ey * d3 / epsilon, w);
        EquinoctialOrbit.fillHalfRow(this.ex * d1, position, -this.ey * d2, w, epsilon / sqrtMuA, drDotSdEy, jacobian[1], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorExRDot, jacobian[1], 3);
        Vector3D vectorEyRDot = new Vector3D((2.0 * xDot * y - x * yDot) / mu, f, -x * xDot / mu, g, this.ex * d3 / epsilon, w);
        EquinoctialOrbit.fillHalfRow(this.ey * d1, position, this.ex * d2, w, -epsilon / sqrtMuA, drDotSdEx, jacobian[2], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorEyRDot, jacobian[2], 3);
        double h = (1.0 + hx2 + hy2) / (2.0 * sqrtMuA * epsilon);
        EquinoctialOrbit.fillHalfRow(-h * xDot, w, jacobian[3], 0);
        EquinoctialOrbit.fillHalfRow(h * x, w, jacobian[3], 3);
        EquinoctialOrbit.fillHalfRow(-h * yDot, w, jacobian[4], 0);
        EquinoctialOrbit.fillHalfRow(h * y, w, jacobian[4], 3);
        double l = -ratio / sqrtMuA;
        EquinoctialOrbit.fillHalfRow(-1.0 / sqrtMuA, velocity, d2, w, l * this.ex, drDotSdEx, l * this.ey, drDotSdEy, jacobian[5], 0);
        EquinoctialOrbit.fillHalfRow(-2.0 / sqrtMuA, position, this.ex * beta, vectorEyRDot, -this.ey * beta, vectorExRDot, d3, w, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesian();
        SinCos scLe = FastMath.sinCos((double)this.getLE());
        double cosLe = scLe.cos();
        double sinLe = scLe.sin();
        double aOr = 1.0 / (1.0 - this.ex * cosLe - this.ey * sinLe);
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = aOr * (rowL[j] + sinLe * rowEx[j] - cosLe * rowEy[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesian();
        SinCos scLe = FastMath.sinCos((double)this.getLE());
        double cosLe = scLe.cos();
        double sinLe = scLe.sin();
        double eSinE = this.ex * sinLe - this.ey * cosLe;
        double ecosE = this.ex * cosLe + this.ey * sinLe;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double epsilon = FastMath.sqrt((double)(1.0 - e2));
        double onePeps = 1.0 + epsilon;
        double d = onePeps - ecosE;
        double cT = (d * d + eSinE * eSinE) / 2.0;
        double cE = ecosE * onePeps - e2;
        double cX = this.ex * eSinE / epsilon - this.ey + sinLe * onePeps;
        double cY = this.ey * eSinE / epsilon + this.ex - cosLe * onePeps;
        double factorLe = (cT + cE) / cT;
        double factorEx = cX / cT;
        double factorEy = cY / cT;
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = factorLe * rowL[j] + factorEx * rowEx[j] + factorEy * rowEy[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngleType type, double gm, double[] pDot) {
        pDot[5] = pDot[5] + EquinoctialOrbit.computeKeplerianLDot(type, this.a, this.ex, this.ey, gm, this.cachedL, this.cachedPositionAngleType);
    }

    private static double computeKeplerianLDot(PositionAngleType type, double a, double ex, double ey, double mu, double l, PositionAngleType cachedType) {
        double n = FastMath.sqrt((double)(mu / a)) / a;
        if (type == PositionAngleType.MEAN) {
            return n;
        }
        if (type == PositionAngleType.ECCENTRIC) {
            SinCos sc = FastMath.sinCos((double)EquinoctialLongitudeArgumentUtility.convertL(cachedType, l, ex, ey, type));
            double ksi = 1.0 / (1.0 - ex * sc.cos() - ey * sc.sin());
            return n * ksi;
        }
        SinCos sc = FastMath.sinCos((double)EquinoctialLongitudeArgumentUtility.convertL(cachedType, l, ex, ey, type));
        double oMe2 = 1.0 - ex * ex - ey * ey;
        double ksi = 1.0 + ex * sc.cos() + ey * sc.sin();
        return n * ksi * ksi / (oMe2 * FastMath.sqrt((double)oMe2));
    }

    public String toString() {
        return "equinoctial parameters: " + '{' + "a: " + this.a + "; ex: " + this.ex + "; ey: " + this.ey + "; hx: " + this.hx + "; hy: " + this.hy + "; lv: " + FastMath.toDegrees((double)this.getLv()) + ";}";
    }

    @Override
    public PositionAngleType getCachedPositionAngleType() {
        return this.cachedPositionAngleType;
    }

    @Override
    public boolean hasRates() {
        return this.hasDerivatives();
    }

    @Override
    public EquinoctialOrbit removeRates() {
        PositionAngleType positionAngleType = this.getCachedPositionAngleType();
        return new EquinoctialOrbit(this.getA(), this.getEquinoctialEx(), this.getEquinoctialEy(), this.getHx(), this.getHy(), this.getL(positionAngleType), positionAngleType, this.getFrame(), this.getDate(), this.getMu());
    }

    @DefaultDataContext
    private Object writeReplace() {
        return new DTO(this);
    }

    @DefaultDataContext
    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20231217L;
        private final double[] d;
        private final Frame frame;
        private final PositionAngleType positionAngleType;

        private DTO(EquinoctialOrbit orbit) {
            TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
            this.positionAngleType = orbit.cachedPositionAngleType;
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            double epoch = FastMath.floor((double)pv.getDate().durationFrom(j2000Epoch));
            double offset = pv.getDate().durationFrom(j2000Epoch.shiftedBy(epoch));
            this.d = orbit.hasDerivatives() ? new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.hx, orbit.hy, orbit.cachedL, orbit.aDot, orbit.exDot, orbit.eyDot, orbit.hxDot, orbit.hyDot, orbit.cachedLDot} : new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.hx, orbit.hy, orbit.cachedL};
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            if (this.d.length >= 15) {
                return new EquinoctialOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.d[9], this.d[10], this.d[11], this.d[12], this.d[13], this.d[14], this.positionAngleType, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
            }
            return new EquinoctialOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.positionAngleType, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

