/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.stk;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Pattern;
import org.hipparchus.exception.Localizable;
import org.hipparchus.exception.LocalizedCoreFormats;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.orekit.data.DataSource;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.general.EphemerisFileParser;
import org.orekit.files.stk.STKEphemerisFile;
import org.orekit.frames.Frame;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateTimeComponents;
import org.orekit.time.Month;
import org.orekit.time.TimeScale;
import org.orekit.time.UTCScale;
import org.orekit.utils.CartesianDerivativesFilter;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class STKEphemerisFileParser
implements EphemerisFileParser<STKEphemerisFile> {
    private static final Pattern SEPARATOR = Pattern.compile("\\s+");
    private static final Pattern IGNORABLE_LINE = Pattern.compile("^\\s*(#.*)?");
    private static final String MATCH_ANY_REGEX = ".*";
    private static final List<LineParser> KEYWORDS = Arrays.asList(LineParser.NUMBER_OF_EPHEMERIS_POINTS, LineParser.SCENARIO_EPOCH, LineParser.INTERPOLATION_METHOD, LineParser.INTERPOLATION_SAMPLESM1, LineParser.CENTRAL_BODY, LineParser.COORDINATE_SYSTEM, LineParser.BEGIN_SEGMENT_BOUNDARY_TIMES, LineParser.EPHEMERIS_TIME_POS, LineParser.EPHEMERIS_TIME_POS_VEL, LineParser.EPHEMERIS_TIME_POS_VEL_ACC);
    private final String satelliteId;
    private final double mu;
    private final UTCScale utc;
    private final Map<STKEphemerisFile.STKCoordinateSystem, Frame> frameMapping;

    public STKEphemerisFileParser(String satelliteId, double mu, UTCScale utc, Map<STKEphemerisFile.STKCoordinateSystem, Frame> frameMapping) {
        this.satelliteId = Objects.requireNonNull(satelliteId);
        this.mu = mu;
        this.utc = Objects.requireNonNull(utc);
        this.frameMapping = Collections.unmodifiableMap(new EnumMap<STKEphemerisFile.STKCoordinateSystem, Frame>(frameMapping));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public STKEphemerisFile parse(DataSource source) {
        try (Reader reader = source.getOpener().openReaderOnce();
             BufferedReader br = reader == null ? null : new BufferedReader(reader);){
            if (br == null) {
                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_FIND_FILE, source.getName());
            }
            ParseInfo pi = new ParseInfo();
            int lineNumber = 0;
            Iterable<LineParser> parsers = Collections.singleton(LineParser.VERSION);
            String line = br.readLine();
            while (line != null) {
                ++lineNumber;
                if (pi.file != null) break;
                if (!IGNORABLE_LINE.matcher(line).matches()) {
                    LineParser candidate;
                    Iterator iterator = parsers.iterator();
                    do {
                        if (iterator.hasNext()) continue;
                        throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, source.getName(), line);
                    } while (!(candidate = (LineParser)((Object)iterator.next())).canHandle(line));
                    try {
                        candidate.parse(line, pi);
                        parsers = candidate.allowedNext();
                    }
                    catch (IllegalArgumentException | StringIndexOutOfBoundsException e) {
                        throw new OrekitException(e, OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, source.getName(), line);
                    }
                }
                line = br.readLine();
            }
            if (pi.file != null) {
                STKEphemerisFile sTKEphemerisFile = pi.file;
                return sTKEphemerisFile;
            }
            throw new OrekitException((Localizable)OrekitMessages.STK_UNEXPECTED_END_OF_FILE, lineNumber);
        }
        catch (IOException ioe) {
            throw new OrekitException(ioe, (Localizable)LocalizedCoreFormats.SIMPLE_MESSAGE, ioe.getLocalizedMessage());
        }
    }

    private static enum STKDistanceUnit {
        KILOMETERS(1000.0),
        METERS(1.0);

        private final double conversionToMetersFactor;

        private STKDistanceUnit(double conversionToMetersFactor) {
            this.conversionToMetersFactor = conversionToMetersFactor;
        }
    }

    private static enum LineParser {
        VERSION("^stk\\.v\\.\\d+\\.\\d+$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.stkVersion = line;
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.singleton(BEGIN_EPHEMERIS);
            }
        }
        ,
        BEGIN_EPHEMERIS("^\\s*BEGIN Ephemeris\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        NUMBER_OF_EPHEMERIS_POINTS("^\\s*NumberOfEphemerisPoints\\s*\\d+\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.numberOfEphemerisPoints = Integer.parseInt(SEPARATOR.split(line.trim())[1]);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        SCENARIO_EPOCH("^\\s*ScenarioEpoch\\s* \\d{2} [a-zA-Z]{3} \\d{4} \\d{2}:\\d{2}:\\d{2}(\\.\\d*)?\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] tokens = SEPARATOR.split(line.trim());
                int dayOfMonth = Integer.parseInt(tokens[1]);
                Month month = Month.parseMonth(tokens[2]);
                int year = Integer.parseInt(tokens[3]);
                int hour = Integer.parseInt(tokens[4].substring(0, 2));
                int minute = Integer.parseInt(tokens[4].substring(3, 5));
                double seconds = Double.parseDouble(tokens[4].substring(6));
                DateTimeComponents dateTimeComponents = new DateTimeComponents(year, month, dayOfMonth, hour, minute, seconds);
                pi.scenarioEpoch = new AbsoluteDate(dateTimeComponents, (TimeScale)pi.getUTCScale());
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        INTERPOLATION_METHOD("^\\s*InterpolationMethod\\s+[a-zA-Z]+\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        INTERPOLATION_SAMPLESM1("^\\s*InterpolationSamplesM1\\s+\\d+\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.interpolationSamplesM1 = Integer.parseInt(SEPARATOR.split(line.trim())[1]);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        CENTRAL_BODY("^\\s*CentralBody\\s+[a-zA-Z]+\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        COORDINATE_SYSTEM("^\\s*CoordinateSystem\\s+[a-zA-Z0-9]+\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.coordinateSystem = STKEphemerisFile.STKCoordinateSystem.parse(SEPARATOR.split(line.trim())[1]);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        DISTANCE_UNIT("^\\s*DistanceUnit\\s+[a-zA-Z0-9]+\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.distanceUnit = STKDistanceUnit.valueOf(SEPARATOR.split(line.trim())[1].toUpperCase());
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        BEGIN_SEGMENT_BOUNDARY_TIMES("^\\s*BEGIN SegmentBoundaryTimes\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.singleton(SEGMENT_BOUNDARY_TIME);
            }
        }
        ,
        SEGMENT_BOUNDARY_TIME(".*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.segmentBoundaryTimes.add(Double.parseDouble(SEPARATOR.split(line.trim())[0]));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(END_SEGMENT_BOUNDARY_TIMES, SEGMENT_BOUNDARY_TIME);
            }
        }
        ,
        END_SEGMENT_BOUNDARY_TIMES("^\\s*END SegmentBoundaryTimes\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return KEYWORDS;
            }
        }
        ,
        EPHEMERIS_TIME_POS("^\\s*EphemerisTimePos\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.cartesianDerivativesFilter = CartesianDerivativesFilter.USE_P;
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.singleton(EPHEMERIS_TIME_POS_DATUM);
            }
        }
        ,
        EPHEMERIS_TIME_POS_DATUM(".*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] tokens = SEPARATOR.split(line.trim());
                double time = Double.parseDouble(tokens[0]);
                double px = Double.parseDouble(tokens[1]) * pi.distanceUnit.conversionToMetersFactor;
                double py = Double.parseDouble(tokens[2]) * pi.distanceUnit.conversionToMetersFactor;
                double pz = Double.parseDouble(tokens[3]) * pi.distanceUnit.conversionToMetersFactor;
                Vector3D position = new Vector3D(px, py, pz);
                Vector3D velocity = Vector3D.ZERO;
                pi.addEphemeris(time, new PVCoordinates(position, velocity));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(END_EPHEMERIS, EPHEMERIS_TIME_POS_DATUM);
            }
        }
        ,
        EPHEMERIS_TIME_POS_VEL("^\\s*EphemerisTimePosVel\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.cartesianDerivativesFilter = CartesianDerivativesFilter.USE_PV;
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.singleton(EPHEMERIS_TIME_POS_VEL_DATUM);
            }
        }
        ,
        EPHEMERIS_TIME_POS_VEL_DATUM(".*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] tokens = SEPARATOR.split(line.trim());
                double time = Double.parseDouble(tokens[0]);
                double px = Double.parseDouble(tokens[1]) * pi.distanceUnit.conversionToMetersFactor;
                double py = Double.parseDouble(tokens[2]) * pi.distanceUnit.conversionToMetersFactor;
                double pz = Double.parseDouble(tokens[3]) * pi.distanceUnit.conversionToMetersFactor;
                double vx = Double.parseDouble(tokens[4]) * pi.distanceUnit.conversionToMetersFactor;
                double vy = Double.parseDouble(tokens[5]) * pi.distanceUnit.conversionToMetersFactor;
                double vz = Double.parseDouble(tokens[6]) * pi.distanceUnit.conversionToMetersFactor;
                Vector3D position = new Vector3D(px, py, pz);
                Vector3D velocity = new Vector3D(vx, vy, vz);
                pi.addEphemeris(time, new PVCoordinates(position, velocity));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(END_EPHEMERIS, EPHEMERIS_TIME_POS_VEL_DATUM);
            }
        }
        ,
        EPHEMERIS_TIME_POS_VEL_ACC("^\\s*EphemerisTimePosVelAcc\\s*(#.*)?$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.cartesianDerivativesFilter = CartesianDerivativesFilter.USE_PVA;
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.singleton(EPHEMERIS_TIME_POS_VEL_ACC_DATUM);
            }
        }
        ,
        EPHEMERIS_TIME_POS_VEL_ACC_DATUM(".*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] tokens = SEPARATOR.split(line.trim());
                double time = Double.parseDouble(tokens[0]);
                double px = Double.parseDouble(tokens[1]) * pi.distanceUnit.conversionToMetersFactor;
                double py = Double.parseDouble(tokens[2]) * pi.distanceUnit.conversionToMetersFactor;
                double pz = Double.parseDouble(tokens[3]) * pi.distanceUnit.conversionToMetersFactor;
                double vx = Double.parseDouble(tokens[4]) * pi.distanceUnit.conversionToMetersFactor;
                double vy = Double.parseDouble(tokens[5]) * pi.distanceUnit.conversionToMetersFactor;
                double vz = Double.parseDouble(tokens[6]) * pi.distanceUnit.conversionToMetersFactor;
                double ax = Double.parseDouble(tokens[7]) * pi.distanceUnit.conversionToMetersFactor;
                double ay = Double.parseDouble(tokens[8]) * pi.distanceUnit.conversionToMetersFactor;
                double az = Double.parseDouble(tokens[9]) * pi.distanceUnit.conversionToMetersFactor;
                Vector3D position = new Vector3D(px, py, pz);
                Vector3D velocity = new Vector3D(vx, vy, vz);
                Vector3D acceleration = new Vector3D(ax, ay, az);
                pi.addEphemeris(time, new PVCoordinates(position, velocity, acceleration));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(END_EPHEMERIS, EPHEMERIS_TIME_POS_VEL_ACC_DATUM);
            }
        }
        ,
        END_EPHEMERIS("\\s*END Ephemeris\\s*(#.*)?"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.complete();
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.emptyList();
            }
        };

        private final Pattern pattern;

        private LineParser(String regex) {
            this.pattern = Pattern.compile(regex, 2);
        }

        public abstract void parse(String var1, ParseInfo var2);

        public abstract Iterable<LineParser> allowedNext();

        public boolean canHandle(String line) {
            return this.pattern.matcher(line).matches();
        }
    }

    private final class ParseInfo {
        private String stkVersion;
        private AbsoluteDate scenarioEpoch;
        private Integer numberOfEphemerisPoints;
        private int interpolationSamplesM1 = 5;
        private CartesianDerivativesFilter cartesianDerivativesFilter;
        private STKEphemerisFile.STKCoordinateSystem coordinateSystem;
        private STKDistanceUnit distanceUnit = STKDistanceUnit.METERS;
        private int numberOfEphemerisPointsRead;
        private SortedSet<Double> segmentBoundaryTimes;
        private List<STKEphemerisFile.STKEphemerisSegment> ephemerisSegments;
        private TimeStampedPVCoordinates lastSavedEphemeris;
        private List<TimeStampedPVCoordinates> segmentEphemeris;
        private STKEphemerisFile file;

        private ParseInfo() {
            this.coordinateSystem = STKEphemerisFile.STKCoordinateSystem.FIXED;
            this.ephemerisSegments = new ArrayList<STKEphemerisFile.STKEphemerisSegment>();
            this.segmentBoundaryTimes = new TreeSet<Double>();
            this.segmentEphemeris = new ArrayList<TimeStampedPVCoordinates>();
        }

        private UTCScale getUTCScale() {
            return STKEphemerisFileParser.this.utc;
        }

        private void addEphemeris(double time, PVCoordinates pvCoordinates) {
            if (this.numberOfEphemerisPoints != null && this.numberOfEphemerisPointsRead == this.numberOfEphemerisPoints) {
                return;
            }
            AbsoluteDate date = this.scenarioEpoch.shiftedBy(time);
            TimeStampedPVCoordinates timeStampedPVCoordinates = new TimeStampedPVCoordinates(date, pvCoordinates);
            if (this.segmentBoundaryTimes.contains(time) && this.numberOfEphemerisPointsRead > 0) {
                if (this.segmentEphemeris.isEmpty()) {
                    if (!date.equals(this.lastSavedEphemeris.getDate())) {
                        this.segmentEphemeris.add(this.lastSavedEphemeris);
                    }
                    this.segmentEphemeris.add(timeStampedPVCoordinates);
                } else {
                    this.segmentEphemeris.add(timeStampedPVCoordinates);
                    this.ephemerisSegments.add(new STKEphemerisFile.STKEphemerisSegment(STKEphemerisFileParser.this.mu, this.getFrame(), 1 + this.interpolationSamplesM1, this.cartesianDerivativesFilter, this.segmentEphemeris));
                    this.segmentEphemeris = new ArrayList<TimeStampedPVCoordinates>();
                }
            } else {
                this.segmentEphemeris.add(timeStampedPVCoordinates);
            }
            this.lastSavedEphemeris = timeStampedPVCoordinates;
            ++this.numberOfEphemerisPointsRead;
        }

        private Frame getFrame() {
            STKEphemerisFile.STKCoordinateSystem stkCoordinateSystem = this.coordinateSystem == null ? STKEphemerisFile.STKCoordinateSystem.FIXED : this.coordinateSystem;
            Frame frame = (Frame)STKEphemerisFileParser.this.frameMapping.get((Object)stkCoordinateSystem);
            if (frame == null) {
                throw new OrekitException((Localizable)OrekitMessages.STK_UNMAPPED_COORDINATE_SYSTEM, new Object[]{stkCoordinateSystem});
            }
            return frame;
        }

        private void complete() {
            if (!this.segmentEphemeris.isEmpty()) {
                this.ephemerisSegments.add(new STKEphemerisFile.STKEphemerisSegment(STKEphemerisFileParser.this.mu, this.getFrame(), 1 + this.interpolationSamplesM1, this.cartesianDerivativesFilter, this.segmentEphemeris));
            }
            STKEphemerisFile.STKEphemeris ephemeris = new STKEphemerisFile.STKEphemeris(STKEphemerisFileParser.this.satelliteId, STKEphemerisFileParser.this.mu, this.ephemerisSegments);
            this.file = new STKEphemerisFile(this.stkVersion, STKEphemerisFileParser.this.satelliteId, ephemeris);
        }
    }
}

