/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.sp3;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.hipparchus.util.FastMath;
import org.orekit.files.sp3.DataUsed;
import org.orekit.files.sp3.SP3;
import org.orekit.files.sp3.SP3Coordinate;
import org.orekit.files.sp3.SP3Ephemeris;
import org.orekit.files.sp3.SP3Header;
import org.orekit.files.sp3.SP3Segment;
import org.orekit.files.sp3.SP3Utils;
import org.orekit.gnss.TimeSystem;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateTimeComponents;
import org.orekit.time.TimeScale;
import org.orekit.time.TimeScales;
import org.orekit.utils.CartesianDerivativesFilter;

public class SP3Writer {
    private static final String EOL = System.lineSeparator();
    private static final String ACCURACY_LINE_PREFIX = "++       ";
    private static final String COMMENT_LINE_PREFIX = "/* ";
    private static final String ACCURACY_BASE_FORMAT = "%%f %10.7f %12.9f %14.11f %18.15f%n";
    private static final String ADDITIONAL_PARAMETERS_LINE = "%i    0    0    0    0      0      0      0      0         0";
    private static final String TWO_DIGITS_INTEGER = "%2d";
    private static final String THREE_DIGITS_INTEGER = "%3d";
    private static final String FOURTEEN_SIX_DIGITS_FLOAT = "%14.6f";
    private static final String THREE_BLANKS = "   ";
    private static final String TIME_SYSTEM_DEFAULT = "%c cc cc ccc ccc cccc cccc cccc cccc ccccc ccccc ccccc ccccc";
    private final Appendable output;
    private final String outputName;
    private final TimeScales timeScales;

    public SP3Writer(Appendable output, String outputName, TimeScales timeScales) {
        this.output = output;
        this.outputName = outputName;
        this.timeScales = timeScales;
    }

    public void write(SP3 sp3) throws IOException {
        sp3.validate(false, this.outputName);
        this.writeHeader(sp3.getHeader());
        CoordinatesIterator[] iterators = new CoordinatesIterator[sp3.getSatelliteCount()];
        int k = 0;
        for (Map.Entry<String, SP3Ephemeris> entry : sp3.getSatellites().entrySet()) {
            iterators[k++] = new CoordinatesIterator(entry.getValue());
        }
        TimeScale timeScale = sp3.getHeader().getTimeSystem().getTimeScale(this.timeScales);
        AbsoluteDate date = this.earliest(iterators);
        while (!date.equals(AbsoluteDate.FUTURE_INFINITY)) {
            DateTimeComponents dtc = date.getComponents(timeScale);
            this.output.append(String.format(Locale.US, "*  %4d %2d %2d %2d %2d %11.8f%n", dtc.getDate().getYear(), dtc.getDate().getMonth(), dtc.getDate().getDay(), dtc.getTime().getHour(), dtc.getTime().getMinute(), dtc.getTime().getSecond()));
            for (CoordinatesIterator iter : iterators) {
                SP3Coordinate coordinate;
                if (iter.pending != null && FastMath.abs((double)iter.pending.getDate().durationFrom(date)) <= 0.001 * sp3.getHeader().getEpochInterval()) {
                    coordinate = iter.pending;
                    iter.advance();
                } else {
                    coordinate = SP3Coordinate.DUMMY;
                }
                this.writePosition(sp3.getHeader(), iter.id, coordinate);
                if (sp3.getHeader().getFilter() == CartesianDerivativesFilter.USE_P) continue;
                this.writeVelocity(sp3.getHeader(), iter.id, coordinate);
            }
            date = this.earliest(iterators);
        }
        this.output.append("EOF").append(EOL);
    }

    private AbsoluteDate earliest(CoordinatesIterator[] iterators) {
        AbsoluteDate date = AbsoluteDate.FUTURE_INFINITY;
        for (CoordinatesIterator iter : iterators) {
            if (iter.pending == null || !iter.pending.getDate().isBefore(date)) continue;
            date = iter.pending.getDate();
        }
        return date;
    }

    private void writePosition(SP3Header header, String satId, SP3Coordinate coordinate) throws IOException {
        StringBuilder lineBuilder = new StringBuilder();
        lineBuilder.append(String.format(Locale.US, "P%3s%14.6f%14.6f%14.6f", satId, SP3Utils.POSITION_UNIT.fromSI(coordinate.getPosition().getX()), SP3Utils.POSITION_UNIT.fromSI(coordinate.getPosition().getY()), SP3Utils.POSITION_UNIT.fromSI(coordinate.getPosition().getZ())));
        lineBuilder.append(String.format(Locale.US, FOURTEEN_SIX_DIGITS_FLOAT, SP3Utils.CLOCK_UNIT.fromSI(coordinate.getClockCorrection())));
        if (coordinate.getPositionAccuracy() == null) {
            lineBuilder.append(THREE_BLANKS).append(THREE_BLANKS).append(THREE_BLANKS);
        } else {
            lineBuilder.append(' ');
            lineBuilder.append(String.format(Locale.US, TWO_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.POSITION_ACCURACY_UNIT, header.getPosVelBase(), coordinate.getPositionAccuracy().getX())));
            lineBuilder.append(' ');
            lineBuilder.append(String.format(Locale.US, TWO_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.POSITION_ACCURACY_UNIT, header.getPosVelBase(), coordinate.getPositionAccuracy().getY())));
            lineBuilder.append(' ');
            lineBuilder.append(String.format(Locale.US, TWO_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.POSITION_ACCURACY_UNIT, header.getPosVelBase(), coordinate.getPositionAccuracy().getZ())));
        }
        lineBuilder.append(' ');
        if (Double.isNaN(coordinate.getClockAccuracy())) {
            lineBuilder.append(THREE_BLANKS);
        } else {
            lineBuilder.append(String.format(Locale.US, THREE_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.CLOCK_ACCURACY_UNIT, header.getClockBase(), coordinate.getClockAccuracy())));
        }
        lineBuilder.append(' ');
        lineBuilder.append(coordinate.hasClockEvent() ? (char)'E' : ' ');
        lineBuilder.append(coordinate.hasClockPrediction() ? (char)'P' : ' ');
        lineBuilder.append(' ');
        lineBuilder.append(' ');
        lineBuilder.append(coordinate.hasOrbitManeuverEvent() ? (char)'M' : ' ');
        lineBuilder.append(coordinate.hasOrbitPrediction() ? (char)'P' : ' ');
        this.output.append(lineBuilder.toString().trim()).append(EOL);
    }

    private void writeVelocity(SP3Header header, String satId, SP3Coordinate coordinate) throws IOException {
        StringBuilder lineBuilder = new StringBuilder();
        lineBuilder.append(String.format(Locale.US, "V%3s%14.6f%14.6f%14.6f", satId, SP3Utils.VELOCITY_UNIT.fromSI(coordinate.getVelocity().getX()), SP3Utils.VELOCITY_UNIT.fromSI(coordinate.getVelocity().getY()), SP3Utils.VELOCITY_UNIT.fromSI(coordinate.getVelocity().getZ())));
        lineBuilder.append(String.format(Locale.US, FOURTEEN_SIX_DIGITS_FLOAT, SP3Utils.CLOCK_RATE_UNIT.fromSI(coordinate.getClockRateChange())));
        if (coordinate.getVelocityAccuracy() == null) {
            lineBuilder.append(THREE_BLANKS).append(THREE_BLANKS).append(THREE_BLANKS);
        } else {
            lineBuilder.append(' ');
            lineBuilder.append(String.format(Locale.US, TWO_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.VELOCITY_ACCURACY_UNIT, header.getPosVelBase(), coordinate.getVelocityAccuracy().getX())));
            lineBuilder.append(' ');
            lineBuilder.append(String.format(Locale.US, TWO_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.VELOCITY_ACCURACY_UNIT, header.getPosVelBase(), coordinate.getVelocityAccuracy().getY())));
            lineBuilder.append(' ');
            lineBuilder.append(String.format(Locale.US, TWO_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.VELOCITY_ACCURACY_UNIT, header.getPosVelBase(), coordinate.getVelocityAccuracy().getZ())));
        }
        lineBuilder.append(' ');
        if (Double.isNaN(coordinate.getClockRateAccuracy())) {
            lineBuilder.append(THREE_BLANKS);
        } else {
            lineBuilder.append(String.format(Locale.US, THREE_DIGITS_INTEGER, SP3Utils.indexAccuracy(SP3Utils.CLOCK_RATE_ACCURACY_UNIT, header.getClockBase(), coordinate.getClockRateAccuracy())));
        }
        this.output.append(lineBuilder.toString().trim()).append(EOL);
    }

    private void writeHeader(SP3Header header) throws IOException {
        TimeScale timeScale = header.getTimeSystem().getTimeScale(this.timeScales);
        DateTimeComponents dtc = header.getEpoch().getComponents(timeScale);
        StringBuilder dataUsedBuilder = new StringBuilder();
        for (DataUsed du : header.getDataUsed()) {
            if (dataUsedBuilder.length() > 0) {
                dataUsedBuilder.append('+');
            }
            dataUsedBuilder.append(du.getKey());
        }
        String dataUsed = dataUsedBuilder.length() <= 5 ? dataUsedBuilder.toString() : DataUsed.MIXED.getKey();
        this.output.append(String.format(Locale.US, "#%c%c%4d %2d %2d %2d %2d %11.8f %7d %5s %5s %3s %4s%n", Character.valueOf(header.getVersion()), Character.valueOf(header.getFilter() == CartesianDerivativesFilter.USE_P ? (char)'P' : 'V'), dtc.getDate().getYear(), dtc.getDate().getMonth(), dtc.getDate().getDay(), dtc.getTime().getHour(), dtc.getTime().getMinute(), dtc.getTime().getSecond(), header.getNumberOfEpochs(), dataUsed, header.getCoordinateSystem(), header.getOrbitTypeKey(), header.getAgency()));
        this.output.append(String.format(Locale.US, "## %4d %15.8f %14.8f %5d %15.13f%n", header.getGpsWeek(), header.getSecondsOfWeek(), header.getEpochInterval(), header.getModifiedJulianDay(), header.getDayFraction()));
        List<String> satellites = header.getSatIds();
        this.output.append(String.format(Locale.US, "+  %3d   ", satellites.size()));
        int lines = 0;
        int column = 9;
        int remaining = satellites.size();
        for (String satId : satellites) {
            this.output.append(String.format(Locale.US, "%3s", satId));
            if ((column += 3) < 60 || --remaining <= 0) continue;
            this.output.append(EOL);
            ++lines;
            this.output.append("+        ");
            column = 9;
        }
        while (column < 60) {
            this.output.append(' ').append(' ').append('0');
            column += 3;
        }
        this.output.append(EOL);
        while (true) {
            int n = ++lines;
            ++lines;
            if (n >= 5) break;
            this.output.append("+          0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0").append(EOL);
        }
        this.output.append(ACCURACY_LINE_PREFIX);
        lines = 0;
        column = 9;
        remaining = satellites.size();
        for (String satId : satellites) {
            double accuracy = header.getAccuracy(satId);
            int accuracyExp = SP3Utils.indexAccuracy(SP3Utils.POSITION_ACCURACY_UNIT, 2.0, accuracy);
            this.output.append(String.format(Locale.US, THREE_DIGITS_INTEGER, accuracyExp));
            if ((column += 3) < 60 || --remaining <= 0) continue;
            this.output.append(EOL);
            ++lines;
            this.output.append(ACCURACY_LINE_PREFIX);
            column = 9;
        }
        while (column < 60) {
            this.output.append(' ').append(' ').append('0');
            column += 3;
        }
        this.output.append(EOL);
        while (true) {
            int n = ++lines;
            ++lines;
            if (n >= 5) break;
            this.output.append("++         0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0  0").append(EOL);
        }
        if (header.getVersion() == 'a') {
            this.output.append(TIME_SYSTEM_DEFAULT).append(EOL);
        } else {
            TimeSystem ts = header.getTimeSystem().getKey() == null ? TimeSystem.UTC : header.getTimeSystem();
            this.output.append(String.format(Locale.US, "%%c %1s  cc %3s ccc cccc cccc cccc cccc ccccc ccccc ccccc ccccc%n", header.getType().getKey(), ts.getKey()));
        }
        this.output.append(TIME_SYSTEM_DEFAULT).append(EOL);
        this.output.append(String.format(Locale.US, ACCURACY_BASE_FORMAT, header.getPosVelBase(), header.getClockBase(), 0.0, 0.0));
        this.output.append(String.format(Locale.US, ACCURACY_BASE_FORMAT, 0.0, 0.0, 0.0, 0.0));
        this.output.append(ADDITIONAL_PARAMETERS_LINE).append(EOL);
        this.output.append(ADDITIONAL_PARAMETERS_LINE).append(EOL);
        int count = 0;
        for (String comment : header.getComments()) {
            ++count;
            this.output.append(COMMENT_LINE_PREFIX).append(comment).append(EOL);
        }
        while (count < 4) {
            ++count;
            this.output.append(COMMENT_LINE_PREFIX).append(EOL);
        }
    }

    private static class CoordinatesIterator {
        private final String id;
        private Iterator<SP3Segment> segmentsIterator;
        private Iterator<SP3Coordinate> coordinatesIterator;
        private SP3Coordinate pending;

        CoordinatesIterator(SP3Ephemeris ephemeris) {
            this.id = ephemeris.getId();
            this.segmentsIterator = ephemeris.getSegments().iterator();
            this.coordinatesIterator = null;
            this.advance();
        }

        private void advance() {
            while (this.coordinatesIterator == null || !this.coordinatesIterator.hasNext()) {
                if (this.segmentsIterator != null && this.segmentsIterator.hasNext()) {
                    this.coordinatesIterator = this.segmentsIterator.next().getCoordinates().iterator();
                    continue;
                }
                this.segmentsIterator = null;
                this.pending = null;
                return;
            }
            this.pending = this.coordinatesIterator.next();
        }
    }
}

