/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.ilrs;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import org.hipparchus.exception.Localizable;
import org.hipparchus.exception.LocalizedCoreFormats;
import org.hipparchus.util.FastMath;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.data.DataSource;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.ilrs.CRD;
import org.orekit.files.ilrs.CRDConfiguration;
import org.orekit.files.ilrs.CRDHeader;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateComponents;
import org.orekit.time.TimeComponents;
import org.orekit.time.TimeScale;
import org.orekit.utils.units.Unit;
import org.orekit.utils.units.UnitsConverter;

public class CRDParser {
    public static final String DEFAULT_CRD_SUPPORTED_NAMES = "^(?!0+$)\\w{1,12}\\_\\d{6,8}.\\w{3}$";
    private static final Unit NM = Unit.parse("nm");
    private static final Unit KHZ = Unit.parse("kHz");
    private static final Unit US = Unit.parse("\u00b5s");
    private static final Unit NS = Unit.parse("ns");
    private static final Unit PS = Unit.parse("ps");
    private static final UnitsConverter MBAR_TO_BAR = new UnitsConverter(Unit.parse("mbar"), Unit.parse("bar"));
    private static final String FILE_FORMAT = "CRD";
    private static final Pattern SEPARATOR = Pattern.compile("\\s+");
    private static final Pattern COMMA = Pattern.compile(",");
    private static final String COMMENTS_IDENTIFIER = "00";
    private static final Pattern PATTERN_NA = Pattern.compile(" [-]?(na)");
    private final TimeScale timeScale;

    @DefaultDataContext
    public CRDParser() {
        this(DataContext.getDefault().getTimeScales().getUTC());
    }

    public CRDParser(TimeScale utc) {
        this.timeScale = utc;
    }

    public TimeScale getTimeScale() {
        return this.timeScale;
    }

    /*
     * WARNING - Removed back jump from a try to a catch block - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public CRD parse(DataSource source) throws IOException {
        ParseInfo pi = new ParseInfo();
        int lineNumber = 0;
        Iterable<LineParser> crdParsers = Collections.singleton(LineParser.H1);
        try (BufferedReader reader = new BufferedReader(source.getOpener().openReaderOnce());){
            String line = reader.readLine();
            while (line != null) {
                ++lineNumber;
                if (line.startsWith(COMMENTS_IDENTIFIER)) {
                    crdParsers = Arrays.asList(LineParser.COMMENTS);
                }
                for (LineParser candidate : crdParsers) {
                    if (!candidate.canHandle(line)) continue;
                    try {
                        line = PATTERN_NA.matcher(line).replaceAll(" NaN");
                        candidate.parse(line, pi);
                        if (pi.done) {
                            CRD cRD = pi.file;
                            return cRD;
                        }
                    }
                    catch (NumberFormatException | StringIndexOutOfBoundsException e) {
                        throw new OrekitException(e, OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, source.getName(), line);
                    }
                    {
                        crdParsers = candidate.allowedNext();
                        break;
                    }
                }
                line = reader.readLine();
            }
            throw new OrekitException((Localizable)OrekitMessages.CRD_UNEXPECTED_END_OF_FILE, lineNumber);
        }
        catch (IOException ioe) {
            throw new OrekitException(ioe, (Localizable)LocalizedCoreFormats.SIMPLE_MESSAGE, ioe.getLocalizedMessage());
        }
    }

    private static AbsoluteDate checkRollover(AbsoluteDate epoch, AbsoluteDate startEpoch) {
        return epoch.durationFrom(startEpoch) < -36000.0 ? epoch.shiftedBy(86400.0) : epoch;
    }

    private static enum LineParser {
        H1(new String[]{"H1", "h1"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                String format = values[1];
                pi.version = Integer.parseInt(values[2]);
                if (!format.equalsIgnoreCase(CRDParser.FILE_FORMAT)) {
                    throw new OrekitException((Localizable)OrekitMessages.UNEXPECTED_FORMAT_FOR_ILRS_FILE, CRDParser.FILE_FORMAT, format);
                }
                pi.header.setFormat(format);
                pi.header.setVersion(pi.version);
                int year = Integer.parseInt(values[3]);
                int month = Integer.parseInt(values[4]);
                int day = Integer.parseInt(values[5]);
                pi.header.setProductionEpoch(new DateComponents(year, month, day));
                pi.header.setProductionHour(Integer.parseInt(values[6]));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H2, COMMENTS);
            }
        }
        ,
        H2(new String[]{"H2", "h2"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setStationName(values[1]);
                pi.header.setSystemIdentifier(Integer.parseInt(values[2]));
                pi.header.setSystemNumber(Integer.parseInt(values[3]));
                pi.header.setSystemOccupancy(Integer.parseInt(values[4]));
                pi.header.setEpochIdentifier(Integer.parseInt(values[5]));
                if (pi.version == 2) {
                    pi.header.setStationNetword(values[6]);
                } else {
                    pi.header.setStationNetword("na");
                }
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, C0, C1, C2, C3, C4, C5, C6, C7, COMMENTS);
            }
        }
        ,
        H3(new String[]{"H3", "h3"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setName(values[1]);
                pi.header.setIlrsSatelliteId(values[2]);
                pi.header.setSic(values[3]);
                pi.header.setNoradId(values[4]);
                pi.header.setSpacecraftEpochTimeScale(Integer.parseInt(values[5]));
                pi.header.setTargetClass(Integer.parseInt(values[6]));
                if (pi.version == 2) {
                    pi.header.setTargetLocation(LineParser.readIntegerWithNaN(values[7], -1));
                }
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H4, C0, C1, C2, C3, C4, C5, C6, C7, COMMENTS);
            }
        }
        ,
        H4(new String[]{"H4", "h4"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setDataType(Integer.parseInt(values[1]));
                int yearS = Integer.parseInt(values[2]);
                int monthS = Integer.parseInt(values[3]);
                int dayS = Integer.parseInt(values[4]);
                int hourS = Integer.parseInt(values[5]);
                int minuteS = Integer.parseInt(values[6]);
                double secondS = Integer.parseInt(values[7]);
                pi.startEpochDateComponents = new DateComponents(yearS, monthS, dayS);
                pi.header.setStartEpoch(new AbsoluteDate(yearS, monthS, dayS, hourS, minuteS, secondS, pi.timeScale));
                if (pi.version == 2 && values[8].equalsIgnoreCase("")) {
                    pi.header.setEndEpoch(null);
                } else {
                    int yearE = Integer.parseInt(values[8]);
                    int monthE = Integer.parseInt(values[9]);
                    int dayE = Integer.parseInt(values[10]);
                    int hourE = Integer.parseInt(values[11]);
                    int minuteE = Integer.parseInt(values[12]);
                    double secondE = Integer.parseInt(values[13]);
                    if (monthE == -1) {
                        pi.header.setEndEpoch(null);
                    } else {
                        pi.header.setEndEpoch(new AbsoluteDate(yearE, monthE, dayE, hourE, minuteE, secondE, pi.timeScale));
                    }
                }
                pi.header.setDataReleaseFlag(Integer.parseInt(values[14]));
                pi.header.setIsTroposphericRefractionApplied(LineParser.readBoolean(values[15]));
                pi.header.setIsCenterOfMassCorrectionApplied(LineParser.readBoolean(values[16]));
                pi.header.setIsReceiveAmplitudeCorrectionApplied(LineParser.readBoolean(values[17]));
                pi.header.setIsStationSystemDelayApplied(LineParser.readBoolean(values[18]));
                pi.header.setIsTransponderDelayApplied(LineParser.readBoolean(values[19]));
                pi.header.setRangeType(Integer.parseInt(values[20]));
                pi.header.setQualityIndicator(Integer.parseInt(values[21]));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        H5(new String[]{"H5", "h5"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setPredictionType(Integer.parseInt(values[1]));
                pi.header.setYearOfCentury(Integer.parseInt(values[2]));
                pi.header.setDateAndTime(values[3]);
                pi.header.setPredictionProvider(values[4]);
                pi.header.setSequenceNumber(Integer.parseInt(values[5]));
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C0(new String[]{"C0", "c0"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.SystemConfiguration systemRecord = new CRDConfiguration.SystemConfiguration();
                String[] values = SEPARATOR.split(line);
                systemRecord.setWavelength(NM.toSI(Double.parseDouble(values[2])));
                systemRecord.setSystemId(values[3]);
                systemRecord.setComponents(Arrays.copyOfRange(values, 4, values.length));
                pi.configurationRecords.addConfigurationRecord(systemRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C1(new String[]{"C1", "c1"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.LaserConfiguration laserRecord = new CRDConfiguration.LaserConfiguration();
                String[] values = SEPARATOR.split(line);
                laserRecord.setLaserId(values[2]);
                laserRecord.setLaserType(values[3]);
                laserRecord.setPrimaryWavelength(NM.toSI(Double.parseDouble(values[4])));
                laserRecord.setNominalFireRate(Double.parseDouble(values[5]));
                laserRecord.setPulseEnergy(Double.parseDouble(values[6]));
                laserRecord.setPulseWidth(Double.parseDouble(values[7]));
                laserRecord.setBeamDivergence(Double.parseDouble(values[8]));
                laserRecord.setPulseInOutgoingSemiTrain(LineParser.readIntegerWithNaN(values[9], 1));
                pi.configurationRecords.addConfigurationRecord(laserRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(C2, C3, C4, C5, C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C2(new String[]{"C2", "c2"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.DetectorConfiguration detectorRecord = new CRDConfiguration.DetectorConfiguration();
                String[] values = SEPARATOR.split(line);
                detectorRecord.setDetectorId(values[2]);
                detectorRecord.setDetectorType(values[3]);
                detectorRecord.setApplicableWavelength(NM.toSI(Double.parseDouble(values[4])));
                detectorRecord.setQuantumEfficiency(Double.parseDouble(values[5]));
                detectorRecord.setAppliedVoltage(Double.parseDouble(values[6]));
                detectorRecord.setDarkCount(KHZ.toSI(Double.parseDouble(values[7])));
                detectorRecord.setOutputPulseType(values[8]);
                detectorRecord.setOutputPulseWidth(Double.parseDouble(values[9]));
                detectorRecord.setSpectralFilter(NM.toSI(Double.parseDouble(values[10])));
                detectorRecord.setTransmissionOfSpectralFilter(Double.parseDouble(values[11]));
                detectorRecord.setSpatialFilter(Double.parseDouble(values[12]));
                detectorRecord.setExternalSignalProcessing(values[13]);
                if (pi.version == 2) {
                    detectorRecord.setAmplifierGain(Double.parseDouble(values[14]));
                    detectorRecord.setAmplifierBandwidth(KHZ.toSI(Double.parseDouble(values[15])));
                    detectorRecord.setAmplifierInUse(values[16]);
                } else {
                    detectorRecord.setAmplifierGain(Double.NaN);
                    detectorRecord.setAmplifierBandwidth(Double.NaN);
                    detectorRecord.setAmplifierInUse("na");
                }
                pi.configurationRecords.addConfigurationRecord(detectorRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C3(new String[]{"C3", "c3"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.TimingSystemConfiguration timingRecord = new CRDConfiguration.TimingSystemConfiguration();
                String[] values = SEPARATOR.split(line);
                timingRecord.setLocalTimingId(values[2]);
                timingRecord.setTimeSource(values[3]);
                timingRecord.setFrequencySource(values[4]);
                timingRecord.setTimer(values[5]);
                String timerSerialNumber = values[6];
                if ("NaN".equalsIgnoreCase(timerSerialNumber)) {
                    timingRecord.setTimerSerialNumber("na");
                } else {
                    timingRecord.setTimerSerialNumber(timerSerialNumber);
                }
                timingRecord.setEpochDelayCorrection(US.toSI(Double.parseDouble(values[7])));
                pi.configurationRecords.addConfigurationRecord(timingRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C4(new String[]{"C4", "c4"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.TransponderConfiguration transponderRecord = new CRDConfiguration.TransponderConfiguration();
                String[] values = SEPARATOR.split(line);
                transponderRecord.setTransponderId(values[2]);
                transponderRecord.setStationUTCOffset(NS.toSI(Double.parseDouble(values[3])));
                transponderRecord.setStationOscDrift(Double.parseDouble(values[4]));
                transponderRecord.setTranspUTCOffset(NS.toSI(Double.parseDouble(values[5])));
                transponderRecord.setTranspOscDrift(Double.parseDouble(values[6]));
                transponderRecord.setTranspClkRefTime(Double.parseDouble(values[7]));
                transponderRecord.setStationClockAndDriftApplied(Integer.parseInt(values[8]));
                transponderRecord.setSpacecraftClockAndDriftApplied(Integer.parseInt(values[9]));
                transponderRecord.setIsSpacecraftTimeSimplified(LineParser.readBoolean(values[10]));
                pi.configurationRecords.addConfigurationRecord(transponderRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C5(new String[]{"C5", "c5"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.SoftwareConfiguration softwareRecord = new CRDConfiguration.SoftwareConfiguration();
                String[] values = SEPARATOR.split(line);
                softwareRecord.setSoftwareId(values[2]);
                softwareRecord.setTrackingSoftwares(COMMA.split(values[3]));
                softwareRecord.setTrackingSoftwareVersions(COMMA.split(values[4]));
                softwareRecord.setProcessingSoftwares(COMMA.split(values[5]));
                softwareRecord.setProcessingSoftwareVersions(COMMA.split(values[6]));
                pi.configurationRecords.addConfigurationRecord(softwareRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C6(new String[]{"C6", "c6"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.MeteorologicalConfiguration meteoRecord = new CRDConfiguration.MeteorologicalConfiguration();
                String[] values = SEPARATOR.split(line);
                meteoRecord.setMeteorologicalId(values[2]);
                meteoRecord.setPressSensorManufacturer(values[3]);
                meteoRecord.setPressSensorModel(values[4]);
                meteoRecord.setPressSensorSerialNumber(values[5]);
                meteoRecord.setTempSensorManufacturer(values[6]);
                meteoRecord.setTempSensorModel(values[7]);
                meteoRecord.setTempSensorSerialNumber(values[8]);
                meteoRecord.setHumiSensorManufacturer(values[9]);
                meteoRecord.setHumiSensorModel(values[10]);
                meteoRecord.setHumiSensorSerialNumber(values[11]);
                pi.configurationRecords.addConfigurationRecord(meteoRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        C7(new String[]{"C7", "c7"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.CalibrationTargetConfiguration calibRecord = new CRDConfiguration.CalibrationTargetConfiguration();
                String[] values = SEPARATOR.split(line);
                calibRecord.setConfigurationId(values[2]);
                calibRecord.setTargetName(values[3]);
                calibRecord.setSurveyedTargetDistance(Double.parseDouble(values[4]));
                calibRecord.setSurveyError(Double.parseDouble(values[5]) * 0.001);
                calibRecord.setSumOfAllConstantDelays(Double.parseDouble(values[6]));
                calibRecord.setPulseEnergy(Double.parseDouble(values[7]));
                calibRecord.setProcessingSoftwareName(values[8]);
                calibRecord.setProcessingSoftwareVersion(values[9]);
                pi.configurationRecords.addConfigurationRecord(calibRecord);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H3, H4, H5, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        TEN(new String[]{"10"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double timeOfFlight = Double.parseDouble(values[2]);
                String systemConfigId = values[3];
                int epochEvent = Integer.parseInt(values[4]);
                int filterFlag = Integer.parseInt(values[5]);
                int detectorChannel = Integer.parseInt(values[6]);
                int stopNumber = Integer.parseInt(values[7]);
                int receiveAmplitude = LineParser.readIntegerWithNaN(values[8], -1);
                int transmitAmplitude = -1;
                if (pi.version == 2) {
                    transmitAmplitude = LineParser.readIntegerWithNaN(values[9], -1);
                }
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.FrRangeMeasurement range = new CRD.FrRangeMeasurement(epoch, timeOfFlight, epochEvent, systemConfigId, filterFlag, detectorChannel, stopNumber, receiveAmplitude, transmitAmplitude);
                pi.dataBlock.addRangeData(range);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        ELEVEN(new String[]{"11"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double timeOfFlight = Double.parseDouble(values[2]);
                String systemConfigId = values[3];
                int epochEvent = Integer.parseInt(values[4]);
                double windowLength = Double.parseDouble(values[5]);
                int numberOfRawRanges = Integer.parseInt(values[6]);
                double binRms = PS.toSI(Double.parseDouble(values[7]));
                double binSkew = Double.parseDouble(values[8]);
                double binKurtosis = Double.parseDouble(values[9]);
                double binPeakMinusMean = PS.toSI(Double.parseDouble(values[10]));
                double returnRate = Double.parseDouble(values[11]);
                int detectorChannel = Integer.parseInt(values[12]);
                double snr = Double.NaN;
                if (pi.version == 2) {
                    snr = Double.parseDouble(values[13]);
                }
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.NptRangeMeasurement range = new CRD.NptRangeMeasurement(epoch, timeOfFlight, epochEvent, snr, systemConfigId, windowLength, numberOfRawRanges, binRms, binSkew, binKurtosis, binPeakMinusMean, returnRate, detectorChannel);
                pi.dataBlock.addRangeData(range);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        TWELVE(new String[]{"12"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                String systemConfigId = values[2];
                double troposphericRefractionCorr = PS.toSI(Double.parseDouble(values[3]));
                double centerOfMassCorr = Double.parseDouble(values[4]);
                double ndFilterValue = Double.parseDouble(values[5]);
                double timeBiasApplied = Double.parseDouble(values[6]);
                double rangeRate = Double.NaN;
                if (pi.version == 2) {
                    rangeRate = Double.parseDouble(values[7]);
                }
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.RangeSupplement rangeSup = new CRD.RangeSupplement(epoch, systemConfigId, troposphericRefractionCorr, centerOfMassCorr, ndFilterValue, timeBiasApplied, rangeRate);
                pi.dataBlock.addRangeSupplementData(rangeSup);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        METEO(new String[]{"20"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double pressure = MBAR_TO_BAR.convert(Double.parseDouble(values[2]));
                double temperature = Double.parseDouble(values[3]);
                double humidity = Double.parseDouble(values[4]);
                int originOfValues = Integer.parseInt(values[5]);
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.MeteorologicalMeasurement meteo = new CRD.MeteorologicalMeasurement(epoch, pressure, temperature, humidity, originOfValues);
                pi.dataBlock.addMeteoData(meteo);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        METEO_SUPP(new String[]{"21"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        ANGLES(new String[]{"30"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double azmiuth = FastMath.toRadians((double)Double.parseDouble(values[2]));
                double elevation = FastMath.toRadians((double)Double.parseDouble(values[3]));
                int directionFlag = Integer.parseInt(values[4]);
                int orginFlag = Integer.parseInt(values[5]);
                boolean isRefractionCorrected = LineParser.readBoolean(values[6]);
                double azimuthRate = Double.NaN;
                double elevationRate = Double.NaN;
                if (pi.version == 2) {
                    azimuthRate = FastMath.toRadians((double)Double.parseDouble(values[7]));
                    elevationRate = FastMath.toRadians((double)Double.parseDouble(values[8]));
                }
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.AnglesMeasurement angles = new CRD.AnglesMeasurement(epoch, azmiuth, elevation, directionFlag, orginFlag, isRefractionCorrected, azimuthRate, elevationRate);
                pi.dataBlock.addAnglesData(angles);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        CALIB(new String[]{"40"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                int typeOfData = Integer.parseInt(values[2]);
                String systemConfigId = values[3];
                int numberOfPointsRecorded = LineParser.readIntegerWithNaN(values[4], -1);
                int numberOfPointsUsed = LineParser.readIntegerWithNaN(values[5], -1);
                double oneWayDistance = Double.parseDouble(values[6]);
                double systemDelay = PS.toSI(Double.parseDouble(values[7]));
                double delayShift = PS.toSI(Double.parseDouble(values[8]));
                double rms = PS.toSI(Double.parseDouble(values[9]));
                double skew = Double.parseDouble(values[10]);
                double kurtosis = Double.parseDouble(values[11]);
                double peakMinusMean = PS.toSI(Double.parseDouble(values[12]));
                int typeIndicator = Integer.parseInt(values[13]);
                int shiftTypeIndicator = Integer.parseInt(values[14]);
                int detectorChannel = Integer.parseInt(values[15]);
                int span = 0;
                double returnRate = Double.NaN;
                if (pi.version == 2) {
                    span = LineParser.readIntegerWithNaN(values[16], -1);
                    returnRate = Double.parseDouble(values[17]);
                }
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.Calibration cal = new CRD.Calibration(epoch, typeOfData, systemConfigId, numberOfPointsRecorded, numberOfPointsUsed, oneWayDistance, systemDelay, delayShift, rms, skew, kurtosis, peakMinusMean, typeIndicator, shiftTypeIndicator, detectorChannel, span, returnRate);
                pi.dataBlock.addCalibrationData(cal);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        CALIB_DETAILS(new String[]{"41"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                int typeOfData = Integer.parseInt(values[2]);
                String systemConfigId = values[3];
                int numberOfPointsRecorded = LineParser.readIntegerWithNaN(values[4], -1);
                int numberOfPointsUsed = LineParser.readIntegerWithNaN(values[5], -1);
                double oneWayDistance = Double.parseDouble(values[6]);
                double systemDelay = PS.toSI(Double.parseDouble(values[7]));
                double delayShift = PS.toSI(Double.parseDouble(values[8]));
                double rms = PS.toSI(Double.parseDouble(values[9]));
                double skew = Double.parseDouble(values[10]);
                double kurtosis = Double.parseDouble(values[11]);
                double peakMinusMean = PS.toSI(Double.parseDouble(values[12]));
                int typeIndicator = Integer.parseInt(values[13]);
                int shiftTypeIndicator = Integer.parseInt(values[14]);
                int detectorChannel = Integer.parseInt(values[15]);
                int span = 0;
                double returnRate = Double.NaN;
                if (pi.version == 2) {
                    span = Integer.parseInt(values[16]);
                    returnRate = Double.parseDouble(values[17]);
                }
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpochDateComponents, new TimeComponents(secOfDay), pi.timeScale);
                epoch = CRDParser.checkRollover(epoch, pi.header.getStartEpoch());
                CRD.CalibrationDetail cal = new CRD.CalibrationDetail(epoch, typeOfData, systemConfigId, numberOfPointsRecorded, numberOfPointsUsed, oneWayDistance, systemDelay, delayShift, rms, skew, kurtosis, peakMinusMean, typeIndicator, shiftTypeIndicator, detectorChannel, span, returnRate);
                pi.dataBlock.addCalibrationDetailData(cal);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        CALIB_SHOT(new String[]{"42"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        STAT(new String[]{"50"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                String systemConfigId = values[1];
                double rms = PS.toSI(Double.parseDouble(values[2]));
                double skewness = Double.parseDouble(values[3]);
                double kurtosis = Double.parseDouble(values[4]);
                double peakMinusMean = values[5].contains("*") ? Double.NaN : PS.toSI(Double.parseDouble(values[5]));
                int dataQualityIndicator = Integer.parseInt(values[6]);
                CRD.SessionStatistics stat = new CRD.SessionStatistics(systemConfigId, rms, skewness, kurtosis, peakMinusMean, dataQualityIndicator);
                pi.dataBlock.addSessionStatisticsData(stat);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        COMPATIBILITY(new String[]{"60"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        COMMENTS(new String[]{"00"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String comment = line.substring(2).trim();
                pi.file.getComments().add(comment);
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H1, H2, H3, H4, H5, H8, H9, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        CUSTOM(new String[]{"9\\d"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H8, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS, CUSTOM);
            }
        }
        ,
        H8(new String[]{"H8", "h8"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                if (pi.header.getEndEpoch() == null) {
                    List<CRD.RangeMeasurement> rangeData = pi.dataBlock.getRangeData();
                    pi.header.setEndEpoch(rangeData.get(rangeData.size() - 1).getDate());
                }
                pi.dataBlock.setHeader(pi.header);
                pi.dataBlock.setConfigurationRecords(pi.configurationRecords);
                pi.file.addDataBlock(pi.dataBlock);
                pi.startEpochDateComponents = DateComponents.J2000_EPOCH;
                CRDHeader lastHeader = pi.header;
                pi.header = new CRDHeader();
                pi.configurationRecords = new CRDConfiguration();
                pi.dataBlock = new CRD.CRDDataBlock();
                pi.header.setFormat(lastHeader.getFormat());
                pi.header.setVersion(lastHeader.getVersion());
                pi.header.setProductionEpoch(lastHeader.getProductionEpoch());
                pi.header.setProductionHour(lastHeader.getProductionHour());
                pi.header.setStationName(lastHeader.getStationName());
                pi.header.setSystemIdentifier(lastHeader.getSystemIdentifier());
                pi.header.setSystemNumber(lastHeader.getSystemNumber());
                pi.header.setSystemOccupancy(lastHeader.getSystemOccupancy());
                pi.header.setEpochIdentifier(lastHeader.getEpochIdentifier());
                pi.header.setStationNetword(lastHeader.getStationNetword());
                pi.header.setName(lastHeader.getName());
                pi.header.setIlrsSatelliteId(lastHeader.getIlrsSatelliteId());
                pi.header.setSic(lastHeader.getSic());
                pi.header.setNoradId(lastHeader.getNoradId());
                pi.header.setSpacecraftEpochTimeScale(lastHeader.getSpacecraftEpochTimeScale());
                pi.header.setTargetClass(lastHeader.getTargetClass());
                pi.header.setTargetLocation(lastHeader.getTargetLocation());
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Arrays.asList(H1, H4, H9, COMMENTS);
            }
        }
        ,
        H9(new String[]{"H9", "h9"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.done = true;
            }

            @Override
            public Iterable<LineParser> allowedNext() {
                return Collections.singleton(H9);
            }
        };

        private final Pattern[] patterns;
        private final String[] identifiers;

        private LineParser(String ... identifier) {
            this.identifiers = identifier;
            this.patterns = new Pattern[this.identifiers.length];
            for (int index = 0; index < this.patterns.length; ++index) {
                this.patterns[index] = Pattern.compile(this.identifiers[index]);
            }
        }

        public abstract void parse(String var1, ParseInfo var2);

        public abstract Iterable<LineParser> allowedNext();

        public boolean canHandle(String line) {
            String lineId = SEPARATOR.split(line)[0];
            for (Pattern pattern : this.patterns) {
                if (!pattern.matcher(lineId).matches()) continue;
                return true;
            }
            return false;
        }

        private static boolean readBoolean(String value) {
            return Integer.parseInt(value) == 1;
        }

        private static int readIntegerWithNaN(String value, int defaultValue) {
            return "NaN".equalsIgnoreCase(value) ? defaultValue : Integer.parseInt(value);
        }
    }

    private class ParseInfo {
        private CRD file;
        private int version = 1;
        private CRD.CRDDataBlock dataBlock;
        private CRDHeader header;
        private CRDConfiguration configurationRecords;
        private TimeScale timeScale;
        private DateComponents startEpochDateComponents = DateComponents.J2000_EPOCH;
        private boolean done = false;

        protected ParseInfo() {
            this.file = new CRD();
            this.header = new CRDHeader();
            this.configurationRecords = new CRDConfiguration();
            this.dataBlock = new CRD.CRDDataBlock();
            this.timeScale = CRDParser.this.timeScale;
        }
    }
}

