/*
 * Decompiled with CFR 0.152.
 */
package org.hipparchus.linear;

import java.lang.reflect.Array;
import org.hipparchus.FieldElement;
import org.hipparchus.complex.Complex;
import org.hipparchus.complex.ComplexField;
import org.hipparchus.exception.LocalizedCoreFormats;
import org.hipparchus.exception.MathIllegalArgumentException;
import org.hipparchus.exception.MathRuntimeException;
import org.hipparchus.linear.FieldDecompositionSolver;
import org.hipparchus.linear.FieldMatrix;
import org.hipparchus.linear.FieldQRDecomposition;
import org.hipparchus.linear.FieldVector;
import org.hipparchus.linear.MatrixUtils;
import org.hipparchus.linear.RealMatrix;
import org.hipparchus.linear.SchurTransformer;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.Precision;

public class ComplexEigenDecomposition {
    public static final double DEFAULT_EIGENVECTORS_EQUALITY = 1.0E-5;
    public static final double DEFAULT_EPSILON = 1.0E-12;
    public static final double DEFAULT_EPSILON_AV_VD_CHECK = 1.0E-6;
    private static final int MAX_ITER = 10;
    private Complex[] eigenvalues;
    private FieldVector<Complex>[] eigenvectors;
    private FieldMatrix<Complex> V;
    private FieldMatrix<Complex> D;
    private final double eigenVectorsEquality;
    private final double epsilon;
    private final double epsilonAVVDCheck;

    public ComplexEigenDecomposition(RealMatrix matrix) {
        this(matrix, 1.0E-5, 1.0E-12, 1.0E-6);
    }

    public ComplexEigenDecomposition(RealMatrix matrix, double eigenVectorsEquality, double epsilon, double epsilonAVVDCheck) {
        if (!matrix.isSquare()) {
            throw new MathIllegalArgumentException(LocalizedCoreFormats.NON_SQUARE_MATRIX, matrix.getRowDimension(), matrix.getColumnDimension());
        }
        this.eigenVectorsEquality = eigenVectorsEquality;
        this.epsilon = epsilon;
        this.epsilonAVVDCheck = epsilonAVVDCheck;
        this.findEigenValues(matrix);
        this.findEigenVectors(this.convertToFieldComplex(matrix));
        int m = this.eigenvectors.length;
        this.V = MatrixUtils.createFieldMatrix(ComplexField.getInstance(), m, m);
        for (int k = 0; k < m; ++k) {
            this.V.setColumnVector(k, this.eigenvectors[k]);
        }
        this.D = MatrixUtils.createFieldDiagonalMatrix((FieldElement[])this.eigenvalues);
        this.checkDefinition(matrix);
    }

    public Complex[] getEigenvalues() {
        return (Complex[])this.eigenvalues.clone();
    }

    public FieldVector<Complex> getEigenvector(int i) {
        return this.eigenvectors[i].copy();
    }

    protected void matricesToEigenArrays() {
        int i;
        for (i = 0; i < this.eigenvalues.length; ++i) {
            this.eigenvalues[i] = this.D.getEntry(i, i);
        }
        for (i = 0; i < this.eigenvectors.length; ++i) {
            for (int j = 0; j < this.eigenvectors[i].getDimension(); ++j) {
                this.eigenvectors[i].setEntry(j, this.V.getEntry(j, i));
            }
        }
    }

    public boolean hasComplexEigenvalues() {
        for (int i = 0; i < this.eigenvalues.length; ++i) {
            if (Precision.equals(this.eigenvalues[i].getImaginary(), 0.0, this.epsilon)) continue;
            return true;
        }
        return false;
    }

    public double getDeterminant() {
        Complex determinant = new Complex(1.0, 0.0);
        for (Complex lambda : this.eigenvalues) {
            determinant = determinant.multiply(lambda);
        }
        return determinant.getReal();
    }

    public FieldMatrix<Complex> getV() {
        return this.V;
    }

    public FieldMatrix<Complex> getD() {
        return this.D;
    }

    public FieldMatrix<Complex> getVT() {
        return this.V.transpose();
    }

    protected void findEigenValues(RealMatrix matrix) {
        SchurTransformer schurTransform = new SchurTransformer(matrix);
        double[][] matT = schurTransform.getT().getData();
        this.eigenvalues = new Complex[matT.length];
        for (int i = 0; i < this.eigenvalues.length; ++i) {
            if (i == this.eigenvalues.length - 1 || Precision.equals(matT[i + 1][i], 0.0, this.epsilon)) {
                this.eigenvalues[i] = new Complex(matT[i][i]);
                continue;
            }
            double x = matT[i + 1][i + 1];
            double p = 0.5 * (matT[i][i] - x);
            double z = FastMath.sqrt(FastMath.abs(p * p + matT[i + 1][i] * matT[i][i + 1]));
            this.eigenvalues[i] = new Complex(x + p, z);
            this.eigenvalues[i + 1] = new Complex(x + p, -z);
            ++i;
        }
    }

    protected void findEigenVectors(FieldMatrix<Complex> matrix) {
        int n = this.eigenvalues.length;
        this.eigenvectors = (FieldVector[])Array.newInstance(FieldVector.class, n);
        for (int i = 0; i < this.eigenvalues.length; ++i) {
            Complex mu = this.eigenvalues[i].add(this.epsilon);
            FieldMatrix<Complex> shifted = matrix.copy();
            for (int k = 0; k < matrix.getColumnDimension(); ++k) {
                shifted.setEntry(k, k, shifted.getEntry(k, k).subtract(mu));
            }
            FieldDecompositionSolver<Complex> solver = new FieldQRDecomposition<Complex>(shifted).getSolver();
            for (int p = 0; this.eigenvectors[i] == null && p < matrix.getColumnDimension(); ++p) {
                FieldVector<Complex> b = this.findStart(p);
                if (!(this.getNorm(b).norm() > Precision.SAFE_MIN)) continue;
                double delta = Double.POSITIVE_INFINITY;
                for (int k = 0; delta > this.epsilon && k < 10; ++k) {
                    FieldVector<Complex> bNext = solver.solve(b);
                    this.normalize(bNext);
                    delta = this.separation(b, bNext);
                    b = bNext;
                }
                for (int j = 0; b != null && j < i; ++j) {
                    if (!(this.separation(this.eigenvectors[j], b) <= this.eigenVectorsEquality)) continue;
                    b = null;
                }
                this.eigenvectors[i] = b;
            }
            if (this.eigenvectors[i] != null) continue;
            this.eigenvectors[i] = MatrixUtils.createFieldVector(ComplexField.getInstance(), n);
        }
    }

    private FieldVector<Complex> findStart(int index) {
        FieldVector<Complex> start = MatrixUtils.createFieldVector(ComplexField.getInstance(), this.eigenvalues.length);
        start.setEntry(index, Complex.ONE);
        return start;
    }

    private Complex getNorm(FieldVector<Complex> vector) {
        double normR = 0.0;
        Complex normC = Complex.ZERO;
        for (int i = 0; i < vector.getDimension(); ++i) {
            Complex ci = vector.getEntry(i);
            double ni = FastMath.hypot(ci.getReal(), ci.getImaginary());
            if (!(ni > normR)) continue;
            normR = ni;
            normC = ci;
        }
        return normC;
    }

    private void normalize(FieldVector<Complex> v) {
        Complex invNorm = this.getNorm(v).reciprocal();
        for (int j = 0; j < v.getDimension(); ++j) {
            v.setEntry(j, v.getEntry(j).multiply(invNorm));
        }
    }

    private double separation(FieldVector<Complex> v1, FieldVector<Complex> v2) {
        double deltaPlus = 0.0;
        double deltaMinus = 0.0;
        for (int j = 0; j < v1.getDimension(); ++j) {
            Complex bCurrj = v1.getEntry(j);
            Complex bNextj = v2.getEntry(j);
            deltaPlus = FastMath.max(deltaPlus, FastMath.hypot(bNextj.getReal() + bCurrj.getReal(), bNextj.getImaginary() + bCurrj.getImaginary()));
            deltaMinus = FastMath.max(deltaMinus, FastMath.hypot(bNextj.getReal() - bCurrj.getReal(), bNextj.getImaginary() - bCurrj.getImaginary()));
        }
        return FastMath.min(deltaPlus, deltaMinus);
    }

    protected void checkDefinition(RealMatrix matrix) {
        FieldMatrix<Complex> VD;
        FieldMatrix<Complex> matrixC = this.convertToFieldComplex(matrix);
        FieldMatrix<Complex> AV = matrixC.multiply(this.getV());
        if (!this.equalsWithPrecision(AV, VD = this.getV().multiply(this.getD()), this.epsilonAVVDCheck)) {
            throw new MathRuntimeException(LocalizedCoreFormats.FAILED_DECOMPOSITION, matrix.getRowDimension(), matrix.getColumnDimension());
        }
    }

    private boolean equalsWithPrecision(FieldMatrix<Complex> matrix1, FieldMatrix<Complex> matrix2, double tolerance) {
        boolean toRet = true;
        block0: for (int i = 0; i < matrix1.getRowDimension(); ++i) {
            for (int j = 0; j < matrix1.getColumnDimension(); ++j) {
                Complex c2;
                Complex c1 = matrix1.getEntry(i, j);
                if (!(c1.add((c2 = matrix2.getEntry(i, j)).negate()).norm() > tolerance)) continue;
                toRet = false;
                continue block0;
            }
        }
        return toRet;
    }

    private FieldMatrix<Complex> convertToFieldComplex(RealMatrix matrix) {
        FieldMatrix<Complex> toRet = MatrixUtils.createFieldIdentityMatrix(ComplexField.getInstance(), matrix.getRowDimension());
        for (int i = 0; i < toRet.getRowDimension(); ++i) {
            for (int j = 0; j < toRet.getColumnDimension(); ++j) {
                toRet.setEntry(i, j, new Complex(matrix.getEntry(i, j)));
            }
        }
        return toRet;
    }
}

