/*  debyer -- program for calculation of diffration patterns
 *  Copyright 2009 Marcin Wojdyr (only code, not the tabular data)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "atomtables.h"

#include <ctype.h>
#include <string.h>
#include <math.h>


/* Periodic system of elements data.
 * This data can be also found in cctbx, ObjCryst and atominfo.
 * I'm not sure from which of these programs/libraries this was copied.
 * Reference:
 *     CRC Handbook of Chemistry & Physics, 63rd edition, 1982-1983
 *     CRC Handbook of Chemistry & Physics, 70th edition, 1989-1990
 */
const t_pse pse_table[] =
  {
    {   1, "H",  "hydrogen",       1.008f },
    {   1, "D",  "deuterium",      2.000f },
    {   2, "He", "helium",         4.003f },
    {   3, "Li", "lithium",        6.941f },
    {   4, "Be", "beryllium",      9.012f },
    {   5, "B",  "boron",         10.811f },
    {   6, "C",  "carbon",        12.011f },
    {   7, "N",  "nitrogen",      14.007f },
    {   8, "O",  "oxygen",        15.999f },
    {   9, "F",  "fluorine",      18.998f },
    {  10, "Ne", "neon",          20.180f },
    {  11, "Na", "sodium",        22.990f },
    {  12, "Mg", "magnesium",     24.305f },
    {  13, "Al", "aluminium",     26.982f },
    {  14, "Si", "silicon",       28.086f },
    {  15, "P",  "phosphorus",    30.974f },
    {  16, "S",  "sulphur",       32.066f },
    {  17, "Cl", "chlorine",      35.452f },
    {  18, "Ar", "argon",         39.948f },
    {  19, "K",  "potassium",     39.098f },
    {  20, "Ca", "calcium",       40.078f },
    {  21, "Sc", "scandium",      44.956f },
    {  22, "Ti", "titanium",      47.883f },
    {  23, "V",  "vanadium",      50.941f },
    {  24, "Cr", "chromium",      51.996f },
    {  25, "Mn", "manganese",     54.938f },
    {  26, "Fe", "iron",          55.847f },
    {  27, "Co", "cobalt",        58.933f },
    {  28, "Ni", "nickel",        58.691f },
    {  29, "Cu", "copper",        63.546f },
    {  30, "Zn", "zinc",          65.392f },
    {  31, "Ga", "gallium",       69.723f },
    {  32, "Ge", "germanium",     72.612f },
    {  33, "As", "arsenic",       74.922f },
    {  34, "Se", "selenium",      78.963f },
    {  35, "Br", "bromine",       79.904f },
    {  36, "Kr", "krypton",       83.801f },
    {  37, "Rb", "rubidium",      85.468f },
    {  38, "Sr", "strontium",     87.621f },
    {  39, "Y",  "yttrium",       88.906f },
    {  40, "Zr", "zirconium",     91.224f },
    {  41, "Nb", "niobium",       92.906f },
    {  42, "Mo", "molybdenum",    95.941f },
    {  43, "Tc", "technetium",    98.000f },
    {  44, "Ru", "ruthenium",    101.072f },
    {  45, "Rh", "rhodium",      102.905f },
    {  46, "Pd", "palladium",    106.421f },
    {  47, "Ag", "silver",       107.868f },
    {  48, "Cd", "cadmium",      112.411f },
    {  49, "In", "indium",       114.821f },
    {  50, "Sn", "tin",          118.710f },
    {  51, "Sb", "antimony",     121.753f },
    {  52, "Te", "tellurium",    127.603f },
    {  53, "I",  "iodine",       126.904f },
    {  54, "Xe", "xenon",        131.292f },
    {  55, "Cs", "caesium",      132.905f },
    {  56, "Ba", "barium",       137.327f },
    {  57, "La", "lanthanum",    138.906f },
    {  58, "Ce", "cerium",       140.115f },
    {  59, "Pr", "praseodymium", 140.908f },
    {  60, "Nd", "neodymium",    144.243f },
    {  61, "Pm", "promethium",   145.000f },
    {  62, "Sm", "samarium",     150.363f },
    {  63, "Eu", "europium",     151.965f },
    {  64, "Gd", "gadolinium",   157.253f },
    {  65, "Tb", "terbium",      158.925f },
    {  66, "Dy", "dysprosium",   162.503f },
    {  67, "Ho", "holmium",      164.930f },
    {  68, "Er", "erbium",       167.263f },
    {  69, "Tm", "thulium",      168.934f },
    {  70, "Yb", "ytterbium",    173.043f },
    {  71, "Lu", "lutetium",     174.967f },
    {  72, "Hf", "hafnium",      178.492f },
    {  73, "Ta", "tantalum",     180.948f },
    {  74, "W",  "tungsten",     183.853f },
    {  75, "Re", "rhenium",      186.207f },
    {  76, "Os", "osmium",       190.210f },
    {  77, "Ir", "iridium",      192.223f },
    {  78, "Pt", "platinum",     195.083f },
    {  79, "Au", "gold",         196.967f },
    {  80, "Hg", "mercury",      200.593f },
    {  81, "Tl", "thallium",     204.383f },
    {  82, "Pb", "lead",         207.210f },
    {  83, "Bi", "bismuth",      208.980f },
    {  84, "Po", "polonium",     209.000f },
    {  85, "At", "astatine",     210.000f },
    {  86, "Rn", "radon",        222.000f },
    {  87, "Fr", "francium",     223.000f },
    {  88, "Ra", "radium",       226.025f },
    {  89, "Ac", "actinium",     227.028f },
    {  90, "Th", "thorium",      232.038f },
    {  91, "Pa", "protactinium", 231.035f },
    {  92, "U",  "uranium",      238.028f },
    {  93, "Np", "neptunium",    237.048f },
    {  94, "Pu", "plutonium",    244.000f },
    {  95, "Am", "americium",    243.000f },
    {  96, "Cm", "curium",       247.000f },
    {  97, "Bk", "berkelium",    247.000f },
    {  98, "Cf", "californium",  251.000f },
    {  99, "Es", "einsteinium",  254.000f },
    { 100, "Fm", "fermium",      257.000f },
    { 101, "Md", "mendelevium",  258.000f },
    { 102, "No", "nobelium",     259.000f },
    { 103, "Lr", "lawrencium",   260.000f },
    {   0, NULL, NULL,             0.f }
  };


const t_pse *find_in_pse(const char *label)
{
    const t_pse  *it;
    char buf[3] = { '\0', '\0', '\0' };

    /* empty label */
    if (!label || strlen(label) == 0)
        return NULL;

    buf[0] = toupper(label[0]);
    if (strlen(label) > 1 && isalpha(label[1]))
        buf[1] = tolower(label[1]);

    for (it = pse_table; it->Z != 0; it++)
        if (strcmp(it->symbol, buf) == 0)
            return it;
    return NULL;
}

/***************************************************************************/


const t_it92_coeff it92_table[] =
{
    { "H",     { 0.493002f, 0.322912f, 0.140191f, 0.040810f },
               { 10.5109f, 26.1257f, 3.14236f, 57.7997f },
               0.003038f },
    { "H'",    { 0.489918f, 0.262003f, 0.196767f, 0.049879f },
               { 20.6593f, 7.74039f, 49.5519f, 2.20159f },
               0.001305f },
               /* "'" added */
    { "D",     { 0.489918f, 0.262003f, 0.196767f, 0.049879f },
               { 20.6593f, 7.74039f, 49.5519f, 2.20159f },
               0.001305f },
               /* copy of "H'" */
    { "H1-",   { 0.897661f, 0.565616f, 0.415815f, 0.116973f },
               { 53.1368f, 15.1870f, 186.576f, 3.56709f },
               0.002389f },
    { "He",    { 0.873400f, 0.630900f, 0.311200f, 0.178000f },
               { 9.10370f, 3.35680f, 22.9276f, 0.982100f },
               0.006400f },
    { "Li",    { 1.12820f, 0.750800f, 0.617500f, 0.465300f },
               { 3.95460f, 1.05240f, 85.3905f, 168.261f },
               0.037700f },
    { "Li1+",  { 0.696800f, 0.788800f, 0.341400f, 0.156300f },
               { 4.62370f, 1.95570f, 0.631600f, 10.0953f },
               0.016700f },
    { "Be",    { 1.59190f, 1.12780f, 0.539100f, 0.702900f },
               { 43.6427f, 1.86230f, 103.483f, 0.542000f },
               0.038500f },
    { "Be2+",  { 6.26030f, 0.884900f, 0.799300f, 0.164700f },
               { 0.002700f, 0.831300f, 2.27580f, 5.11460f },
               -6.1092f },
    { "B",     { 2.05450f, 1.33260f, 1.09790f, 0.706800f },
               { 23.2185f, 1.02100f, 60.3498f, 0.140300f },
               -0.19320f },
    { "C",     { 2.31000f, 1.02000f, 1.58860f, 0.865000f },
               { 20.8439f, 10.2075f, 0.568700f, 51.6512f },
               0.215600f },
    { "Cval",  { 2.26069f, 1.56165f, 1.05075f, 0.839259f },
               { 22.6907f, 0.656665f, 9.75618f, 55.5949f },
               0.286977f },
    { "N",     { 12.2126f, 3.13220f, 2.01250f, 1.16630f },
               { 0.005700f, 9.89330f, 28.9975f, 0.582600f },
               -11.529f },
    { "O",     { 3.04850f, 2.28680f, 1.54630f, 0.867000f },
               { 13.2771f, 5.70110f, 0.323900f, 32.9089f },
               0.250800f },
    { "O1-",   { 4.19160f, 1.63969f, 1.52673f, -20.307f },
               { 12.8573f, 4.17236f, 47.0179f, -0.01404f },
               21.9412f },
    { "O2-",   { 3.75040f, 2.84294f, 1.54298f, 1.62091f },
               { 16.5151f, 6.59203f, 0.319201f, 43.3486f },
               0.242060f },
               /* Hovestreydt, Acta Cryst. (1983) A39f, 268-269 */
    { "F",     { 3.53920f, 2.64120f, 1.51700f, 1.02430f },
               { 10.2825f, 4.29440f, 0.261500f, 26.1476f },
               0.277600f },
    { "F1-",   { 3.63220f, 3.51057f, 1.26064f, 0.940706f },
               { 5.27756f, 14.7353f, 0.442258f, 47.3437f },
               0.653396f },
    { "Ne",    { 3.95530f, 3.11250f, 1.45460f, 1.12510f },
               { 8.40420f, 3.42620f, 0.230600f, 21.7184f },
               0.351500f },
    { "Na",    { 4.76260f, 3.17360f, 1.26740f, 1.11280f },
               { 3.28500f, 8.84220f, 0.313600f, 129.424f },
               0.676000f },
    { "Na1+",  { 3.25650f, 3.93620f, 1.39980f, 1.00320f },
               { 2.66710f, 6.11530f, 0.200100f, 14.0390f },
               0.404000f },
    { "Mg",    { 5.42040f, 2.17350f, 1.22690f, 2.30730f },
               { 2.82750f, 79.2611f, 0.380800f, 7.19370f },
               0.858400f },
    { "Mg2+",  { 3.49880f, 3.83780f, 1.32840f, 0.849700f },
               { 2.16760f, 4.75420f, 0.185000f, 10.1411f },
               0.485300f },
    { "Al",    { 6.42020f, 1.90020f, 1.59360f, 1.96460f },
               { 3.03870f, 0.742600f, 31.5472f, 85.0886f },
               1.11510f },
    { "Al3+",  { 4.17448f, 3.38760f, 1.20296f, 0.528137f },
               { 1.93816f, 4.14553f, 0.228753f, 8.28524f },
               0.706786f },
    { "Si",    { 6.29150f, 3.03530f, 1.98910f, 1.54100f },
               { 2.43860f, 32.3337f, 0.678500f, 81.6937f },
               1.14070f },
               /* copy of "Siv" */
    { "Siv",   { 6.29150f, 3.03530f, 1.98910f, 1.54100f },
               { 2.43860f, 32.3337f, 0.678500f, 81.6937f },
               1.14070f },
    { "Sival", { 5.66269f, 3.07164f, 2.62446f, 1.39320f },
               { 2.66520f, 38.6634f, 0.916946f, 93.5458f },
               1.24707f },
    { "Si4+",  { 4.43918f, 3.20345f, 1.19453f, 0.416530f },
               { 1.64167f, 3.43757f, 0.214900f, 6.65365f },
               0.746297f },
    { "P",     { 6.43450f, 4.17910f, 1.78000f, 1.49080f },
               { 1.90670f, 27.1570f, 0.526000f, 68.1645f },
               1.11490f },
    { "S",     { 6.90530f, 5.20340f, 1.43790f, 1.58630f },
               { 1.46790f, 22.2151f, 0.253600f, 56.1720f },
               0.866900f },
    { "Cl",    { 11.4604f, 7.19640f, 6.25560f, 1.64550f },
               { 0.010400f, 1.16620f, 18.5194f, 47.7784f },
               -9.5574f },
    { "Cl1-",  { 18.2915f, 7.20840f, 6.53370f, 2.33860f },
               { 0.006600f, 1.17170f, 19.5424f, 60.4486f },
               -16.378f },
    { "Ar",    { 7.48450f, 6.77230f, 0.653900f, 1.64420f },
               { 0.907200f, 14.8407f, 43.8983f, 33.3929f },
               1.44450f },
    { "K",     { 8.21860f, 7.43980f, 1.05190f, 0.865900f },
               { 12.7949f, 0.774800f, 213.187f, 41.6841f },
               1.42280f },
    { "K1+",   { 7.95780f, 7.49170f, 6.35900f, 1.19150f },
               { 12.6331f, 0.767400f, -0.00200f, 31.9128f },
               -4.9978f },
    { "Ca",    { 8.62660f, 7.38730f, 1.58990f, 1.02110f },
               { 10.4421f, 0.659900f, 85.7484f, 178.437f },
               1.37510f },
    { "Ca2+",  { 15.6348f, 7.95180f, 8.43720f, 0.853700f },
               { -0.00740f, 0.608900f, 10.3116f, 25.9905f },
               -14.875f },
    { "Sc",    { 9.18900f, 7.36790f, 1.64090f, 1.46800f },
               { 9.02130f, 0.572900f, 136.108f, 51.3531f },
               1.33290f },
    { "Sc3+",  { 13.4008f, 8.02730f, 1.65943f, 1.57936f },
               { 0.298540f, 7.96290f, -0.28604f, 16.0662f },
               -6.6667f },
    { "Ti",    { 9.75950f, 7.35580f, 1.69910f, 1.90210f },
               { 7.85080f, 0.500000f, 35.6338f, 116.105f },
               1.28070f },
    { "Ti2+",  { 9.11423f, 7.62174f, 2.27930f, 0.087899f },
               { 7.52430f, 0.457585f, 19.5361f, 61.6558f },
               0.897155f },
    { "Ti3+",  { 17.7344f, 8.73816f, 5.25691f, 1.92134f },
               { 0.220610f, 7.04716f, -0.15762f, 15.9768f },
               -14.652f },
    { "Ti4+",  { 19.5114f, 8.23473f, 2.01341f, 1.52080f },
               { 0.178847f, 6.67018f, -0.29263f, 12.9464f },
               -13.280f },
    { "V",     { 10.2971f, 7.35110f, 2.07030f, 2.05710f },
               { 6.86570f, 0.438500f, 26.8938f, 102.478f },
               1.21990f },
    { "V2+",   { 10.1060f, 7.35410f, 2.28840f, 0.022300f },
               { 6.88180f, 0.440900f, 20.3004f, 115.122f },
               1.22980f },
    { "V3+",   { 9.43141f, 7.74190f, 2.15343f, 0.016865f },
               { 6.39535f, 0.383349f, 15.1908f, 63.9690f },
               0.656565f },
    { "V5+",   { 15.6887f, 8.14208f, 2.03081f, -9.5760f },
               { 0.679003f, 5.40135f, 9.97278f, 0.940464f },
               1.71430f },
    { "Cr",    { 10.6406f, 7.35370f, 3.32400f, 1.49220f },
               { 6.10380f, 0.392000f, 20.2626f, 98.7399f },
               1.18320f },
    { "Cr2+",  { 9.54034f, 7.75090f, 3.58274f, 0.509107f },
               { 5.66078f, 0.344261f, 13.3075f, 32.4224f },
               0.616898f },
    { "Cr3+",  { 9.68090f, 7.81136f, 2.87603f, 0.113575f },
               { 5.59463f, 0.334393f, 12.8288f, 32.8761f },
               0.518275f },
    { "Mn",    { 11.2819f, 7.35730f, 3.01930f, 2.24410f },
               { 5.34090f, 0.343200f, 17.8674f, 83.7543f },
               1.08960f },
    { "Mn2+",  { 10.8061f, 7.36200f, 3.52680f, 0.218400f },
               { 5.27960f, 0.343500f, 14.3430f, 41.3235f },
               1.08740f },
    { "Mn3+",  { 9.84521f, 7.87194f, 3.56531f, 0.323613f },
               { 4.91797f, 0.294393f, 10.8171f, 24.1281f },
               0.393974f },
    { "Mn4+",  { 9.96253f, 7.97057f, 2.76067f, 0.054447f },
               { 4.84850f, 0.283303f, 10.4852f, 27.5730f },
               0.251877f },
    { "Fe",    { 11.7695f, 7.35730f, 3.52220f, 2.30450f },
               { 4.76110f, 0.307200f, 15.3535f, 76.8805f },
               1.03690f },
    { "Fe2+",  { 11.0424f, 7.37400f, 4.13460f, 0.439900f },
               { 4.65380f, 0.305300f, 12.0546f, 31.2809f },
               1.00970f },
    { "Fe3+",  { 11.1764f, 7.38630f, 3.39480f, 0.072400f },
               { 4.61470f, 0.300500f, 11.6729f, 38.5566f },
               0.970700f },
    { "Co",    { 12.2841f, 7.34090f, 4.00340f, 2.34880f },
               { 4.27910f, 0.278400f, 13.5359f, 71.1692f },
               1.01180f },
    { "Co2+",  { 11.2296f, 7.38830f, 4.73930f, 0.710800f },
               { 4.12310f, 0.272600f, 10.2443f, 25.6466f },
               0.932400f },
    { "Co3+",  { 10.3380f, 7.88173f, 4.76795f, 0.725591f },
               { 3.90969f, 0.238668f, 8.35583f, 18.3491f },
               0.286667f },
    { "Ni",    { 12.8376f, 7.29200f, 4.44380f, 2.38000f },
               { 3.87850f, 0.256500f, 12.1763f, 66.3421f },
               1.03410f },
    { "Ni2+",  { 11.4166f, 7.40050f, 5.34420f, 0.977300f },
               { 3.67660f, 0.244900f, 8.87300f, 22.1626f },
               0.861400f },
    { "Ni3+",  { 10.7806f, 7.75868f, 5.22746f, 0.847114f },
               { 3.54770f, 0.223140f, 7.64468f, 16.9673f },
               0.386044f },
    { "Cu",    { 13.3380f, 7.16760f, 5.61580f, 1.67350f },
               { 3.58280f, 0.247000f, 11.3966f, 64.8126f },
               1.19100f },
    { "Cu1+",  { 11.9475f, 7.35730f, 6.24550f, 1.55780f },
               { 3.36690f, 0.227400f, 8.66250f, 25.8487f },
               0.89000f },
    { "Cu2+",  { 11.8168f, 7.11181f, 5.78135f, 1.14523f },
               { 3.37484f, 0.244078f, 7.98760f, 19.8970f },
               1.14431f },
    { "Zn",    { 14.0743f, 7.03180f, 5.16520f, 2.41000f },
               { 3.26550f, 0.233300f, 10.3163f, 58.7097f },
               1.30410f },
    { "Zn2+",  { 11.9719f, 7.38620f, 6.46680f, 1.39400f },
               { 2.99460f, 0.203100f, 7.08260f, 18.0995f },
               0.780700f },
    { "Ga",    { 15.2354f, 6.70060f, 4.35910f, 2.96230f },
               { 3.06690f, 0.241200f, 10.7805f, 61.4135f },
               1.71890f },
    { "Ga3+",  { 12.6920f, 6.69883f, 6.06692f, 1.00660f },
               { 2.81262f, 0.227890f, 6.36441f, 14.4122f },
               1.53545f },
    { "Ge",    { 16.0816f, 6.37470f, 3.70680f, 3.68300f },
               { 2.85090f, 0.251600f, 11.4468f, 54.7625f },
               2.13130f },
    { "Ge4+",  { 12.9172f, 6.70003f, 6.06791f, 0.859041f },
               { 2.53718f, 0.205855f, 5.47913f, 11.6030f },
               1.45572f },
    { "As",    { 16.6723f, 6.07010f, 3.43130f, 4.27790f },
               { 2.63450f, 0.264700f, 12.9479f, 47.7972f },
               2.53100f },
    { "Se",    { 17.0006f, 5.81960f, 3.97310f, 4.35430f },
               { 2.40980f, 0.272600f, 15.2372f, 43.8163f },
               2.84090f },
    { "Br",    { 17.1789f, 5.23580f, 5.63770f, 3.98510f },
               { 2.17230f, 16.5796f, 0.260900f, 41.4328f },
               2.95570f },
    { "Br1-",  { 17.1718f, 6.33380f, 5.57540f, 3.72720f },
               { 2.20590f, 19.3345f, 0.287100f, 58.1535f },
               3.17760f },
    { "Kr",    { 17.3555f, 6.72860f, 5.54930f, 3.53750f },
               { 1.93840f, 16.5623f, 0.226100f, 39.3972f },
               2.82500f },
    { "Rb",    { 17.1784f, 9.64350f, 5.13990f, 1.52920f },
               { 1.78880f, 17.3151f, 0.274800f, 164.934f },
               3.48730f },
    { "Rb1+",  { 17.5816f, 7.65980f, 5.89810f, 2.78170f },
               { 1.71390f, 14.7957f, 0.160300f, 31.2087f },
               2.07820f },
    { "Sr",    { 17.5663f, 9.81840f, 5.42200f, 2.66940f },
               { 1.55640f, 14.0988f, 0.166400f, 132.376f },
               2.50640f },
    { "Sr2+",  { 18.0874f, 8.13730f, 2.56540f, -34.193f },
               { 1.49070f, 12.6963f, 24.5651f, -0.01380f },
               41.4025f },
    { "Y",     { 17.7760f, 10.2946f, 5.72629f, 3.26588f },
               { 1.40290f, 12.8006f, 0.125599f, 104.354f },
               1.91213f },
    { "Y3+",   { 17.9268f, 9.15310f, 1.76795f, -33.108f },
               { 1.35417f, 11.2145f, 22.6599f, -0.01319f },
               40.2602f },
    { "Zr",    { 17.8765f, 10.9480f, 5.41732f, 3.65721f },
               { 1.27618f, 11.9160f, 0.117622f, 87.6627f },
               2.06929f },
    { "Zr4+",  { 18.1668f, 10.0562f, 1.01118f, -2.6479f },
               { 1.21480f, 10.1483f, 21.6054f, -0.10276f },
               9.41454f },
    { "Nb",    { 17.6142f, 12.0144f, 4.04183f, 3.53346f },
               { 1.18865f, 11.7660f, 0.204785f, 69.7957f },
               3.75591f },
    { "Nb3+",  { 19.8812f, 18.0653f, 11.0177f, 1.94715f },
               { 0.019175f, 1.13305f, 10.1621f, 28.3389f },
               -12.912f },
    { "Nb5+",  { 17.9163f, 13.3417f, 10.7990f, 0.337905f },
               { 1.12446f, 0.028781f, 9.28206f, 25.7228f },
               -6.3934f },
    { "Mo",    { 3.70250f, 17.2356f, 12.8876f, 3.74290f },
               { 0.277200f, 1.09580f, 11.0040f, 61.6584f },
               4.38750f },
    { "Mo3+",  { 21.1664f, 18.2017f, 11.7423f, 2.30951f },
               { 0.014734f, 1.03031f, 9.53659f, 26.6307f },
               -14.421f },
    { "Mo5+",  { 21.0149f, 18.0992f, 11.4632f, 0.740625f },
               { 0.014345f, 1.02238f, 8.78809f, 23.3452f },
               -14.316f },
    { "Mo6+",  { 17.8871f, 11.1750f, 6.57891f, 0.000000f },
               { 1.03649f, 8.48061f, 0.058881f, 0.000000f },
               0.344941f },
    { "Tc",    { 19.1301f, 11.0948f, 4.64901f, 2.71263f },
               { 0.864132f, 8.14487f, 21.5707f, 86.8472f },
               5.40428f },
    { "Ru",    { 19.2674f, 12.9182f, 4.86337f, 1.56756f },
               { 0.808520f, 8.43467f, 24.7997f, 94.2928f },
               5.37874f },
    { "Ru3+",  { 18.5638f, 13.2885f, 9.32602f, 3.00964f },
               { 0.847329f, 8.37164f, 0.017662f, 22.8870f },
               -3.1892f },
    { "Ru4+",  { 18.5003f, 13.1787f, 4.71304f, 2.18535f },
               { 0.844582f, 8.12534f, 0.36495f, 20.8504f },
               1.42357f },
    { "Rh",    { 19.2957f, 14.3501f, 4.73425f, 1.28918f },
               { 0.751536f, 8.21758f, 25.8749f, 98.6062f },
               5.32800f },
    { "Rh3+",  { 18.8785f, 14.1259f, 3.32515f, -6.1989f },
               { 0.764252f, 7.84438f, 21.2487f, -0.01036f },
               11.8678f },
    { "Rh4+",  { 18.8545f, 13.9806f, 2.53464f, -5.6526f },
               { 0.760825f, 7.62436f, 19.3317f, -0.01020f },
               11.2835f },
    { "Pd",    { 19.3319f, 15.5017f, 5.29537f, 0.605844f },
               { 0.698655f, 7.98929f, 25.2052f, 76.8986f },
               5.26593f },
    { "Pd2+",  { 19.1701f, 15.2096f, 4.32234f, 0.000000f },
               { 0.696219f, 7.55573f, 22.5057f, 0.000000f },
               5.29160f },
    { "Pd4+",  { 19.2493f, 14.7900f, 2.89289f, -7.9492f },
               { 0.683839f, 7.14833f, 17.9144f, 0.005127f },
               13.0174f },
    { "Ag",    { 19.2808f, 16.6885f, 4.80450f, 1.04630f },
               { 0.644600f, 7.47260f, 24.6605f, 99.8156f },
               5.17900f },
    { "Ag1+",  { 19.1812f, 15.9719f, 5.27475f, 0.357534f },
               { 0.646179f, 7.19123f, 21.7326f, 66.1147f },
               5.21572f },
    { "Ag2+",  { 19.1643f, 16.2456f, 4.37090f, 0.000000f },
               { 0.645643f, 7.18544f, 21.4072f, 0.000000f },
               5.21404f },
    { "Cd",    { 19.2214f, 17.6444f, 4.46100f, 1.60290f },
               { 0.594600f, 6.90890f, 24.7008f, 87.4825f },
               5.06940f },
    { "Cd2+",  { 19.1514f, 17.2535f, 4.47128f, 0.000000f },
               { 0.597922f, 6.80639f, 20.2521f, 0.000000f },
               5.11937f },
    { "In",    { 19.1624f, 18.5596f, 4.29480f, 2.03960f },
               { 0.547600f, 6.37760f, 25.8499f, 92.8029f },
               4.93910f },
    { "In3+",  { 19.1045f, 18.1108f, 3.78897f, 0.000000f },
               { 0.551522f, 6.32470f, 17.3595f, 0.000000f },
               4.99635f },
    { "Sn",    { 19.1889f, 19.1005f, 4.45850f, 2.46630f },
               { 5.83030f, 0.503100f, 26.8909f, 83.9571f },
               4.78210f },
    { "Sn2+",  { 19.1094f, 19.0548f, 4.56480f, 0.487000f },
               { 0.503600f, 5.83780f, 23.3752f, 62.2061f },
               4.78610f },
    { "Sn4+",  { 18.9333f, 19.7131f, 3.41820f, 0.019300f },
               { 5.76400f, 0.465500f, 14.0049f, -0.75830f },
               3.91820f },
    { "Sb",    { 19.6418f, 19.0455f, 5.03710f, 2.68270f },
               { 5.30340f, 0.460700f, 27.9074f, 75.2825f },
               4.59090f },
    { "Sb3+",  { 18.9755f, 18.9330f, 5.10789f, 0.288753f },
               { 0.467196f, 5.22126f, 19.5902f, 55.5113f },
               4.69626f },
    { "Sb5+",  { 19.8685f, 19.0302f, 2.41253f, 0.000000f },
               { 5.44853f, 0.467973f, 14.1259f, 0.000000f },
               4.69263f },
    { "Te",    { 19.9644f, 19.0138f, 6.14487f, 2.52390f },
               { 4.81742f, 0.420885f, 28.5284f, 70.8403f },
               4.35200f },
    { "I",     { 20.1472f, 18.9949f, 7.51380f, 2.27350f },
               { 4.34700f, 0.381400f, 27.7660f, 66.8776f },
               4.07120f },
    { "I1-",   { 20.2332f, 18.9970f, 7.80690f, 2.88680f },
               { 4.35790f, 0.381500f, 29.5259f, 84.9304f },
               4.07140f },
    { "Xe",    { 20.2933f, 19.0298f, 8.97670f, 1.99000f },
               { 3.92820f, 0.344000f, 26.4659f, 64.2658f },
               3.71180f },
    { "Cs",    { 20.3892f, 19.1062f, 10.6620f, 1.49530f },
               { 3.56900f, 0.310700f, 24.3879f, 213.904f },
               3.33520f },
    { "Cs1+",  { 20.3524f, 19.1278f, 10.2821f, 0.961500f },
               { 3.55200f, 0.308600f, 23.7128f, 59.4565f },
               3.27910f },
    { "Ba",    { 20.3361f, 19.2970f, 10.8880f, 2.69590f },
               { 3.21600f, 0.275600f, 20.2073f, 167.202f },
               2.77310f },
    { "Ba2+",  { 20.1807f, 19.1136f, 10.9054f, 0.77634f },
               { 3.21367f, 0.283310f, 20.0558f, 51.7460f },
               3.02902f },
    { "La",    { 20.5780f, 19.5990f, 11.3727f, 3.28719f },
               { 2.94817f, 0.244475f, 18.7726f, 133.124f },
               2.14678f },
    { "La3+",  { 20.2489f, 19.3763f, 11.6323f, 0.336048f },
               { 2.92070f, 0.250698f, 17.8211f, 54.9453f },
               2.40860f },
    { "Ce",    { 21.1671f, 19.7695f, 11.8513f, 3.33049f },
               { 2.81219f, 0.226836f, 17.6083f, 127.113f },
               1.86264f },
    { "Ce3+",  { 20.8036f, 19.5590f, 11.9369f, 0.612376f },
               { 2.77691f, 0.231540f, 16.5408f, 43.1692f },
               2.09013f },
    { "Ce4+",  { 20.3235f, 19.8186f, 12.1233f, 0.144583f },
               { 2.65941f, 0.218850f, 15.7992f, 62.2355f },
               1.59180f },
    { "Pr",    { 22.0440f, 19.6697f, 12.3856f, 2.82428f },
               { 2.77393f, 0.222087f, 16.7669f, 143.644f },
               2.05830f },
    { "Pr3+",  { 21.3727f, 19.7491f, 12.1329f, 0.975180f },
               { 2.64520f, 0.214299f, 15.3230f, 36.4065f },
               1.77132f },
    { "Pr4+",  { 20.9413f, 20.0539f, 12.4668f, 0.296689f },
               { 2.54467f, 0.202481f, 14.8137f, 45.4643f },
               1.24285f },
    { "Nd",    { 22.6845f, 19.6847f, 12.7740f, 2.85137f },
               { 2.66248f, 0.210628f, 15.8850f, 137.903f },
               1.98486f },
    { "Nd3+",  { 21.9610f, 19.9339f, 12.1200f, 1.51031f },
               { 2.52722f, 0.199237f, 14.1783f, 30.8717f },
               1.47588f },
    { "Pm",    { 23.3405f, 19.6095f, 13.1235f, 2.87516f },
               { 2.56270f, 0.202088f, 15.1009f, 132.721f },
               2.02876f },
    { "Pm3+",  { 22.5527f, 20.1108f, 12.0671f, 2.07492f },
               { 2.41740f, 0.185769f, 13.1275f, 27.4491f },
               1.19499f },
    { "Sm",    { 24.0042f, 19.4258f, 13.4396f, 2.89604f },
               { 2.47274f, 0.196451f, 14.3996f, 128.007f },
               2.20963f },
    { "Sm3+",  { 23.1504f, 20.2599f, 11.9202f, 2.71488f },
               { 2.31641f, 0.174081f, 12.1571f, 24.8242f },
               0.954586f },
    { "Eu",    { 24.6274f, 19.0886f, 13.7603f, 2.92270f },
               { 2.38790f, 0.194200f, 13.7546f, 123.174f },
               2.57450f },
    { "Eu2+",  { 24.0063f, 19.9504f, 11.8034f, 3.87243f },
               { 2.27783f, 0.173530f, 11.6096f, 26.5156f },
               1.36389f },
    { "Eu3+",  { 23.7497f, 20.3745f, 11.8509f, 3.26503f },
               { 2.22258f, 0.163940f, 11.3110f, 22.9966f },
               0.759344f },
    { "Gd",    { 25.0709f, 19.0798f, 13.8518f, 3.54545f },
               { 2.25341f, 0.181951f, 12.9331f, 101.398f },
               2.41960f },
    { "Gd3+",  { 24.3466f, 20.4208f, 11.8708f, 3.71490f },
               { 2.13553f, 0.155525f, 10.5782f, 21.7029f },
               0.645089f },
    { "Tb",    { 25.8976f, 18.2185f, 14.3167f, 2.95354f },
               { 2.24256f, 0.196143f, 12.6648f, 115.362f },
               3.58324f },
    { "Tb3+",  { 24.9559f, 20.3271f, 12.2471f, 3.77300f },
               { 2.05601f, 0.149525f, 10.0499f, 21.2773f },
               0.691967f },
    { "Dy",    { 26.5070f, 17.6383f, 14.5596f, 2.96577f },
               { 2.18020f, 0.202172f, 12.1899f, 111.874f },
               4.29728f },
    { "Dy3+",  { 25.5395f, 20.2861f, 11.9812f, 4.50073f },
               { 1.98040f, 0.143384f, 9.34972f, 19.5810f },
               0.689690f },
    { "Ho",    { 26.9049f, 17.2940f, 14.5583f, 3.63837f },
               { 2.07051f, 0.197940f, 11.4407f, 92.6566f },
               4.56796f },
    { "Ho3+",  { 26.1296f, 20.0994f, 11.9788f, 4.93676f },
               { 1.91072f, 0.139358f, 8.80018f, 18.5908f },
               0.852795f },
    { "Er",    { 27.6563f, 16.4285f, 14.9779f, 2.98233f },
               { 2.07356f, 0.223545f, 11.3604f, 105.703f },
               5.92046f },
    { "Er3+",  { 26.7220f, 19.7748f, 12.1506f, 5.17379f },
               { 1.84659f, 0.137290f, 8.36225f, 17.8974f },
               1.17613f },
    { "Tm",    { 28.1819f, 15.8851f, 15.1542f, 2.98706f },
               { 2.02859f, 0.238849f, 10.9975f, 102.961f },
               6.75621f },
    { "Tm3+",  { 27.3083f, 19.3320f, 12.3339f, 5.38348f },
               { 1.78711f, 0.136974f, 7.96778f, 17.2922f },
               1.63929f },
    { "Yb",    { 28.6641f, 15.4345f, 15.3087f, 2.98963f },
               { 1.98890f, 0.257119f, 10.6647f, 100.417f },
               7.56672f },
    { "Yb2+",  { 28.1209f, 17.6817f, 13.3335f, 5.14657f },
               { 1.78503f, 0.159970f, 8.18304f, 20.3900f },
               3.70983f },
    { "Yb3+",  { 27.8917f, 18.7614f, 12.6072f, 5.47647f },
               { 1.73272f, 0.138790f, 7.64412f, 16.8153f },
               2.26001f },
    { "Lu",    { 28.9476f, 15.2208f, 15.1000f, 3.71601f },
               { 1.90182f, 9.98519f, 0.261033f, 84.3298f },
               7.97628f },
    { "Lu3+",  { 28.4628f, 18.1210f, 12.8429f, 5.59415f },
               { 1.68216f, 0.142292f, 7.33727f, 16.3535f },
               2.97573f },
    { "Hf",    { 29.1440f, 15.1726f, 14.7586f, 4.30013f },
               { 1.83262f, 9.59990f, 0.275116f, 72.0290f },
               8.58154f },
    { "Hf4+",  { 28.8131f, 18.4601f, 12.7285f, 5.59927f },
               { 1.59136f, 0.128903f, 6.76232f, 14.0366f },
               2.39699f },
    { "Ta",    { 29.2024f, 15.2293f, 14.5135f, 4.76492f },
               { 1.77333f, 9.37046f, 0.295977f, 63.3644f },
               9.24354f },
    { "Ta5+",  { 29.1587f, 18.8407f, 12.8268f, 5.38695f },
               { 1.50711f, 0.116741f, 6.31524f, 12.4244f },
               1.78555f },
    { "W",     { 29.0818f, 15.4300f, 14.4327f, 5.11982f },
               { 1.72029f, 9.22590f, 0.321703f, 57.0560f },
               9.88750f },
    { "W6+",   { 29.4936f, 19.3763f, 13.0544f, 5.06412f },
               { 1.42755f, 0.104621f, 5.93667f, 11.1972f },
               1.01074f },
    { "Re",    { 28.7621f, 15.7189f, 14.5564f, 5.44174f },
               { 1.67191f, 9.09227f, 0.350500f, 52.0861f },
               10.4720f },
    { "Os",    { 28.1894f, 16.1550f, 14.9305f, 5.67589f },
               { 1.62903f, 8.97948f, 0.382661f, 48.1647f },
               11.0005f },
    { "Os4+",  { 30.4190f, 15.2637f, 14.7458f, 5.06795f },
               { 1.37113f, 6.84706f, 0.165191f, 18.0030f },
               6.49804f },
    { "Ir",    { 27.3049f, 16.7296f, 15.6115f, 5.83377f },
               { 1.59279f, 8.86553f, 0.417916f, 45.0011f },
               11.4722f },
    { "Ir3+",  { 30.4156f, 15.8620f, 13.6145f, 5.82008f },
               { 1.34323f, 7.10909f, 0.204633f, 20.3254f },
               8.27903f },
    { "Ir4+",  { 30.7058f, 15.5512f, 14.2326f, 5.53672f },
               { 1.30923f, 6.71983f, 0.167252f, 17.4911f },
               6.96824f },
    { "Pt",    { 27.0059f, 17.7639f, 15.7131f, 5.78370f },
               { 1.51293f, 8.81174f, 0.424593f, 38.6103f },
               11.6883f },
    { "Pt2+",  { 29.8429f, 16.7224f, 13.2153f, 6.35234f },
               { 1.32927f, 7.38979f, 0.263297f, 22.9426f },
               9.85329f },
    { "Pt4+",  { 30.9612f, 15.9829f, 13.7348f, 5.92034f },
               { 1.24813f, 6.60834f, 0.168640f, 16.9392f },
               7.39534f },
    { "Au",    { 16.8819f, 18.5913f, 25.5582f, 5.86000f },
               { 0.461100f, 8.62160f, 1.48260f, 36.3956f },
               12.0658f },
    { "Au1+",  { 28.0109f, 17.8204f, 14.3359f, 6.58077f },
               { 1.35321f, 7.73950f, 0.356752f, 26.4043f },
               11.2299f },
    { "Au3+",  { 30.6886f, 16.9029f, 12.7801f, 6.52354f },
               { 1.21990f, 6.82872f, 0.212867f, 18.6590f },
               9.09680f },
    { "Hg",    { 20.6809f, 19.0417f, 21.6575f, 5.96760f },
               { 0.545000f, 8.44840f, 1.57290f, 38.3246f },
               12.6089f },
    { "Hg1+",  { 25.0853f, 18.4973f, 16.8883f, 6.48216f },
               { 1.39507f, 7.65105f, 0.443378f, 28.2262f },
               12.0205f },
    { "Hg2+",  { 29.5641f, 18.0600f, 12.8374f, 6.89912f },
               { 1.21152f, 7.05639f, 0.284738f, 20.7482f },
               10.6268f },
    { "Tl",    { 27.5446f, 19.1584f, 15.5380f, 5.52593f },
               { 0.655150f, 8.70751f, 1.96347f, 45.8149f },
               13.1746f },
    { "Tl1+",  { 21.3985f, 20.4723f, 18.7478f, 6.82847f },
               { 1.47110f, 0.517394f, 7.43463f, 28.8482f },
               12.5258f },
    { "Tl3+",  { 30.8695f, 18.3841f, 11.9328f, 7.00574f },
               { 1.10080f, 6.53852f, 0.219074f, 17.2114f },
               9.80270f },
               /* IT Vol IV 1974: a2 = 18.3841
                  IT Vol C  1992: a2 = 18.3481 */
    { "Pb",    { 31.0617f, 13.0637f, 18.4420f, 5.96960f },
               { 0.690200f, 2.35760f, 8.61800f, 47.2579f },
               13.4118f },
    { "Pb2+",  { 21.7886f, 19.5682f, 19.1406f, 7.01107f },
               { 1.33660f, 0.488383f, 6.77270f, 23.8132f },
               12.4734f },
    { "Pb4+",  { 32.1244f, 18.8003f, 12.0175f, 6.96886f },
               { 1.00566f, 6.10926f, 0.147041f, 14.7140f },
               8.08428f },
    { "Bi",    { 33.3689f, 12.9510f, 16.5877f, 6.46920f },
               { 0.704000f, 2.92380f, 8.79370f, 48.0093f },
               13.5782f },
    { "Bi3+",  { 21.8053f, 19.5026f, 19.1053f, 7.10295f },
               { 1.23560f, 6.24149f, 0.469999f, 20.3185f },
               12.4711f },
    { "Bi5+",  { 33.5364f, 25.0946f, 19.2497f, 6.91555f },
               { 0.916540f, 0.39042f, 5.71414f, 12.8285f },
               -6.7994f },
    { "Po",    { 34.6726f, 15.4733f, 13.1138f, 7.02588f },
               { 0.700999f, 3.55078f, 9.55642f, 47.0045f },
               13.6770f },
    { "At",    { 35.3163f, 19.0211f, 9.49887f, 7.42518f },
               { 0.685870f, 3.97458f, 11.3824f, 45.4715f },
               13.7108f },
    { "Rn",    { 35.5631f, 21.2816f, 8.00370f, 7.44330f },
               { 0.663100f, 4.06910f, 14.0422f, 44.2473f },
               13.6905f },
    { "Fr",    { 35.9299f, 23.0547f, 12.1439f, 2.11253f },
               { 0.646453f, 4.17619f, 23.1052f, 150.645f },
               13.7247f },
    { "Ra",    { 35.7630f, 22.9064f, 12.4739f, 3.21097f },
               { 0.616341f, 3.87135f, 19.9887f, 142.325f },
               13.6211f },
    { "Ra2+",  { 35.2150f, 21.6700f, 7.91342f, 7.65078f },
               { 0.604909f, 3.57670f, 12.6010f, 29.8436f },
               13.5431f },
    { "Ac",    { 35.6597f, 23.1032f, 12.5977f, 4.08655f },
               { 0.589092f, 3.65155f, 18.5990f, 117.020f },
               13.5266f },
    { "Ac3+",  { 35.1736f, 22.1112f, 8.19216f, 7.05545f },
               { 0.579689f, 3.41437f, 12.9187f, 25.9443f },
               13.4637f },
    { "Th",    { 35.5645f, 23.4219f, 12.7473f, 4.80703f },
               { 0.563359f, 3.46204f, 17.8309f, 99.1722f },
               13.4314f },
    { "Th4+",  { 35.1007f, 22.4418f, 9.78554f, 5.29444f },
               { 0.555054f, 3.24498f, 13.4661f, 23.9533f },
               13.3760f },
    { "Pa",    { 35.8847f, 23.2948f, 14.1891f, 4.17287f },
               { 0.547751f, 3.41519f, 16.9235f, 105.251f },
               13.4287f },
    { "U",     { 36.0228f, 23.4128f, 14.9491f, 4.18800f },
               { 0.529300f, 3.32530f, 16.0927f, 100.613f },
               13.3966f },
    { "U3+",   { 35.5747f, 22.5259f, 12.2165f, 5.37073f },
               { 0.520480f, 3.12293f, 12.7148f, 26.3394f },
               13.3092f },
    { "U4+",   { 35.3715f, 22.5326f, 12.0291f, 4.79840f },
               { 0.516598f, 3.05053f, 12.5723f, 23.4582f },
               13.2671f },
    { "U6+",   { 34.8509f, 22.7584f, 14.0099f, 1.21457f },
               { 0.507079f, 2.89030f, 13.1767f, 25.2017f },
               13.1665f },
    { "Np",    { 36.1874f, 23.5964f, 15.6402f, 4.18550f },
               { 0.511929f, 3.25396f, 15.3622f, 97.4908f },
               13.3573f },
    { "Np3+",  { 35.7074f, 22.6130f, 12.9898f, 5.43227f },
               { 0.502322f, 3.03807f, 12.1449f, 25.4928f },
               13.2544f },
    { "Np4+",  { 35.5103f, 22.5787f, 12.7766f, 4.92159f },
               { 0.498626f, 2.96627f, 11.9484f, 22.7502f },
               13.2116f },
    { "Np6+",  { 35.0136f, 22.7286f, 14.3884f, 1.75669f },
               { 0.489810f, 2.81099f, 12.3300f, 22.6581f },
               13.1130f },
    { "Pu",    { 36.5254f, 23.8083f, 16.7707f, 3.47947f },
               { 0.499384f, 3.26371f, 14.9455f, 105.980f },
               13.3812f },
    { "Pu3+",  { 35.8400f, 22.7169f, 13.5807f, 5.66016f },
               { 0.484938f, 2.96118f, 11.5331f, 24.3992f },
               13.1991f },
    { "Pu4+",  { 35.6493f, 22.6460f, 13.3595f, 5.18831f },
               { 0.481422f, 2.89020f, 11.3160f, 21.8301f },
               13.1555f },
    { "Pu6+",  { 35.1736f, 22.7181f, 14.7635f, 2.28678f },
               { 0.473204f, 2.73848f, 11.5530f, 20.9303f },
               13.0582f },
    { "Am",    { 36.6706f, 24.0992f, 17.3415f, 3.49331f },
               { 0.483629f, 3.20647f, 14.3136f, 102.273f },
               13.3592f },
    { "Cm",    { 36.6488f, 24.4096f, 17.3990f, 4.21665f },
               { 0.465154f, 3.08997f, 13.4346f, 88.4834f },
               13.2887f },
    { "Bk",    { 36.7881f, 24.7736f, 17.8919f, 4.23284f },
               { 0.451018f, 3.04619f, 12.8946f, 86.0030f },
               13.2754f },
    { "Cf",    { 36.9185f, 25.1995f, 18.3317f, 4.24391f },
               { 0.437533f, 3.00775f, 12.4044f, 83.7881f },
               13.2674f },
    /*
    { "Es",    { 0., 0., 0., 0. },
               { 0., 0., 0., 0. },
               0. },
    { "Fm",    { 0., 0., 0., 0. },
               { 0., 0., 0., 0. },
               0. },
    { "Md",    { 0., 0., 0., 0. },
               { 0., 0., 0., 0. },
               0. },
    { "No",    { 0., 0., 0., 0. },
               { 0., 0., 0., 0. },
               0. },
    { "Lr",    { 0., 0., 0., 0. },
               { 0., 0., 0., 0. },
               0. },
   */
    { NULL,    { 0., 0., 0., 0. },
               { 0., 0., 0., 0. },
               0. }
};

/* Changes (in place) case of letters to match symbols in tables (i.e. to Abc),
 * replaces +/- (if doesn't follow a digit) with 1+/1-,
 * strips junk from the end.
 * Returns number of initial alpha characters.
 * `buf' should have one more character allocated after the string,
 *  in case + is changed to 1+.
 */
int fix_label(char *buf)
{
    int i;

    if (!buf || strlen(buf) == 0)
        return 0;
    buf[0] = toupper(buf[0]);
    for (i = 1; isalpha(buf[i]); i++)
        buf[i] = tolower(buf[i]);

    if (buf[i] == '+' || buf[i] == '-') {
        buf[i+1] = buf[i];
        buf[i] = '1';
        buf[i+2] = '\0';
    }
    else if (isdigit(buf[i]) && (buf[i+1] == '+' || buf[i+1] == '-'))
        buf[i+2] = '\0';
    else
        buf[i] = '\0';
    return i;
}

const t_it92_coeff *find_in_it92(const char *label)
{
    size_t alpha_len;
    char buf[16];
    const t_it92_coeff *p;

    strncpy(buf, label, 14);
    alpha_len = fix_label(buf);

    for (p = it92_table; p->symbol != NULL; p++)
        if (strcmp(buf, p->symbol) == 0)
            return p;
    /* try again, comparing only alpha characters */
    if (alpha_len != strlen(buf))
        for (p = it92_table; p->symbol != NULL; p++)
            if (strncmp(buf, p->symbol, alpha_len) == 0)
                return p;
    return NULL;
}

double calculate_it92_factor(const t_it92_coeff *p, double stol2)
{
    int i;
    double sf = p->c;
    for (i = 0; i < 4; i++)
        sf += p->a[i] * exp(-p->b[i] * stol2);
    return sf;
}

/***************************************************************************/


const t_nn92_record nn92_table[] =
{
   {"H",        -3.7390f,0.0f,      0.3326f},
   {"D",        6.671f,  0.0f,      0.000519f},
   {"He",       3.26f,   0.0f,      0.00747f},
   {"Li",       -1.90f,  0.0f,      70.5f},
   {"Be",       7.79f,   0.0f,      0.0076f},
   {"B",        5.30f,   -0.213f, 767.f},
   {"C",        6.6460f, 0.0f,      0.0035f},
   {"N",        9.36f,   0.0f,      1.9f},
   {"O",        5.803f,  0.0f,      0.00019f},
   {"F",        5.654f,  0.0f,      0.0096f},
   {"Ne",       4.566f,  0.0f,      0.039f},
   {"Na",       3.63f,   0.0f,      0.53f},
   {"Mg",       5.375f,  0.0f,      0.063f},
   {"Al",       3.449f,  0.0f,      0.231f},
   {"Si",       4.1491f, 0.0f,      0.171f},
   {"P",        5.13f,   0.0f,      0.172f},
   {"S",        2.847f,  0.0f,      0.53f},
   {"Cl",       9.5770f, 0.0f,      33.5f},
   {"Ar",       1.909f,  0.0f,      0.675f},
   {"K",        3.67f,   0.0f,      2.1f},
   {"Ca",       4.70f,   0.0f,      0.43f},
   {"Sc",       12.29f,  0.0f,      27.5f},
   {"Ti",       -3.438f, 0.0f,      6.09f},
   {"V",        -0.3824f,0.0f,      5.08f},
   {"Cr",       3.635f,  0.0f,      3.05f},
   {"Mn",       -3.73f,  0.0f,      13.3f},
   {"Fe",       9.45f,   0.0f,      2.56f},
   {"Co",       2.49f,   0.0f,      37.18f},
   {"Ni",       10.3f,   0.0f,      4.49f},
   {"Cu",       7.718f,  0.0f,      3.78f},
   {"Zn",       5.680f,  0.0f,      1.11f},
   {"Ga",       7.288f,  0.0f,      2.75f},
   {"Ge",       8.185f,  0.0f,      2.2f},
   {"As",       6.58f,   0.0f,      4.5f},
   {"Se",       7.970f,  0.0f,      11.7f},
   {"Br",       6.795f,  0.0f,      6.9f},
   {"Kr",       7.81f,   0.0f,      25.f},
   {"Rb",       7.09f,   0.0f,      0.38f},
   {"Sr",       7.02f,   0.0f,      1.28f},
   {"Y",        7.75f,   0.0f,      1.28f},
   {"Zr",       7.16f,   0.0f,      0.185f},
   {"Nb",       7.054f,  0.0f,      1.15f},
   {"Mo",       6.715f,  0.0f,      2.48f},
   {"Tc",       6.8f,    0.0f,      20.f},
   {"Ru",       7.03f,   0.0f,      2.56f},
   {"Rh",       5.88f,   0.0f,      144.8f},
   {"Pd",       5.91f,   0.0f,      6.9f},
   {"Ag",       5.922f,  0.0f,      63.3f},
   {"Cd",       4.87f,   -0.70f,  2520.f},
   {"In",       4.065f,  -0.0539f,193.8f},
   {"Sn",       6.225f,  0.0f,      0.626f},
   {"Sb",       5.57f,   0.0f,      4.91f},
   {"Te",       5.80f,   0.0f,      4.7f},
   {"I",        5.28f,   0.0f,      6.15f},
   {"Xe",       4.92f,   0.0f,      23.9f},
   {"Cs",       5.42f,   0.0f,      29.0f},
   {"Ba",       5.07f,   0.0f,      1.1f},
   {"La",       8.24f,   0.0f,      8.97f},
   {"Ce",       4.84f,   0.0f,      0.63f},
   {"Pr",       4.58f,   0.0f,      11.5f},
   {"Nd",       7.69f,   0.0f,      50.5f},
   {"Pm",       12.6f,   0.0f,      168.4f},
   {"Sm",       0.80f,   -1.65f,  5922.f},
   {"Eu",       7.22f,   -1.26f,  4530.f},
   {"Gd",       6.5f,    -13.82f, 49700.f},
   {"Tb",       7.38f,   0.0f,      23.4f},
   {"Dy",       16.9f,   -0.276f, 994.f},
   {"Ho",       8.01f,   0.0f,      64.7f},
   {"Er",       7.79f,   0.0f,      159.f},
   {"Tm",       7.07f,   0.0f,      100.f},
   {"Yb",       12.43f,  0.0f,      34.8f},
   {"Lu",       7.21f,   0.0f,      74.f},
   {"Hf",       7.7f,    0.0f,      104.1f},
   {"Ta",       6.91f,   0.0f,      20.6f},
   {"W",        4.86f,   0.0f,      18.3f},
   {"Re",       9.2f,    0.0f,      89.7f},
   {"Os",       10.7f,   0.0f,      16.0f},
   {"Ir",       10.6f,   0.0f,      425.f},
   {"Pt",       9.60f,   0.0f,      10.3f},
   {"Au",       7.63f,   0.0f,      98.65f},
   {"Hg",       12.692f, 0.0f,      372.3f},
   {"Tl",       8.776f,  0.0f,      3.43f},
   {"Pb",       9.405f,  0.0f,      0.171f},
   {"Bi",       8.532f,  0.0f,      0.0338f},
   {"Th",       10.31f,  0.0f,      7.37f},
   {"U",        8.417f,  0.0f,      7.57f},
   {NULL,       0.0f,  0.0f,         0.0f}
};

const t_nn92_record *find_in_nn92(const char *label)
{
    const t_nn92_record  *it;
    char buf[3] = { '\0', '\0', '\0' };

    /* empty label */
    if (!label || strlen(label) == 0)
        return NULL;

    buf[0] = toupper(label[0]);
    if (strlen(label) > 1 && isalpha(label[1]))
        buf[1] = tolower(label[1]);

    for (it = nn92_table; it->symbol != NULL; it++)
        if (strcmp(it->symbol, buf) == 0)
            return it;
    return NULL;
}

