"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionInfo = void 0;
const fact_1 = require("./fact");
/**
 * Information pertaining to an AWS region.
 */
class RegionInfo {
    constructor(name) {
        this.name = name;
    }
    /**
     * @returns the list of names of AWS regions for which there is at least one registered fact. This
     *          may not be an exaustive list of all available AWS regions.
     */
    static get regions() {
        return fact_1.Fact.regions.map(RegionInfo.get);
    }
    /**
     * Retrieves a collection of all fact values for all regions that fact is defined in.
     *
     * @param factName the name of the fact to retrieve values for.
     *   For a list of common fact names, see the FactName class
     * @returns a mapping with AWS region codes as the keys,
     *   and the fact in the given region as the value for that key
     */
    static regionMap(factName) {
        const ret = {};
        for (const regionInfo of RegionInfo.regions) {
            const fact = fact_1.Fact.find(regionInfo.name, factName);
            if (fact) {
                ret[regionInfo.name] = fact;
            }
        }
        return ret;
    }
    /**
     * Obtain region info for a given region name.
     *
     * @param name the name of the region (e.g: us-east-1)
     */
    static get(name) {
        return new RegionInfo(name);
    }
    /**
     * Whether the `AWS::CDK::Metadata` CloudFormation Resource is available in this region or not.
     */
    get cdkMetadataResourceAvailable() {
        return fact_1.Fact.find(this.name, fact_1.FactName.CDK_METADATA_RESOURCE_AVAILABLE) === 'YES';
    }
    /**
     * The domain name suffix (e.g: amazonaws.com) for this region.
     */
    get domainSuffix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DOMAIN_SUFFIX);
    }
    /**
     * The name of the ARN partition for this region (e.g: aws).
     */
    get partition() {
        return fact_1.Fact.find(this.name, fact_1.FactName.PARTITION);
    }
    /**
     * The endpoint used by S3 static website hosting in this region (e.g: s3-static-website-us-east-1.amazonaws.com)
     */
    get s3StaticWebsiteEndpoint() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ENDPOINT);
    }
    /**
     * The hosted zone ID used by Route 53 to alias a S3 static website in this region (e.g: Z2O1EMRO9K5GLX)
     */
    get s3StaticWebsiteHostedZoneId() {
        return fact_1.Fact.find(this.name, fact_1.FactName.S3_STATIC_WEBSITE_ZONE_53_HOSTED_ZONE_ID);
    }
    /**
     * The prefix for VPC Endpoint Service names,
     * cn.com.amazonaws.vpce for China regions,
     * com.amazonaws.vpce otherwise.
     */
    get vpcEndpointServiceNamePrefix() {
        return fact_1.Fact.find(this.name, fact_1.FactName.VPC_ENDPOINT_SERVICE_NAME_PREFIX);
    }
    /**
     * The name of the service principal for a given service in this region.
     * @param service the service name (e.g: s3.amazonaws.com)
     */
    servicePrincipal(service) {
        return fact_1.Fact.find(this.name, fact_1.FactName.servicePrincipal(service));
    }
    /**
     * The account ID for ELBv2 in this region
     *
     */
    get elbv2Account() {
        return fact_1.Fact.find(this.name, fact_1.FactName.ELBV2_ACCOUNT);
    }
    /**
     * The ID of the AWS account that owns the public ECR repository containing the
     * AWS Deep Learning Containers images in this region.
     */
    get dlcRepositoryAccount() {
        return fact_1.Fact.find(this.name, fact_1.FactName.DLC_REPOSITORY_ACCOUNT);
    }
}
exports.RegionInfo = RegionInfo;
//# sourceMappingURL=data:application/json;base64,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