"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleSynthAction = void 0;
const path = require("path");
const codebuild = require("../../../aws-codebuild"); // Automatically re-written from '@aws-cdk/aws-codebuild'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const codepipeline_actions = require("../../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const construct_internals_1 = require("../private/construct-internals");
const _util_1 = require("./_util");
/**
 * A standard synth with a generated buildspec
 */
class SimpleSynthAction {
    constructor(props) {
        var _a, _b, _c;
        this.props = props;
        // A number of actionProperties get read before bind() is even called (so before we
        // have made the Project and can construct the actual CodeBuildAction)
        //
        // - actionName
        // - resource
        // - region
        // - category
        // - role
        // - owner
        this._actionProperties = {
            actionName: (_a = props.actionName) !== null && _a !== void 0 ? _a : 'Synth',
            category: codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 0, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.sourceArtifact],
            outputs: [props.cloudAssemblyArtifact, ...((_b = props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
        };
        const addls = (_c = props.additionalArtifacts) !== null && _c !== void 0 ? _c : [];
        if (Object.keys(addls).length > 0) {
            if (!props.cloudAssemblyArtifact.artifactName) {
                throw new Error('You must give all output artifacts, including the \'cloudAssemblyArtifact\', names when using \'additionalArtifacts\'');
            }
            for (const addl of addls) {
                if (!addl.artifact.artifactName) {
                    throw new Error('You must give all output artifacts passed to SimpleSynthAction names when using \'additionalArtifacts\'');
                }
            }
        }
    }
    /**
     * Create a standard NPM synth action
     *
     * Uses `npm ci` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'npm run build'`.
     */
    static standardNpmSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'npm ci',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Create a standard Yarn synth action
     *
     * Uses `yarn install --frozen-lockfile` to install dependencies and `npx cdk synth` to synthesize.
     *
     * If you need a build step, add `buildCommand: 'yarn build'`.
     */
    static standardYarnSynth(options) {
        var _a, _b;
        return new SimpleSynthAction({
            ...options,
            installCommand: (_a = options.installCommand) !== null && _a !== void 0 ? _a : 'yarn install --frozen-lockfile',
            synthCommand: (_b = options.synthCommand) !== null && _b !== void 0 ? _b : 'npx cdk synth',
        });
    }
    /**
     * Exists to implement IAction
     */
    get actionProperties() {
        return this._actionProperties;
    }
    /**
     * Exists to implement IAction
     */
    bind(scope, stage, options) {
        var _a, _b;
        const buildCommand = this.props.buildCommand;
        const synthCommand = this.props.synthCommand;
        const installCommand = this.props.installCommand;
        const project = new codebuild.PipelineProject(scope, 'CdkBuildProject', {
            projectName: (_a = this.props.projectName) !== null && _a !== void 0 ? _a : this.props.projectName,
            environment: { buildImage: codebuild.LinuxBuildImage.STANDARD_4_0, ...this.props.environment },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: _util_1.filterEmpty([
                            this.props.subdirectory ? `cd ${this.props.subdirectory}` : '',
                            installCommand,
                        ]),
                    },
                    build: {
                        commands: _util_1.filterEmpty([
                            buildCommand,
                            synthCommand,
                        ]),
                    },
                },
                artifacts: renderArtifacts(this),
            }),
            environmentVariables: {
                ..._util_1.copyEnvironmentVariables(...this.props.copyEnvironmentVariables || []),
                ...this.props.environmentVariables,
            },
        });
        if (this.props.rolePolicyStatements !== undefined) {
            this.props.rolePolicyStatements.forEach(policyStatement => {
                project.addToRolePolicy(policyStatement);
            });
        }
        this._action = new codepipeline_actions.CodeBuildAction({
            actionName: this.actionProperties.actionName,
            input: this.props.sourceArtifact,
            outputs: [this.props.cloudAssemblyArtifact, ...((_b = this.props.additionalArtifacts) !== null && _b !== void 0 ? _b : []).map(a => a.artifact)],
            project,
        });
        this._actionProperties = this._action.actionProperties;
        return this._action.bind(scope, stage, options);
        function renderArtifacts(self) {
            var _a;
            // save the generated files in the output artifact
            // This part of the buildspec has to look completely different depending on whether we're
            // using secondary artifacts or not.
            const cloudAsmArtifactSpec = {
                'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', construct_internals_1.cloudAssemblyBuildSpecDir(scope)),
                'files': '**/*',
            };
            if (self.props.additionalArtifacts) {
                const secondary = {};
                if (!self.props.cloudAssemblyArtifact.artifactName) {
                    throw new Error('When using additional output artifacts, you must also name the CloudAssembly artifact');
                }
                secondary[self.props.cloudAssemblyArtifact.artifactName] = cloudAsmArtifactSpec;
                self.props.additionalArtifacts.forEach((art) => {
                    var _a;
                    if (!art.artifact.artifactName) {
                        throw new Error('You must give the output artifact a name');
                    }
                    secondary[art.artifact.artifactName] = {
                        'base-directory': path.join((_a = self.props.subdirectory) !== null && _a !== void 0 ? _a : '.', art.directory),
                        'files': '**/*',
                    };
                });
                return { 'secondary-artifacts': secondary };
            }
            return cloudAsmArtifactSpec;
        }
    }
    /**
     * Exists to implement IAction
     */
    onStateChange(name, target, options) {
        if (!this._action) {
            throw new Error('Need bind() first');
        }
        return this._action.onStateChange(name, target, options);
    }
}
exports.SimpleSynthAction = SimpleSynthAction;
//# sourceMappingURL=data:application/json;base64,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