"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const path = require("path");
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
            ...props,
            restartExecutionOnUpdate: true,
            stages: [
                {
                    stageName: 'Source',
                    actions: [props.sourceAction],
                },
                {
                    stageName: 'Build',
                    actions: [props.synthAction],
                },
                {
                    stageName: 'UpdatePipeline',
                    actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                            cloudAssemblyInput: this._cloudAssemblyArtifact,
                            pipelineStackName: pipelineStack.stackName,
                            cdkCliVersion: props.cdkCliVersion,
                            projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        })],
                },
            ],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
        this.node.applyAspect({ visit: () => this._assets.removeAssetsStageIfEmpty() });
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    this.node.addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this.assetRoles = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
        this.pipeline = this.props.pipeline;
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        // Late-binding here (rather than in the constructor) to prevent creating the role in cases where no asset actions are created.
        if (!this.assetRoles[command.assetType]) {
            this.generateAssetRole(command.assetType);
        }
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            // NOTE: It's important that asset changes don't force a pipeline self-mutation.
            // This can cause an infinite loop of updates (see https://github.com/aws/aws-cdk/issues/9080).
            // For that reason, we use the id as the actionName below, rather than the asset hash.
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: id,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
                role: this.assetRoles[command.assetType],
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
    /**
     * This role is used by both the CodePipeline build action and related CodeBuild project. Consolidating these two
     * roles into one, and re-using across all assets, saves significant size of the final synthesized output.
     * Modeled after the CodePipeline role and 'CodePipelineActionRole' roles.
     * Generates one role per asset type to separate file and Docker/image-based permissions.
     */
    generateAssetRole(assetType) {
        if (this.assetRoles[assetType]) {
            return this.assetRoles[assetType];
        }
        const rolePrefix = assetType === actions_1.AssetType.DOCKER_IMAGE ? 'Docker' : 'File';
        const assetRole = new iam.Role(this, `${rolePrefix}Role`, {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('codebuild.amazonaws.com'), new iam.AccountPrincipal(core_1.Stack.of(this).account)),
        });
        // Logging permissions
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: '/aws/codebuild/*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: [logGroupArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        }));
        // CodeBuild report groups
        const codeBuildArn = core_1.Stack.of(this).formatArn({
            service: 'codebuild',
            resource: 'report-group',
            resourceName: '*',
        });
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'codebuild:CreateReportGroup',
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resources: [codeBuildArn],
        }));
        // CodeBuild start/stop
        assetRole.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // Publishing role access
        const rolePattern = assetType === actions_1.AssetType.DOCKER_IMAGE
            ? 'arn:*:iam::*:role/*-image-publishing-role-*'
            : 'arn:*:iam::*:role/*-file-publishing-role-*';
        assetRole.addToPolicy(new iam.PolicyStatement({
            actions: ['sts:AssumeRole'],
            resources: [rolePattern],
        }));
        // Artifact access
        this.pipeline.artifactBucket.grantRead(assetRole);
        this.assetRoles[assetType] = assetRole.withoutPolicyUpdates();
        return this.assetRoles[assetType];
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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