"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsCustomResource = exports.AwsCustomResourcePolicy = exports.PhysicalResourceId = void 0;
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * Physical ID of the custom resource.
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * Extract the physical resource id from the path (dot notation) to the data in the API call response.
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * Explicit physical resource id.
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * The IAM Policy that will be applied to the different calls.
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * Use this constant to configure access to any resource.
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
        });
        this.grantPrincipal = provider.grantPrincipal;
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                provider.addToRolePolicy(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    provider.addToRolePolicy(new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    }));
                }
            }
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
                installLatestAwsSdk: (_b = props.installLatestAwsSdk) !== null && _b !== void 0 ? _b : true,
            },
        });
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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