"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addDependency = void 0;
const stack_1 = require("./stack");
const stage_1 = require("./stage");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a CloudFormation
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the AWS::CloudFormation::Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    const sourceStage = stage_1.Stage.of(sourceStack);
    const targetStage = stage_1.Stage.of(targetStack);
    if (sourceStage !== targetStage) {
        // eslint-disable-next-line max-len
        throw new Error(`You cannot add a dependency from '${source.node.path}' (in ${describeStage(sourceStage)}) to '${target.node.path}' (in ${describeStage(targetStage)}): dependency cannot cross stage boundaries`);
    }
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define a assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}': ${reason}`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element) ? element.nestedStackResource : element;
        if (!resource) {
            throw new Error('assertion failure'); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
/**
 * Return a string representation of the given assembler, for use in error messages
 */
function describeStage(assembly) {
    if (!assembly) {
        return 'an unrooted construct tree';
    }
    if (!assembly.parentStage) {
        return 'the App';
    }
    return `Stage '${assembly.node.path}'`;
}
//# sourceMappingURL=data:application/json;base64,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