"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstructNode = exports.ConstructOrder = exports.Construct = void 0;
/**
 * Constructs compatibility layer.
 *
 * This file includes types that shadow types in the "constructs" module in
 * order to allow backwards-compatiblity in the AWS CDK v1.0 release line.
 *
 * There are pretty ugly hacks here, which mostly involve downcasting types to
 * adhere to legacy AWS CDK APIs.
 *
 * This file, in its entirety, is expected to be removed in v2.0.
 */
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs = require("constructs");
const annotations_1 = require("./annotations");
const aspect_1 = require("./aspect");
const token_1 = require("./token");
const ORIGINAL_CONSTRUCT_NODE_SYMBOL = Symbol.for('@aws-cdk/core.ConstructNode');
const CONSTRUCT_SYMBOL = Symbol.for('@aws-cdk/core.Construct');
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct extends constructs.Construct {
    constructor(scope, id) {
        super(scope, id, {
            nodeFactory: {
                createNode: (h, s, i) => new ConstructNode(h, s, i)._actualNode,
            },
        });
        if (token_1.Token.isUnresolved(id)) {
            throw new Error(`Cannot use tokens in construct ID: ${id}`);
        }
        Object.defineProperty(this, CONSTRUCT_SYMBOL, { value: true });
        this.node = ConstructNode._unwrap(constructs.Node.of(this));
        const disableTrace = this.node.tryGetContext(cxapi.DISABLE_METADATA_STACK_TRACE) ||
            this.node.tryGetContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA) ||
            process.env.CDK_DISABLE_STACK_TRACE;
        if (disableTrace) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
            this.node.setContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA, true);
            process.env.CDK_DISABLE_STACK_TRACE = '1';
        }
    }
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x) {
        return typeof x === 'object' && x !== null && CONSTRUCT_SYMBOL in x;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    onValidate() {
        return this.validate();
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    onPrepare() {
        this.prepare();
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    onSynthesize(session) {
        this.synthesize({
            outdir: session.outdir,
            assembly: session.assembly,
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    validate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    prepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    synthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
/**
 * Represents the construct node in the scope tree.
 */
class ConstructNode {
    constructor(host, scope, id) {
        this.host = host;
        this._actualNode = new constructs.Node(host, scope, id);
        // store a back reference on _actualNode so we can our ConstructNode from it
        Object.defineProperty(this._actualNode, ORIGINAL_CONSTRUCT_NODE_SYMBOL, {
            value: this,
            configurable: false,
            enumerable: false,
        });
    }
    /**
     * Returns the wrapping `@aws-cdk/core.ConstructNode` instance from a `constructs.ConstructNode`.
     *
     * @internal
     */
    static _unwrap(c) {
        const x = c[ORIGINAL_CONSTRUCT_NODE_SYMBOL];
        if (!x) {
            throw new Error('invalid ConstructNode type');
        }
        return x;
    }
    /**
     * Synthesizes a CloudAssembly from a construct tree.
     * @param node The root of the construct tree.
     * @param options Synthesis options.
     * @deprecated Use `app.synth()` or `stage.synth()` instead
     */
    static synth(node, options = {}) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const a = require('./private/synthesis');
        return a.synthesize(node.root, options);
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     * @param node The root node
     * @deprecated Use `app.synth()` instead
     */
    static prepare(node) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const p = require('./private/prepare-app');
        p.prepareApp(node.root); // resolve cross refs and nested stack assets.
        return node._actualNode.prepare();
    }
    /**
     * Invokes "validate" on all constructs in the tree (depth-first, pre-order) and returns
     * the list of all errors. An empty list indicates that there are no errors.
     *
     * @param node The root node
     */
    static validate(node) {
        return node._actualNode.validate().map(e => ({ source: e.source, message: e.message }));
    }
    /**
     * Returns the scope in which this construct is defined.
     *
     * The value is `undefined` at the root of the construct scope tree.
     */
    get scope() {
        return this._actualNode.scope;
    }
    /**
     * The id of this construct within the current scope.
     *
     * This is a a scope-unique id. To obtain an app-unique id for this construct, use `uniqueId`.
     */
    get id() { return this._actualNode.id; }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() { return this._actualNode.path; }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() { return this._actualNode.uniqueId; }
    /**
     * Return a direct child by id, or undefined
     *
     * @param id Identifier of direct child
     * @returns the child if found, or undefined
     */
    tryFindChild(id) { return this._actualNode.tryFindChild(id); }
    /**
     * Return a direct child by id
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child
     * @returns Child with the given id.
     */
    findChild(id) { return this._actualNode.findChild(id); }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     *
     * @throws if there is more than one child
     * @returns a construct or undefined if there is no default child
     */
    get defaultChild() { return this._actualNode.defaultChild; }
    /**
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     */
    set defaultChild(value) { this._actualNode.defaultChild = value; }
    /**
     * All direct children of this construct.
     */
    get children() { return this._actualNode.children; }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PREORDER) { return this._actualNode.findAll(order); }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        this._actualNode.setContext(key, value);
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or `undefined` if there is no context value for thie key.
     */
    tryGetContext(key) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        return this._actualNode.tryGetContext(key);
    }
    /**
     * An immutable array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() { return this._actualNode.metadata; }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, fromFunction) { this._actualNode.addMetadata(type, data, fromFunction); }
    /**
     * DEPRECATED: Adds a { "info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     * @deprecated use `Annotations.of(construct).addInfo()`
     */
    addInfo(message) {
        annotations_1.Annotations.of(this.host).addInfo(message);
    }
    /**
     * DEPRECATED: Adds a { "warning": <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     * @deprecated use `Annotations.of(construct).addWarning()`
     */
    addWarning(message) {
        annotations_1.Annotations.of(this.host).addWarning(message);
    }
    /**
     * DEPRECATED: Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     * @deprecated use `Annotations.of(construct).addError()`
     */
    addError(message) {
        annotations_1.Annotations.of(this.host).addError(message);
    }
    /**
     * DEPRECATED: Applies the aspect to this Constructs node
     *
     * @deprecated This API is going to be removed in the next major version of
     * the AWS CDK. Please use `Aspects.of(scope).add()` instead.
     */
    applyAspect(aspect) { aspect_1.Aspects.of(this.host).add(aspect); }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     */
    get scopes() { return this._actualNode.scopes; }
    /**
     * @returns The root of the construct tree.
     */
    get root() { return this._actualNode.root; }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() { return this._actualNode.locked; }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) { this._actualNode.addDependency(...dependencies); }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    get dependencies() { return this._actualNode.dependencies; }
    /**
     * Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) { return this._actualNode.tryRemoveChild(childName); }
}
exports.ConstructNode = ConstructNode;
/**
 * Separator used to delimit construct path components.
 */
ConstructNode.PATH_SEP = '/';
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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