"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        // ToDo implement preserveLogicalIds=false
        this.preserveLogicalIds = true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.nestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.nestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the NestedStack with name logicalId.
     * For a nested stack to be returned by this method, it must be specified in the {@link CfnIncludeProps.nestedStacks}
     * property.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the nestedStacks property when including the parent template`);
            }
        }
        return this.nestedStacks[logicalId];
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        cfnMapping.overrideLogicalId(mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        cfnParameter.overrideLogicalId(logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Rule definitions is not allowed'); },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        rule.overrideLogicalId(ruleName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition() {
                    return undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        cfnOutput.overrideLogicalId(logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping() { throw new Error('Using FindInMap in Condition definitions is not allowed'); },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        // ToDo handle renaming of the logical IDs of the conditions
        cfnCondition.overrideLogicalId(conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The ${attribute} resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            if (l1ClassFqn) {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        if (this.preserveLogicalIds) {
            // override the logical ID to match the original template
            l1Instance.overrideLogicalId(logicalId);
        }
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: nestedStackProps.Parameters,
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        const propStack = this.nestedStacksToInclude[nestedStackId];
        const template = new CfnInclude(nestedStack, nestedStackId, {
            templateFile: propStack.templateFile,
            nestedStacks: propStack.nestedStacks,
        });
        const includedStack = { stack: nestedStack, includedTemplate: template };
        this.nestedStacks[nestedStackId] = includedStack;
        return nestedStackResource;
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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