"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lambda = exports.LambdaInvocationType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class Lambda {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.permissionsNode.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.ACCOUNT_ID,
            });
        }
        // Ensure permission is deployed before rule
        const permission = this.props.function.permissionsNode.tryFindChild(permissionId);
        if (permission) { // The Lambda could be imported
            rule.node.addDependency(permission);
        }
        else {
            // eslint-disable-next-line max-len
            rule.node.addWarning('This rule is using a Lambda action with an imported function. Ensure permission is given to SES to invoke that function.');
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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