"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplaceKey = exports.BucketAccessControl = exports.EventType = exports.BucketEncryption = exports.Bucket = exports.InventoryObjectVersion = exports.InventoryFrequency = exports.InventoryFormat = exports.RedirectProtocol = exports.HttpMethods = exports.BlockPublicAccess = void 0;
const os_1 = require("os");
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Indicates if a bucket resource policy should automatically created upon
         * the first call to `addToResourcePolicy`.
         */
        this.autoCreatePolicy = false;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: options.paths ? options.paths.map(p => this.arnForObjects(p)) : [this.bucketArn],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded
     * to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the
     * specified paths (keys) in this bucket are written to.  This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailWriteObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: [
                    'CompleteMultipartUpload',
                    'CopyObject',
                    'PutObject',
                ],
                requestParameters: {
                    bucketName: [this.bucketName],
                    key: options.paths,
                },
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or it's
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     */
    addToResourcePolicy(permission) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(permission);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * The https URL of an S3 object. For example:
     * @example https://s3.us-west-1.amazonaws.com/onlybucket
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     * @example https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key) {
        const stack = core_1.Stack.of(this);
        const prefix = `https://s3.${stack.region}.${stack.urlSuffix}/`;
        return this.buildUrl(prefix, key);
    }
    /**
     * The S3 URL of an S3 object. For example:
     * @example s3://onlybucket
     * @example s3://bucket/key
     * @param key The S3 key of the object. If not specified, the S3 URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    s3UrlForObject(key) {
        return this.buildUrl('s3://', key);
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     *
     * If you specify multiple components for keyPattern, they will be concatenated::
     *
     *   arnForObjects('home/', team, '/', user, '/*')
     *
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantWrite(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_WRITE_ACTIONS, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_PUT_ACTIONS, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantReadWrite(identity, objectsKeyPattern = '*') {
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(perms.BUCKET_WRITE_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.Anyone(),
            resource: this,
        });
    }
    buildUrl(prefix, key) {
        const components = [
            prefix,
            this.bucketName,
        ];
        if (key) {
            // trim prepending '/'
            if (typeof key === 'string' && key.startsWith('/')) {
                key = key.substr(1);
            }
            components.push('/');
            components.push(key);
        }
        return components.join('');
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const ret = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: bucketActions,
            resourceArns: resources,
            resource: this,
        });
        if (this.encryptionKey && keyActions && keyActions.length !== 0) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
}
class BlockPublicAccess {
    constructor(options) {
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
exports.BlockPublicAccess = BlockPublicAccess;
BlockPublicAccess.BLOCK_ALL = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true,
});
BlockPublicAccess.BLOCK_ACLS = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true,
});
/**
 * All http request methods
 */
var HttpMethods;
(function (HttpMethods) {
    /**
     * The GET method requests a representation of the specified resource.
     */
    HttpMethods["GET"] = "GET";
    /**
     * The PUT method replaces all current representations of the target resource with the request payload.
     */
    HttpMethods["PUT"] = "PUT";
    /**
     * The HEAD method asks for a response identical to that of a GET request, but without the response body.
     */
    HttpMethods["HEAD"] = "HEAD";
    /**
     * The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     */
    HttpMethods["POST"] = "POST";
    /**
     * The DELETE method deletes the specified resource.
     */
    HttpMethods["DELETE"] = "DELETE";
})(HttpMethods = exports.HttpMethods || (exports.HttpMethods = {}));
/**
 * All http request methods
 */
var RedirectProtocol;
(function (RedirectProtocol) {
    RedirectProtocol["HTTP"] = "http";
    RedirectProtocol["HTTPS"] = "https";
})(RedirectProtocol = exports.RedirectProtocol || (exports.RedirectProtocol = {}));
/**
 * All supported inventory list formats.
 */
var InventoryFormat;
(function (InventoryFormat) {
    /**
     * Generate the inventory list as CSV.
     */
    InventoryFormat["CSV"] = "CSV";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["PARQUET"] = "Parquet";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["ORC"] = "ORC";
})(InventoryFormat = exports.InventoryFormat || (exports.InventoryFormat = {}));
/**
 * All supported inventory frequencies.
 */
var InventoryFrequency;
(function (InventoryFrequency) {
    /**
     * A report is generated every day.
     */
    InventoryFrequency["DAILY"] = "Daily";
    /**
     * A report is generated every Sunday (UTC timezone) after the initial report.
     */
    InventoryFrequency["WEEKLY"] = "Weekly";
})(InventoryFrequency = exports.InventoryFrequency || (exports.InventoryFrequency = {}));
/**
 * Inventory version support.
 */
var InventoryObjectVersion;
(function (InventoryObjectVersion) {
    /**
     * Includes all versions of each object in the report.
     */
    InventoryObjectVersion["ALL"] = "All";
    /**
     * Includes only the current version of each object in the report.
     */
    InventoryObjectVersion["CURRENT"] = "Current";
})(InventoryObjectVersion = exports.InventoryObjectVersion || (exports.InventoryObjectVersion = {}));
/**
 * An S3 bucket with associated policy objects
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 */
class Bucket extends BucketBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.bucketName,
        });
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        this.cors = [];
        this.inventories = [];
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        this.validateBucketName(this.physicalName);
        const websiteConfiguration = this.renderWebsiteConfiguration(props);
        this.isWebsite = (websiteConfiguration !== undefined);
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: this.physicalName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: core_1.Lazy.anyValue({ produce: () => this.parseLifecycleConfiguration() }),
            websiteConfiguration,
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: core_1.Lazy.anyValue({ produce: () => this.parseMetricConfiguration() }),
            corsConfiguration: core_1.Lazy.anyValue({ produce: () => this.parseCorsConfiguration() }),
            accessControl: core_1.Lazy.stringValue({ produce: () => this.accessControl }),
            loggingConfiguration: this.parseServerAccessLogs(props),
            inventoryConfigurations: core_1.Lazy.anyValue({ produce: () => this.parseInventoryConfiguration() }),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.versioned = props.versioned;
        this.encryptionKey = encryptionKey;
        this.bucketName = this.getResourceNameAttribute(resource.ref);
        this.bucketArn = this.getResourceArnAttribute(resource.attrArn, {
            region: '',
            account: '',
            service: 's3',
            resource: this.physicalName,
        });
        this.bucketDomainName = resource.attrDomainName;
        this.bucketWebsiteUrl = resource.attrWebsiteUrl;
        this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', this.bucketWebsiteUrl));
        this.bucketDualStackDomainName = resource.attrDualStackDomainName;
        this.bucketRegionalDomainName = resource.attrRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        this.accessControl = props.accessControl;
        if (props.serverAccessLogsBucket instanceof Bucket) {
            props.serverAccessLogsBucket.allowLogDelivery();
        }
        for (const inventory of (_a = props.inventories) !== null && _a !== void 0 ? _a : []) {
            this.addInventory(inventory);
        }
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all cors configuration rules
        (props.cors || []).forEach(this.addCorsRule.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        // defines a BucketNotifications construct. Notice that an actual resource will only
        // be added if there are notifications added, so we don't need to condition this.
        this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', { bucket: this });
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
    }
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object. Can be obtained from a call to
     * `bucket.export()` or manually created.
     */
    static fromBucketAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const region = stack.region;
        const urlSuffix = stack.urlSuffix;
        const bucketName = util_1.parseBucketName(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        const newUrlFormat = attrs.bucketWebsiteNewUrlFormat === undefined
            ? false
            : attrs.bucketWebsiteNewUrlFormat;
        const websiteDomain = newUrlFormat
            ? `${bucketName}.s3-website.${region}.${urlSuffix}`
            : `${bucketName}.s3-website-${region}.${urlSuffix}`;
        class Import extends BucketBase {
            constructor() {
                var _a;
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = util_1.parseBucketArn(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || `http://${websiteDomain}`;
                this.bucketWebsiteDomainName = attrs.bucketWebsiteUrl ? core_1.Fn.select(2, core_1.Fn.split('/', attrs.bucketWebsiteUrl)) : websiteDomain;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = newUrlFormat;
                this.encryptionKey = attrs.encryptionKey;
                this.isWebsite = (_a = attrs.isWebsite) !== null && _a !== void 0 ? _a : false;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id, {
            account: attrs.account,
            region: attrs.region,
        });
    }
    /**
     * Add a lifecycle rule to the bucket
     *
     * @param rule The rule to add
     */
    addLifecycleRule(rule) {
        if ((rule.noncurrentVersionExpiration !== undefined
            || (rule.noncurrentVersionTransitions && rule.noncurrentVersionTransitions.length > 0))
            && !this.versioned) {
            throw new Error("Cannot use 'noncurrent' rules on a nonversioned bucket");
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add
     */
    addMetric(metric) {
        this.metrics.push(metric);
    }
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket
     *
     * @param rule The CORS configuration rule to add
     */
    addCorsRule(rule) {
        this.cors.push(rule);
    }
    /**
     * Adds a bucket notification event destination.
     * @param event The event to trigger the notification
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue)
     *
     * @param filters S3 object key filter rules to determine which objects
     * trigger this event. Each filter must include a `prefix` and/or `suffix`
     * that will be matched against the s3 object key. Refer to the S3 Developer Guide
     * for details about allowed filter rules.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering
     *
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     */
    addEventNotification(event, dest, ...filters) {
        this.notifications.addNotification(event, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * created in the bucket. This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectCreatedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_CREATED, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * removed from the bucket. This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectRemovedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_REMOVED, dest, ...filters);
    }
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add
     */
    addInventory(inventory) {
        this.inventories.push(inventory);
    }
    validateBucketName(physicalName) {
        const bucketName = physicalName;
        if (!bucketName || core_1.Token.isUnresolved(bucketName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        let encryptionType = props.encryption;
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey ? BucketEncryption.KMS : BucketEncryption.UNENCRYPTED;
        }
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.UNENCRYPTED) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn,
                        },
                    },
                ],
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } },
                ],
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KMS_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' } },
                ],
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            const enabled = rule.enabled !== undefined ? rule.enabled : true;
            const x = {
                // eslint-disable-next-line max-len
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfter !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfter.toDays() } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: rule.expiration && rule.expiration.toDays(),
                id: rule.id,
                noncurrentVersionExpirationInDays: rule.noncurrentVersionExpiration && rule.noncurrentVersionExpiration.toDays(),
                noncurrentVersionTransitions: mapOrUndefined(rule.noncurrentVersionTransitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionInDays: t.transitionAfter.toDays(),
                })),
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: mapOrUndefined(rule.transitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionDate: t.transitionDate,
                    transitionInDays: t.transitionAfter && t.transitionAfter.toDays(),
                })),
                tagFilters: self.parseTagFilters(rule.tagFilters),
            };
            return x;
        }
    }
    parseServerAccessLogs(props) {
        var _a;
        if (!props.serverAccessLogsBucket && !props.serverAccessLogsPrefix) {
            return undefined;
        }
        return {
            destinationBucketName: (_a = props.serverAccessLogsBucket) === null || _a === void 0 ? void 0 : _a.bucketName,
            logFilePrefix: props.serverAccessLogsPrefix,
        };
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters),
            };
        }
    }
    parseCorsConfiguration() {
        if (!this.cors || this.cors.length === 0) {
            return undefined;
        }
        return { corsRules: this.cors.map(parseCors) };
        function parseCors(rule) {
            return {
                id: rule.id,
                maxAge: rule.maxAge,
                allowedHeaders: rule.allowedHeaders,
                allowedMethods: rule.allowedMethods,
                allowedOrigins: rule.allowedOrigins,
                exposedHeaders: rule.exposedHeaders,
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag],
        }));
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument && !props.websiteRedirect && !props.websiteRoutingRules) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error('"websiteIndexDocument" is required if "websiteErrorDocument" is set');
        }
        if (props.websiteRedirect && (props.websiteErrorDocument || props.websiteIndexDocument || props.websiteRoutingRules)) {
            throw new Error('"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used');
        }
        const routingRules = props.websiteRoutingRules ? props.websiteRoutingRules.map((rule) => {
            if (rule.condition && !rule.condition.httpErrorCodeReturnedEquals && !rule.condition.keyPrefixEquals) {
                throw new Error('The condition property cannot be an empty object');
            }
            return {
                redirectRule: {
                    hostName: rule.hostName,
                    httpRedirectCode: rule.httpRedirectCode,
                    protocol: rule.protocol,
                    replaceKeyWith: rule.replaceKey && rule.replaceKey.withKey,
                    replaceKeyPrefixWith: rule.replaceKey && rule.replaceKey.prefixWithKey,
                },
                routingRuleCondition: rule.condition,
            };
        }) : undefined;
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument,
            redirectAllRequestsTo: props.websiteRedirect,
            routingRules,
        };
    }
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    allowLogDelivery() {
        if (this.accessControl && this.accessControl !== BucketAccessControl.LOG_DELIVERY_WRITE) {
            throw new Error("Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed");
        }
        this.accessControl = BucketAccessControl.LOG_DELIVERY_WRITE;
    }
    parseInventoryConfiguration() {
        if (!this.inventories || this.inventories.length === 0) {
            return undefined;
        }
        return this.inventories.map((inventory, index) => {
            var _a, _b, _c, _d, _e, _f;
            const format = (_a = inventory.format) !== null && _a !== void 0 ? _a : InventoryFormat.CSV;
            const frequency = (_b = inventory.frequency) !== null && _b !== void 0 ? _b : InventoryFrequency.WEEKLY;
            const id = (_c = inventory.inventoryId) !== null && _c !== void 0 ? _c : `${this.node.id}Inventory${index}`;
            if (inventory.destination.bucket instanceof Bucket) {
                inventory.destination.bucket.addToResourcePolicy(new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['s3:PutObject'],
                    resources: [
                        inventory.destination.bucket.bucketArn,
                        inventory.destination.bucket.arnForObjects(`${(_d = inventory.destination.prefix) !== null && _d !== void 0 ? _d : ''}*`),
                    ],
                    principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': this.bucketArn,
                        },
                    },
                }));
            }
            return {
                id,
                destination: {
                    bucketArn: inventory.destination.bucket.bucketArn,
                    bucketAccountId: inventory.destination.bucketOwner,
                    prefix: inventory.destination.prefix,
                    format,
                },
                enabled: (_e = inventory.enabled) !== null && _e !== void 0 ? _e : true,
                includedObjectVersions: (_f = inventory.includeObjectVersions) !== null && _f !== void 0 ? _f : InventoryObjectVersion.ALL,
                scheduleFrequency: frequency,
                optionalFields: inventory.optionalFields,
                prefix: inventory.objectsPrefix,
            };
        });
    }
}
exports.Bucket = Bucket;
/**
 * What kind of server-side encryption to apply to this bucket
 */
var BucketEncryption;
(function (BucketEncryption) {
    /**
     * Objects in the bucket are not encrypted.
     */
    BucketEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     */
    BucketEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a master key managed by S3.
     */
    BucketEncryption["S3_MANAGED"] = "S3MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    BucketEncryption["KMS"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 */
var EventType;
(function (EventType) {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED"] = "s3:ObjectCreated:*";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_PUT"] = "s3:ObjectCreated:Put";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_POST"] = "s3:ObjectCreated:Post";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COPY"] = "s3:ObjectCreated:Copy";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD"] = "s3:ObjectCreated:CompleteMultipartUpload";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED"] = "s3:ObjectRemoved:*";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE"] = "s3:ObjectRemoved:Delete";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE_MARKER_CREATED"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    /**
     * You can use this event type to request Amazon S3 to send a notification
     * message when Amazon S3 detects that an object of the RRS storage class is
     * lost.
     */
    EventType["REDUCED_REDUNDANCY_LOST_OBJECT"] = "s3:ReducedRedundancyLostObject";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 */
var BucketAccessControl;
(function (BucketAccessControl) {
    /**
     * Owner gets FULL_CONTROL. No one else has access rights.
     */
    BucketAccessControl["PRIVATE"] = "Private";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ access.
     */
    BucketAccessControl["PUBLIC_READ"] = "PublicRead";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     */
    BucketAccessControl["PUBLIC_READ_WRITE"] = "PublicReadWrite";
    /**
     * Owner gets FULL_CONTROL. The AuthenticatedUsers group gets READ access.
     */
    BucketAccessControl["AUTHENTICATED_READ"] = "AuthenticatedRead";
    /**
     * The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     */
    BucketAccessControl["LOG_DELIVERY_WRITE"] = "LogDeliveryWrite";
    /**
     * Object owner gets FULL_CONTROL. Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_READ"] = "BucketOwnerRead";
    /**
     * Both the object owner and the bucket owner get FULL_CONTROL over the object.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_FULL_CONTROL"] = "BucketOwnerFullControl";
    /**
     * Owner gets FULL_CONTROL. Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     */
    BucketAccessControl["AWS_EXEC_READ"] = "AwsExecRead";
})(BucketAccessControl = exports.BucketAccessControl || (exports.BucketAccessControl = {}));
class ReplaceKey {
    constructor(withKey, prefixWithKey) {
        this.withKey = withKey;
        this.prefixWithKey = prefixWithKey;
    }
    /**
     * The specific object key to use in the redirect request
     */
    static with(keyReplacement) {
        return new this(keyReplacement);
    }
    /**
     * The object key prefix to use in the redirect request
     */
    static prefixWith(keyReplacement) {
        return new this(undefined, keyReplacement);
    }
}
exports.ReplaceKey = ReplaceKey;
function mapOrUndefined(list, callback) {
    if (!list || list.length === 0) {
        return undefined;
    }
    return list.map(callback);
}
//# sourceMappingURL=data:application/json;base64,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