import * as iam from "../../aws-iam";
import * as secretsmanager from "../../aws-secretsmanager";
import * as core from "../../core";
import { IEngine } from './engine';
import { IParameterGroup } from './parameter-group';
/**
 * The extra options passed to the {@link IClusterEngine.bindToCluster} method.
 */
export interface ClusterEngineBindOptions {
    /**
     * The role used for S3 importing.
     *
     * @default - none
     */
    readonly s3ImportRole?: iam.IRole;
    /**
     * The role used for S3 exporting.
     *
     *  @default - none
     */
    readonly s3ExportRole?: iam.IRole;
    /**
     * The customer-provided ParameterGroup.
     *
     * @default - none
     */
    readonly parameterGroup?: IParameterGroup;
}
/**
 * The type returned from the {@link IClusterEngine.bindToCluster} method.
 */
export interface ClusterEngineConfig {
    /**
     * The ParameterGroup to use for the cluster.
     *
     * @default - no ParameterGroup will be used
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * The port to use for this cluster,
     * unless the customer specified the port directly.
     *
     * @default - use the default port for clusters (3306)
     */
    readonly port?: number;
}
/**
 * The interface representing a database cluster (as opposed to instance) engine.
 */
export interface IClusterEngine extends IEngine {
    /** The application used by this engine to perform rotation for a single-user scenario. */
    readonly singleUserRotationApplication: secretsmanager.SecretRotationApplication;
    /** The application used by this engine to perform rotation for a multi-user scenario. */
    readonly multiUserRotationApplication: secretsmanager.SecretRotationApplication;
    /** The log types that are available with this engine type */
    readonly supportedLogTypes: string[];
    /**
     * Method called when the engine is used to create a new cluster.
     */
    bindToCluster(scope: core.Construct, options: ClusterEngineBindOptions): ClusterEngineConfig;
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
export declare class AuroraEngineVersion {
    /** Version "5.6.10a". */
    static readonly VER_10A: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.17.9". */
    static readonly VER_1_17_9: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.19.0". */
    static readonly VER_1_19_0: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.19.1". */
    static readonly VER_1_19_1: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.19.2". */
    static readonly VER_1_19_2: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.19.5". */
    static readonly VER_1_19_5: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.19.6". */
    static readonly VER_1_19_6: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.20.0". */
    static readonly VER_1_20_0: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.20.1". */
    static readonly VER_1_20_1: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.21.0". */
    static readonly VER_1_21_0: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.22.0". */
    static readonly VER_1_22_0: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.22.1". */
    static readonly VER_1_22_1: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.22.1.3". */
    static readonly VER_1_22_1_3: AuroraEngineVersion;
    /** Version "5.6.mysql_aurora.1.22.2". */
    static readonly VER_1_22_2: AuroraEngineVersion;
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion: string, auroraMajorVersion?: string): AuroraEngineVersion;
    private static builtIn_5_6;
    /** The full version string, for example, "5.6.mysql_aurora.1.78.3.6". */
    readonly auroraFullVersion: string;
    /** The major version of the engine. Currently, it's always "5.6". */
    readonly auroraMajorVersion: string;
    private constructor();
}
/**
 * Creation properties of the plain Aurora database cluster engine.
 * Used in {@link DatabaseClusterEngine.aurora}.
 */
export interface AuroraClusterEngineProps {
    /** The version of the Aurora cluster engine. */
    readonly version: AuroraEngineVersion;
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
export declare class AuroraMysqlEngineVersion {
    /** Version "5.7.12". */
    static readonly VER_5_7_12: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.03.2". */
    static readonly VER_2_03_2: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.03.3". */
    static readonly VER_2_03_3: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.03.4". */
    static readonly VER_2_03_4: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.0". */
    static readonly VER_2_04_0: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.1". */
    static readonly VER_2_04_1: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.2". */
    static readonly VER_2_04_2: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.3". */
    static readonly VER_2_04_3: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.4". */
    static readonly VER_2_04_4: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.5". */
    static readonly VER_2_04_5: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.6". */
    static readonly VER_2_04_6: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.7". */
    static readonly VER_2_04_7: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.04.8". */
    static readonly VER_2_04_8: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.05.0". */
    static readonly VER_2_05_0: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.06.0". */
    static readonly VER_2_06_0: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.07.0". */
    static readonly VER_2_07_0: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.07.1". */
    static readonly VER_2_07_1: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.07.2". */
    static readonly VER_2_07_2: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.08.0". */
    static readonly VER_2_08_0: AuroraMysqlEngineVersion;
    /** Version "5.7.mysql_aurora.2.08.1". */
    static readonly VER_2_08_1: AuroraMysqlEngineVersion;
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion: string, auroraMysqlMajorVersion?: string): AuroraMysqlEngineVersion;
    private static builtIn_5_7;
    /** The full version string, for example, "5.7.mysql_aurora.1.78.3.6". */
    readonly auroraMysqlFullVersion: string;
    /** The major version of the engine. Currently, it's always "5.7". */
    readonly auroraMysqlMajorVersion: string;
    private constructor();
}
/**
 * Creation properties of the Aurora MySQL database cluster engine.
 * Used in {@link DatabaseClusterEngine.auroraMysql}.
 */
export interface AuroraMysqlClusterEngineProps {
    /** The version of the Aurora MySQL cluster engine. */
    readonly version: AuroraMysqlEngineVersion;
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
export declare class AuroraPostgresEngineVersion {
    /** Version "9.6.8". */
    static readonly VER_9_6_8: AuroraPostgresEngineVersion;
    /** Version "9.6.9". */
    static readonly VER_9_6_9: AuroraPostgresEngineVersion;
    /** Version "9.6.11". */
    static readonly VER_9_6_11: AuroraPostgresEngineVersion;
    /** Version "9.6.12". */
    static readonly VER_9_6_12: AuroraPostgresEngineVersion;
    /** Version "9.6.16". */
    static readonly VER_9_6_16: AuroraPostgresEngineVersion;
    /** Version "9.6.17". */
    static readonly VER_9_6_17: AuroraPostgresEngineVersion;
    /** Version "10.4". */
    static readonly VER_10_4: AuroraPostgresEngineVersion;
    /** Version "10.5". */
    static readonly VER_10_5: AuroraPostgresEngineVersion;
    /** Version "10.6". */
    static readonly VER_10_6: AuroraPostgresEngineVersion;
    /** Version "10.7". */
    static readonly VER_10_7: AuroraPostgresEngineVersion;
    /** Version "10.11". */
    static readonly VER_10_11: AuroraPostgresEngineVersion;
    /** Version "10.12". */
    static readonly VER_10_12: AuroraPostgresEngineVersion;
    /** Version "11.4". */
    static readonly VER_11_4: AuroraPostgresEngineVersion;
    /** Version "11.6". */
    static readonly VER_11_6: AuroraPostgresEngineVersion;
    /** Version "11.7". */
    static readonly VER_11_7: AuroraPostgresEngineVersion;
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion: string, auroraPostgresMajorVersion: string): AuroraPostgresEngineVersion;
    /** The full version string, for example, "9.6.25.1". */
    readonly auroraPostgresFullVersion: string;
    /** The major version of the engine, for example, "9.6". */
    readonly auroraPostgresMajorVersion: string;
    private constructor();
}
/**
 * Creation properties of the Aurora PostgreSQL database cluster engine.
 * Used in {@link DatabaseClusterEngine.auroraPostgres}.
 */
export interface AuroraPostgresClusterEngineProps {
    /** The version of the Aurora PostgreSQL cluster engine. */
    readonly version: AuroraPostgresEngineVersion;
}
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
export declare class DatabaseClusterEngine {
    /**
     * The unversioned 'aurora' cluster engine.
     *
     * @deprecated using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link aurora()} method
     */
    static readonly AURORA: IClusterEngine;
    /**
     * The unversioned 'aurora-msql' cluster engine.
     *
     * @deprecated using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link auroraMysql()} method
     */
    static readonly AURORA_MYSQL: IClusterEngine;
    /**
     * The unversioned 'aurora-postgresql' cluster engine.
     *
     * @deprecated using unversioned engines is an availability risk.
     *   We recommend using versioned engines created using the {@link auroraPostgres()} method
     */
    static readonly AURORA_POSTGRESQL: IClusterEngine;
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props: AuroraClusterEngineProps): IClusterEngine;
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props: AuroraMysqlClusterEngineProps): IClusterEngine;
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props: AuroraPostgresClusterEngineProps): IClusterEngine;
}
