"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SingletonFunction = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /** @internal */
    _checkEdgeCompatibility() {
        if (this.lambdaFunction instanceof function_base_1.FunctionBase) {
            return this.lambdaFunction._checkEdgeCompatibility();
        }
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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