"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static get runsLocally() {
        if (LocalBundler._runsLocally !== undefined) {
            return LocalBundler._runsLocally;
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel'), ['--version']);
            LocalBundler._runsLocally = /^2/.test(parcel.stdout.toString().trim()); // Cache result to avoid unnecessary spawns
            return LocalBundler._runsLocally;
        }
        catch (_a) {
            return false;
        }
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b;
        const image = props.buildImage
            ? cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_a = props.buildArgs) !== null && _a !== void 0 ? _a : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_b = props.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
                },
            })
            : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)),
        };
    }
}
exports.DockerBundler = DockerBundler;
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.ts$/, '.js');
    const parcelCommand = chain([
        [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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